#!/usr/bin/python
# This file is part of the FireSync Project Generator.
# Copyright (C) 2008-2016 by LMI Technologies Inc.  All rights reserved.
# Distributed under the terms of the MIT License.
# Redistributed files must retain the above copyright notice.

import re
import os
import errno
from . import Project
from . import WriterBase
from . import Utils
from . import Xml

def GmakePath(projectFileName):
    herePath = os.path.dirname(os.path.realpath(__file__))
    gmakePath = os.path.normpath(os.path.join(herePath, '..', '..', '..', 'extern', 'Make', 'gmake.exe'))
    return os.path.relpath(gmakePath, os.path.dirname(projectFileName))

class Writer(WriterBase.WriterBase):

    #
    # Supported platforms and project types
    #
    _platforms = [
        'Arm7',
        'Arm7u',
    ]

    _projects = [
        'C/CPP',
    ]

    def __init__(self, platforms, projects):
        WriterBase.WriterBase.__init__(self, self._platforms, platforms, self._projects, projects)

    @property
    def Name(self):
        return 'WrWb'

    # unique project identifier
    def Id(self, project, config, suffix):
        name = project.Name + config.Name + suffix

        hash = 0
        for char in name:
            hash = hash * 31 + ord(char)

        return str(hash).zfill(10)[0:10]

    def WriteLinkedResourcesFilters(self, project, sourceGroup, filterItem, filterName = ''):

        for childGroup in sourceGroup.Groups:
            linkItem = filterItem.AddElem('link')

            linkItem.AddElem('name', filterName + childGroup.Name)
            linkItem.AddElem('type', '2')
            linkItem.AddElem('locationURI', 'virtual:/virtual')

            self.WriteLinkedResourcesFilters(project, childGroup, filterItem, filterName + childGroup.Name + '/')

    def WriteLinkedResourcesFiles(self, project, sourceGroup, filterItem, filterName = ''):

        for childGroup in sourceGroup.Groups:
            self.WriteLinkedResourcesFiles(project, childGroup, filterItem, filterName + childGroup.Name + '/')

        for src in sourceGroup.Sources:
            linkItem = filterItem.AddElem('link')

            linkItem.AddElem('name', filterName + os.path.basename(src.Name))
            linkItem.AddElem('type', '1')
            linkItem.AddElem('location', os.path.join(os.path.dirname(project.FileName), src.Name))

    def WriteLinkedResources(self, project, sourceGroup, filterItem):
        self.WriteLinkedResourcesFilters(project, sourceGroup, filterItem)
        self.WriteLinkedResourcesFiles(project, sourceGroup, filterItem)

    def WriteDotProject(self, project, config, fileName):

        root = Xml.Element('projectDescription')
        suffix = '-Uni' if config.Platform == 'Arm7u' else ''

        root.AddElem('name', project.Name + '-' + config.Name + suffix)
        root.AddElem('comment', '')

        dependencies = self.SelectDependencies(project)
        projectsItem = root.AddElem('projects')

        if config.Type == 'Library' or len(dependencies) == 0:
            projectsItem.AddElem()
        else:
            for dependency in dependencies:
                projectsItem.AddElem('project', dependency.Name + '-' + config.Name + suffix)

        buildSpecItem = root.AddElem('buildSpec')
        buildCommandItem = buildSpecItem.AddElem('buildCommand')
        buildCommandItem.AddElem('name', 'com.windriver.ide.core.wrbuilder')
        buildCommandItem.AddElem('triggers', 'clean,full,incremental,')
        argumentsItem = buildCommandItem.AddElem('arguments')
        argumentsItem.AddElem()

        naturesItem = root.AddElem('natures')
        naturesItem.AddElem('nature', 'com.windriver.ide.core.wrnature')
        naturesItem.AddElem('nature', 'com.windriver.ide.core.wrcorenature')
        naturesItem.AddElem('nature', 'org.eclipse.cdt.core.cnature')
        naturesItem.AddElem('nature', 'org.eclipse.cdt.core.ccnature')

        linkedResourcesItem = root.AddElem('linkedResources')
        self.WriteLinkedResources(project, project.SourceGroups, linkedResourcesItem)

        root.WriteDotProject(root, fileName)

    def WriteDotCProject(self, project, config, fileName):

        root = Xml.Element('cproject')
        suffix = '-Uni' if config.Platform == 'Arm7u' else ''

        root.AddAttr('storage_type_id', 'org.eclipse.cdt.core.XmlProjectDescriptionStorage')

        storageModuleItem = root.AddElem('storageModule')
        storageModuleItem.AddAttr('moduleId', 'org.eclipse.cdt.core.settings')
        cconfigurationItem = storageModuleItem.AddElem('cconfiguration')
        cconfigurationItem.AddAttr('id', 'org.eclipse.cdt.core.default.config.' + self.Id(project, config, suffix))

        cconfigStorageModuleItem = cconfigurationItem.AddElem('storageModule')
        cconfigStorageModuleItem.AddAttr('buildSystemId', 'org.eclipse.cdt.core.defaultConfigDataProvider')
        cconfigStorageModuleItem.AddAttr('id', 'org.eclipse.cdt.core.default.config.' + self.Id(project, config, suffix))
        cconfigStorageModuleItem.AddAttr('moduleId', 'org.eclipse.cdt.core.settings')
        cconfigStorageModuleItem.AddAttr('name', 'Configuration')
        cconfigStorageModuleItem.AddElem('externalSettings')

        extensionsItem = cconfigStorageModuleItem.AddElem('extensions')

        extensionItem = extensionsItem.AddElem('extension')
        extensionItem.AddAttr('id', 'com.windriver.ide.core.WRElfParserVxWorks')
        extensionItem.AddAttr('point', 'org.eclipse.cdt.core.BinaryParser')

        extensionItem = extensionsItem.AddElem('extension')
        extensionItem.AddAttr('id', 'org.eclipse.cdt.core.GNU_ELF')
        extensionItem.AddAttr('point', 'org.eclipse.cdt.core.BinaryParser')

        cconfigStorageModuleItem = cconfigurationItem.AddElem('storageModule')
        cconfigStorageModuleItem.AddAttr('moduleId', 'org.eclipse.cdt.core.externalSettings')

        storageModuleItem = root.AddElem('storageModule')
        storageModuleItem.AddAttr('moduleId', 'org.eclipse.cdt.core.pathentry')

        pathentryItem = storageModuleItem.AddElem('pathentry')
        pathentryItem.AddAttr('kind', 'src')
        pathentryItem.AddAttr('path', '')

        pathentryItem = storageModuleItem.AddElem('pathentry')
        pathentryItem.AddAttr('kind', 'out')
        pathentryItem.AddAttr('path', '')

        pathentryItem = storageModuleItem.AddElem('pathentry')
        pathentryItem.AddAttr('kind', 'con')
        pathentryItem.AddAttr('path', 'org.eclipse.cdt.make.core.DISCOVERED_SCANNER_INFO')

        pathentryItem = storageModuleItem.AddElem('pathentry')
        pathentryItem.AddAttr('kind', 'con')
        pathentryItem.AddAttr('path', 'com.windriver.ide.core.build.model.WR_USERDEFINED_CONTAINER')

        scannerConfigurationItem = storageModuleItem = root.AddElem('storageModule')
        storageModuleItem.AddAttr('moduleId', 'scannerConfiguration')

        autodiscoveryItem = scannerConfigurationItem.AddElem('autodiscovery')
        autodiscoveryItem.AddAttr('enabled', 'true')
        autodiscoveryItem.AddAttr('problemReportingEnabled', 'true')
        autodiscoveryItem.AddAttr('selectedProfileId', 'com.windriver.ide.staticanalysis.core.WRStandardMakePerFileProfile')

        profileItem = scannerConfigurationItem.AddElem('profile')
        profileItem.AddAttr('id', 'com.windriver.ide.staticanalysis.core.WRStandardMakePerFileProfile')

        buildOutputProviderItem = profileItem.AddElem('buildOutputProvider')
        openActionItem = buildOutputProviderItem.AddElem('openAction')
        openActionItem.AddAttr('enabled', 'true')
        openActionItem.AddAttr('filePath', '')
        parserItem = buildOutputProviderItem.AddElem('parser')
        parserItem.AddAttr('enabled', 'true')

        scannerInfoProviderItem = profileItem.AddElem('scannerInfoProvider')
        scannerInfoProviderItem.AddAttr('id', 'compilerBuiltinProvider')
        runActionItem = scannerInfoProviderItem.AddElem('runAction')
        runActionItem.AddAttr('useDefault', 'true')
        parserItem = scannerInfoProviderItem.AddElem('parser')
        parserItem.AddAttr('enabled', 'false')

        storageModuleItem = root.AddElem('storageModule')
        storageModuleItem.AddAttr('moduleId', 'org.eclipse.cdt.core.LanguageSettingsProviders')
        storageModuleItem = root.AddElem('storageModule')
        storageModuleItem.AddAttr('moduleId', 'org.eclipse.cdt.internal.ui.text.commentOwnerProjectMappings')

        storageModuleItem = root.AddElem('storageModule')
        storageModuleItem.AddAttr('moduleId', 'userdefinedContainer')
        storageModuleItem.AddElem('indexAllFiles').AddAttr('value', 'false')
        storageModuleItem.AddElem('initialized').AddAttr('value', 'true')

        compilerFlags = ''
        for dir in config.IncludeDirs: compilerFlags += ' -I%s' % (os.path.join(os.path.dirname(project.FileName), dir))
        for define in config.Defines:  compilerFlags += ' -D%s' % (define)
        storageModuleItem.AddElem('compiler').AddAttr('name', 'ccarm' + compilerFlags)

        root.WriteDotCProject(root, fileName, True)

    def WriteDotWrProject(self, project, config, fileName):

        root = Xml.Element('wrxml')
        makeFileSuffix = '-WrWbu' if config.Platform == 'Arm7u' else '-WrWb'
        suffix = '-Uni' if config.Platform == 'Arm7u' else ''

        propertiesItem = root.AddElem('properties')
        propertiesItem.AddAttr('platform', 'VxWorks')
        propertiesItem.AddAttr('platform_name', 'vxworks-6.9')
        propertiesItem.AddAttr('root', '1')
        propertiesItem.AddAttr('type', 'UserProject')

        if config.Type != 'Library':
            subProjectsItem = root.AddElem('subprojects')

            for dependency in self.SelectDependencies(project):
                subProjectItem = subProjectsItem.AddElem('subproject')
                subProjectItem.AddAttr('project', '/' + dependency.Name + '-' + config.Name + suffix)

        attributesItem = root.AddElem('attributes')
        mapAttributeItem = attributesItem.AddElem('mapAttribute')

        for attribute in [

                [ 'intAttribute', 'BuildSupportEnabled', '2' ],
                [ 'stringAttribute', 'BLD::Info|cmd', '%s --no-print-directory -C %s -f %s%s.mk config=%s' % (
                    GmakePath(fileName),
                    os.path.dirname(project.FileName),
                    project.Name,
                    makeFileSuffix,
                    config.Name) ],

                [ 'listAttribute', 'BLD::Info|Tools',
                    [
                        'Build Target Rule',
                    ]
                ],

                [ 'booleanAttribute', 'BLD::Info|Tool|Build Target Rule|passAble', 'true' ],
                [ 'stringAttribute', 'BLD::Info|buildRule', '' ],
                [ 'stringAttribute', 'BLD::Info|folderBuildRule', '' ],
                [ 'stringAttribute', 'BLD::Info|cleanRule', 'clean' ],
                [ 'stringAttribute', 'BLD::Info|folderCleanRule', 'clean' ],
                [ 'stringAttribute', 'BLD::Info|rebuildRule', 'clean all' ],
                [ 'intAttribute', 'BLD::Info|refreshType', '0' ],
            ]:
            if isinstance(attribute[2], list):
                attributeItem = mapAttributeItem.AddElem(attribute[0])
                attributeItem.AddAttr('key', attribute[1])
                for stringAttribute in attribute[2]:
                    stringAttributeItem = attributeItem.AddElem('stringAttribute')
                    stringAttributeItem.AddAttr('value', stringAttribute)
            else:
                attributeItem = mapAttributeItem.AddElem(attribute[0])
                attributeItem.AddAttr('key', attribute[1])
                attributeItem.AddAttr('value', attribute[2])

        root.WriteDotWrProject(root, fileName)

    def WriteProject(self, project):

        solution = project.Solution

        for config in self.SelectConfigurations(project):

            extension = '.wrwbu' if config.Platform == 'Arm7u' else '.wrwb'

            if solution is not None:
                dirName = os.path.join(os.path.dirname(solution.FileName), '.' + solution.Name + extension, project.Name + '-' + config.Name)
            else:
                dirName = os.path.join(os.path.dirname(project.FileName), '.' + project.Name + '-' + config.Name + extension)

            try:
                os.makedirs(dirName)
            except OSError as exc:
                if exc.errno != errno.EEXIST:
                    raise

            self.WriteDotProject(project, config, os.path.join(dirName, '.project'))
            self.WriteDotCProject(project, config, os.path.join(dirName, '.cproject'))
            self.WriteDotWrProject(project, config, os.path.join(dirName, '.wrproject'))

    def WriteSolution(self, solution):
        # not implemented
        return
