/**
* @file    kVsImageNoiseBlockInfo.h
* @brief   Declares the kVsImageNoiseBlockInfo class.
*
* @internal
* Copyright (C) 2014-2022 by LMI Technologies Inc. All rights reserved.
*/
#ifndef LMITECH_KVISION_VS_IMAGE_NOISE_BLOCK_INFO_H_INCLUDED
#define LMITECH_KVISION_VS_IMAGE_NOISE_BLOCK_INFO_H_INCLUDED

#include <kVision/Common/kVision.h>
#include <kApi/Data/kXml.h>

#define kVS_IMAGE_NOISE_BLOCK_NAME           "ImageNoise"
#define kVS_IMAGE_NOISE_BLOCK_TYPE_NAME      "kVsImageNoise"

#define kVS_IMAGE_NOISE_BLOCK_IN_IMAGE_NAME   "InImage"

#define kVS_IMAGE_NOISE_BLOCK_OUT_STATS_NAME                  "OutStats"
#define kVS_IMAGE_NOISE_BLOCK_OUT_SPATIAL_IMAGE_NAME          "OutSpatialImage"
#define kVS_IMAGE_NOISE_BLOCK_OUT_GROUND_IMAGE_NAME           "OutGroundImage"
#define kVS_IMAGE_NOISE_BLOCK_OUT_TEMPORAL_IMAGE_NAME         "OutTemporalImage"

typedef enum
{
    kVS_IMAGE_NOISE_BLOCK_IN_IMAGE  = 0,
    kVS_IMAGE_NOISE_BLOCK_OUT_STATS = 1,
    kVS_IMAGE_NOISE_BLOCK_OUT_SPATIAL_IMAGE= 2,
    kVS_IMAGE_NOISE_BLOCK_OUT_GROUND_IMAGE = 3,
    kVS_IMAGE_NOISE_BLOCK_OUT_TEMPORAL_IMAGE = 4
} kVsImageNoiseBlockPort;

/**
* @class   kVsImageNoiseBlockInfo
* @ingroup kVision
* @brief   Settings class for kVsImageNoiseBlock
*/
typedef kObject kVsImageNoiseBlockInfo;

/**
* Constructs a kVsImageNoiseBlockInfo object
*
* @public              @memberof kVsImageNoiseBlockInfo
* @param   info        Destination for the constructed object handle.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kVsImageNoiseBlockInfo_Construct(kVsImageNoiseBlockInfo* info, kAlloc allocator);

/**
* Reads values from a kXml object
*
* @public              @memberof kVsImageNoiseBlockInfo
* @param   info        Info object.
* @param   xml         Source kXml object
* @param   item        Root xml node
* @return              Operation status.
*/
kVsFx(kStatus) kVsImageNoiseBlockInfo_Read(kVsImageNoiseBlockInfo info, kXml xml, kXmlItem item);

/**
* Writes values to a kXml object
*
* @public              @memberof kVsImageNoiseBlockInfo
* @param   info        Info object.
* @param   xml         destination kXml object
* @param   item        Root xml node
* @return              Operation status.
*/
kVsFx(kStatus) kVsImageNoiseBlockInfo_Write(kVsImageNoiseBlockInfo info, kXml xml, kXmlItem item);

/**
* Loads a new kVsImageNoiseBlockInfo from a kXml object
*
* @public              @memberof kVsImageNoiseBlockInfo
* @param   info        Destination for the constructed object handle.
* @param   xml         Source kXml object
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kVsImageNoiseBlockInfo_Load(kVsImageNoiseBlockInfo* info, kXml xml, kAlloc allocator);

/**
* Saves a kVsImageNoiseBlockInfo to a new kXml object
*
* @public              @memberof kVsImageNoiseBlockInfo
* @param   info        Object to be saved
* @param   xml         Pointer to a destination kXml handle. 
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kVsImageNoiseBlockInfo_Save(kVsImageNoiseBlockInfo info, kXml* xml, kAlloc allocator);

/**
* Get the number of image samples used in noise processing
*
* @public              @memberof kVsImageNoiseBlockInfo
* @param   info        Info object
* @return              Number of image samples used in noise processing
*/
kVsFx(k32u) kVsImageNoiseBlockInfo_SampleCount(kVsImageNoiseBlockInfo info);

/**
* Set the number of image samples used in noise processing
*
* @public              @memberof kVsImageNoiseBlockInfo
* @param   info        Info object
* @param   count       Sample count
* @return              Operation status
*/
kVsFx(kStatus) kVsImageNoiseBlockInfo_SetSampleCount(kVsImageNoiseBlockInfo info, k32u count);

/**
* Get the size of the smoothing window used to remove the smooth ground variations from spatial noise calculations. The value is
* returned as a fraction (0 to 1) of input image width;
*
* @public              @memberof kVsImageNoiseBlockInfo
* @param   info        Info object
* @return              Size of the smoothing window used for ground calculations
*/
kVsFx(k64f) kVsImageNoiseBlockInfo_GroundSmoothWindow(kVsImageNoiseBlockInfo info);

/**
* Set the size of the smoothing window used to remove the smooth ground variations from spatial noise calculations. The value is
* set as a fraction (0 to 1) of input image width;
*
* @public              @memberof kVsImageNoiseBlockInfo
* @param   info        Info object
* @param   window      Sample count
* @return              Operation status
*/
kVsFx(kStatus) kVsImageNoiseBlockInfo_SetGroundSmoothWindow(kVsImageNoiseBlockInfo info, k64f window);

/**
* Get the size of the region of interest used for calculating the image noise statistics. The region is returned as a fraction (0..1) 
* of the image size. The ROI is always centered within the image
*
* @public              @memberof kVsImageNoiseBlockInfo
* @param   info        Info object
* @return              Size of the region of interest used for calculating image noise statistics
*/
kVsFx(k64f) kVsImageNoiseBlockInfo_Roi(kVsImageNoiseBlockInfo info);

/**
* Set the size of the region of interest used for calculating the image noise statistics. The region is set as a fraction (0..1)
* of the image size. The ROI is always centered within the image
*
* @public              @memberof kVsImageNoiseBlockInfo
* @param   info        Info object
* @param   roi         Size of the region of interest used for calculating image noise statistics
* @return              Operation status
*/

kVsFx(kStatus) kVsImageNoiseBlockInfo_SetRoi(kVsImageNoiseBlockInfo info, k64f roi);

#include <kVision/Vs/kVsImageNoiseBlockInfo.x.h>

#endif
