/**
 * @file    kVsImageFilter.h
 * @brief   Declares the ImageFilter class.
 *
 * @internal
 * Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_VISION_IMAGE_FILTER_H
#define K_VISION_IMAGE_FILTER_H

#include <kVision/Common/kVision.h>

/**
* @class       kVsImageFilter
* @extends     kObject
* @ingroup     kVision-Vs
* @brief       Base class for optimized image filters. The class is designed to handle either kArray2 or kImage objects of
*              k8u, k16s, k32s, or k64f types. The user can set parameters such as input data type, border behaviour,
*              filter window size and data buffer width, or the user can use the default settings. Data buffers are allocated
*              to default sizes in the Init function.  Setup and Apply are interface functions extended via VSetup and VApply
*              (to be implemented in the subclass). The intention of Setup is to reallocate buffers if the user sets custom data types
*              and widths, and the intention of Apply is to handle different data types and run the filter operation. Setup
*              can be called prior to Apply so that buffers are preallocated. Otherwise, Apply should reallocate the buffers (if
*              needed) on the first call by calling Setup. See kVsMovingAvg and kVsAdaptiveNorm for examples on extending this class.
*/
typedef kObject kVsImageFilter;

/**
* Image filter border behaviour types. The input object can either be padded by zeroes
* (kVS_IMAGE_FILTER_BORDER_TYPE_ZERO), padded by repeating the closest pixel value
* (kVS_IMAGE_FILTER_BORDER_TYPE_REPEAT), or the output object can be trimmed by nulls
* (kVS_IMAGE_FILTER_BORDER_TYPE_TRIM).
*
* @relates     kVsInitializer
*/
typedef enum kVsImageFilterBorderType
{
    kVS_IMAGE_FILTER_BORDER_TYPE_REPEAT,
    kVS_IMAGE_FILTER_BORDER_TYPE_ZERO,
    kVS_IMAGE_FILTER_BORDER_TYPE_TRIM
} kVsImageFilterBorderType;

/**
* Updates working buffers of the filter subclass to reflect current set of algorithm parameters.
* Calling this function is optional, as this validation step is also performed during each execution
* of the algorithm (kVsImageFilter_Apply). However, the initialization time may be non-negligible, which
* would affect the first execution of the algorithm. To eliminate the added delay from the first algorithm
* execution, the user should call kVsImageFilter_Setup after all of the parameters have been configured.
*
* @public              @memberof kVsImageFilter
* @param   filter      Filter object
* @return              Operation status.
*/
kVsFx(kStatus) kVsImageFilter_Setup(kVsImageFilter filter);

/**
* Applies the operation of the filter subclass to either an image or a kArray2 object of item types k8u,
* k16s, k32s, and k64f. The output object must be the same type (and contain the same item type) as the
* input object. The input and output cannot point to the same object.
*
* @public              @memberof kVsImageFilter
* @param   filter      Filter object
* @param   input       Input object (type: kImage or kArray2. Item type: k8u, k16s, k32s, or k64f.)
* @param   output      Output object (cannot be the same object as the input. Must be the same type and item type as the input)
* @return              Operation status.
*/
kVsFx(kStatus) kVsImageFilter_Apply(kVsImageFilter filter, kObject input, kObject output);

/**
* Set the size of the filtering window. The value must be greater or equal to 1 and odd. Default is
* kVS_IMAGE_FILTER_DEFAULT_WINDOW_SIZE.
*
* @public              @memberof kVsImageFilter
* @param   filter      Filter object
* @param   window      Filter window size (px)
* @return              Operation status.
*/
kVsFx(kStatus) kVsImageFilter_SetWindow(kVsImageFilter filter, kSize window);

/**
*  Get filter window size
*
* @public              @memberof kVsImageFilter
* @param   filter      Filter object
* @return              Filter window size (px)
*/
kVsFx(kSize) kVsImageFilter_Window(kVsImageFilter filter);

/**
* Sets the width of the expected input and output objects. Default is kVS_IMAGE_FILTER_DEFAULT_WIDTH.
*
* @public              @memberof kVsImageFilter
* @param   filter      Filter object
* @param   width       Width of the input and output objects
* @return              Operation status.
*/
kVsFx(kStatus) kVsImageFilter_SetDataWidth(kVsImageFilter filter, kSize width);

/**
* Gets the width of the input and output objects
*
* @public              @memberof kVsImageFilter
* @param   filter      Filter object
* @return              Width of the input and output objects.
*/
kVsFx(kSize) kVsImageFilter_DataWidth(kVsImageFilter filter);

/**
* Sets the border behaviour type. The border region is equal to half the filter window width. For the output
* object width and height to match the input object, the input object can either be padded by zeroes
* (kVS_IMAGE_FILTER_BORDER_TYPE_ZERO), padded by repeating the closest pixel value (kVS_IMAGE_FILTER_BORDER_TYPE_REPEAT),
* or the output object can be trimmed by nulls (kVS_IMAGE_FILTER_BORDER_TYPE_TRIM). Default is kVS_IMAGE_FILTER_BORDER_TYPE_REPEAT.
*
* @public              @memberof kVsImageFilter
* @param   filter      Filter object
* @param   border      Border type
* @return              Operation status.
*/
kVsFx(kStatus) kVsImageFilter_SetBorderType(kVsImageFilter filter, kVsImageFilterBorderType border);

/**
* Gets the border type
*
* @public              @memberof kVsImageFilter
* @param   filter      Filter object
* @return              Border type
*/
kVsFx(kVsImageFilterBorderType) kVsImageFilter_BorderType(kVsImageFilter filter);

/**
* Set the item type of the input object. The possible options are k8u, k16s, k32s, or k64f.
* Default is kVS_IMAGE_FILTER_DEFAULT_DATA_TYPE.
*
* @public              @memberof kVsImageFilter
* @param   filter      Filter object
* @param   type        Input object item type (k8u, k16s, k32s, or k64f)
* @return              Operation status.
*/
kVsFx(kStatus) kVsImageFilter_SetDataType(kVsImageFilter filter, kType type);

/**
* Gets the input object item type
*
* @public              @memberof kVsImageFilter
* @param   filter      Filter object
* @return              Input object item type
*/
kVsFx(kType) kVsImageFilter_DataType(kVsImageFilter filter);

/**
* Sets the minimum fill count. If the number of non-null pixels in the filter window is less than the fill count,
* the average for the window is set to null. The parameter is specified in pixels. Default is set to half the
* default window size (kVS_IMAGE_FILTER_DEFAULT_FILL_COUNT).
*
* @public              @memberof kVsImageFilter
* @param   filter      Filter object
* @param   fillCount   Minimum required number of non-null pixels in window (px)
* @return              Operation status.
*/
kVsFx(kStatus) kVsImageFilter_SetFillCount(kVsImageFilter filter, kSize fillCount);

/**
*  Get the minimum required number of non-null pixels in window for non-null result (px)
*
* @public              @memberof kVsImageFilter
* @param   filter      Filter object
* @return              Minimum required number of non-null pixels in window (px)
*/
kVsFx(kSize) kVsImageFilter_FillCount(kVsImageFilter filter);

#include <kVision/Vs/kVsImageFilter.x.h>

#endif  /* K_VISION_IMAGE_FILTER_H */


