/**
* @file     kVsDynamicExposure.h
* @brief    Declares the kVsDynamicExposure class
*
* @internal
* Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
*/

#ifndef K_VISION_DYNAMIC_EXPOSURE_H
#define K_VISION_DYNAMIC_EXPOSURE_H

#include <kVision/Common/kVision.h>

/**
* @class       kVsDynamicExposure
* @extends     kObject
* @ingroup     kVision-Vs
* @brief       This is the class that simulates Gocator PL dynamic exposure behavior. It expects array of sum threshold and adjustment 
*              as a scale of exposure adjustments, max exposure and min exposure as clamp range, spot pixel average sum from current image stats.
*              Spot pixel average sum is the statistics that represents the spot strength per pixel.
*              Array of sum threshold is a list of spot strength sum level corresponding to the list of exposure adjustment. 
*              For example, when the spot pixel average sum falls in spot sum threshold 1 and 2, the adjustment for the next exposure will be
*              the second adjustment from the list. This class is only responsible for determining the correct exposure value,
*              any setup to a live sensor module should be done by the user. During the operation, this class excepts current exposure,
*              Spot sums to generate next exposure value to be set.
*
*/

typedef kObject kVsDynamicExposure;

/**
* Constructs a kVsDynamicExposure object.
*
* @public                   @memberof kVsDynamicExposure
* @param    dynExp          Destination for the constructed object handle.
* @param    sumThreshold    ArrayList of ranges of spot sum level to determine adjustment to next Exposure. kArrayList<kSize>
* @param    adjustment      ArrayList of discrete value of exposure adjustment to next exposure value to set. kArrayList<kSSize>
* @return                   Operation status.
*/
kVsFx(kStatus) kVsDynamicExposure_Construct(kVsDynamicExposure* dynExp, kArrayList sumThresholds, kArrayList adjustments, kAlloc allocator);



/**
* Updates exposure value. This function requires pixel spot sum average which is the statistics representing the
* the spot sum per pixel, the number of the total spots found, the current exposure value and pointer to the output exposure value. 
* The operation requires the number of values in adjustment to be one more than that of pxiSumThrehold and all buffer is in
* ascending order. After the operation the function will set the output exposure to the next exposure value.
* Adjustment Logic:
*   If total_spots < min_num_spots and new_exposure > max exposure then set new exposure to min exposure
*   If total_spots < min_num_spots and current exposure is < max exposure then use the minimum result from the adjustment table
*   If the quotient result = 0 then use the minimum result from the adjustment table
*   If the spot stat is greater than the max result use the max result
*   If the new_exposure < min_exposure then set new exposure to min exposure
*
* @public                  @memberof kVsDynamicExposure
* @param   dynExp          Dynamic exposure object
* @param   spotSumAverage  Pixel average sum
* @param   currentExposure Current exposure value
* @param   spotCount       Number of spots found
* @param   outputExposure  pointer to the new exposure value
* @return                  Next exposure value.
*/
kVsFx(kStatus) kVsDynamicExposure_Update(kVsDynamicExposure dynExp, kSize spotSumAverage, kSize currentExposure, kSize spotCount, kSize* outputExposure);

/**
* Sets the minExposure of dynamic exposure, this minimum exposure threshold when dynamically adjusting exposure.
*
* @public              @memberof kVsDynamicExposure
* @param   dynExp      Dynamic exposure object
* @param   minExposure Minimum exposure threshold
* @return              Operation status.
*/
kVsFx(kStatus) kVsDynamicExposure_SetMinExposure(kVsDynamicExposure dynExp, kSize minExposure);

/**
* Gets the minExposure of dynamic exposure.
*
* @public              @memberof kVsDynamicExposure
* @param   dynExp      DynamicExposure object
* @return              MinExposure.
*/
kVsFx(kSize) kVsDynamicExposure_MinExposure(kVsDynamicExposure dynExp);

/**
* Sets the maxExposure of dynamic exposure, this maximum exposure threshold when dynamically adjusting exposure.
*
* @public              @memberof kVsDynamicExposure
* @param   dynExp      Dynamic exposure object
* @param   maxExposure Maximum exposure threshold
* @return              Operation status.
*/
kVsFx(kStatus) kVsDynamicExposure_SetMaxExposure(kVsDynamicExposure dynExp, kSize maxExposure);

/**
* Gets the maxExposure of dynamic exposure.
*
* @public              @memberof kVsDynamicExposure
* @param   dynExp      DynamicExposure object
* @return              MaxExposure.
*/
kVsFx(kSize) kVsDynamicExposure_MaxExposure(kVsDynamicExposure dynExp);

/**
* Sets the minimum spot count threshold hold for dynamic exposure.
* If the new exposure is greater than the maximum exposure and the spot count is less than this threshold,
* next exposure will be the minimum exposure. If the new exposure is smaller or equal to the max exposure,
* the adjustment to the next exposure will be the smallest adjustment from the list.
*
* @public              @memberof kVsDynamicExposure
* @param   dynExp      Dynamic exposure object
* @param   exp         Exposure
* @return              Operation status.
*/
kVsFx(kStatus) kVsDynamicExposure_SetMinSpotCount(kVsDynamicExposure dynExp, kSize minSpotCount);

/**
* Gets the minSpotNumber of dynamic exposure
*
* @public              @memberof kVsDynamicExposure
* @param   dynExp      DynamicExposure object
* @return              MinSpotNumber
*/
kVsFx(kSize) kVsDynamicExposure_MinSpotCount(kVsDynamicExposure dynExp);

/**
* Sets the pixel threshold array of dynamic exposure, at current version, the number of elements of this array is expected to be
* one less than the adjustment array. This will be the ranges of spot pixel sums when adjusting exposure.
*
* @public              @memberof kVsDynamicExposure
* @param   dynExp      Dynamic exposure object
* @param   threshold   Array of pixel sum thresholds. kArrayList<kSize>
* @return              Operation status.
*/
kVsFx(kStatus) kVsDynamicExposure_SetSumThresholds(kVsDynamicExposure dynExp, kArrayList thresholds);

/**
* Gets the pointer to pixel sum threshold of dynamic exposure.
*
* @public              @memberof kVsDynamicExposure
* @param   dynExp      DynamicExposure object
* @return              Pointer to pixel sum thresholds
*/
kVsFx(kSize*) kVsDynamicExposure_SumThresholds(kVsDynamicExposure dynExp);

/**
* Sets the adjustment array of dynamic exposure, at current version, the number of elements of this array is expected to be
* one more than the pixel sum threshold array. This will be the ranges of spot pixel sums when adjusting exposure.
*
* @public              @memberof kVsDynamicExposure
* @param   dynExp      Dynamic exposure object
* @param   adjustment  ArrayList of pixel sum adjustment. kArrayList <kSSize>(percentage value)
* @return              Operation status.
*/
kVsFx(kStatus) kVsDynamicExposure_SetAdjustments(kVsDynamicExposure dynExp, kArrayList adjustments);

/**
* Gets the pointer to pixel sum threshold of dynamic exposure.
*
* @public              @memberof kVsDynamicExposure
* @param   dynExp      DynamicExposure object
* @return              Pointer to adjustment buffer
*/
kVsFx(kSSize*) kVsDynamicExposure_Adjustments(kVsDynamicExposure dynExp);

#include <kVision/Vs/kVsDynamicExposure.x.h>

;

#endif /*K_VISION_DYNAMIC_EXPOSURE_H*/

