/**
* @file    kS3dVolumeCheckSparkAlg.h
* @brief   Declares the kS3dVolumeCheckSparkAlg class. 
*
* @internal
* Copyright (C) 2016-2022 by LMI Technologies Inc. All rights reserved.
*/

#ifndef KVISION_KS3D_VOLUMECHECKSPARK_H
#define KVISION_KS3D_VOLUMECHECKSPARK_H

#include <kVision/S3d/kS3dVolumeCheckUtilities.h>
#include <kVision/S3d/kS3dVolumeCheckAlign.h>

/**
* @class       kS3dVolumeCheckSparkAlg
* @extends     kObject
* @ingroup     kVision-S3d
* @brief       Class for software cover of the spark hole.
*/
typedef kObject kS3dVolumeCheckSparkAlg;

/**
* Constructs a kS3dVolumeCheckSparkAlg object
*
* @public              @memberof kS3dVolumeCheckSparkAlg
* @param   spark       Destination for the constructed object handle.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckSparkAlg_Construct(kS3dVolumeCheckSparkAlg* spark, kAlloc allocator);

/**
* Executes the algorithm for software cover of the spark.
*
* @public              @memberof kS3dVolumeCheckSparkAlg
* @param   spark       kS3dVolumeCheckSparkAlg object
* @param   align       kS3dVolumeCheckAlign object
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckSparkAlg_Run(kS3dVolumeCheckSparkAlg spark, kS3dVolumeCheckAlign align);

/**
*  Get center of spark hole in mm.
*
* @public              @memberof kS3dVolumeCheckSparkAlg
* @param   spark       kS3dVolumeCheckSparkAlg object.
* @return              center of spark hole.
*/
kVsFx(kPoint3d64f) kS3dVolumeCheckSparkAlg_SparkCenter(kS3dVolumeCheckSparkAlg spark);

/**
*  Set center of spark hole in mm.
*
* @public              @memberof kS3dVolumeCheckSparkAlg
* @param   spark       kS3dVolumeCheckSparkAlg object.
* @param   center      center of spark hole.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckSparkAlg_SetSparkCenter(kS3dVolumeCheckSparkAlg spark, kPoint3d64f center);

/**
*  Get rotation angle of spark covering plane in degree.
*
* @public              @memberof kS3dVolumeCheckSparkAlg
* @param   spark       kS3dVolumeCheckSparkAlg object.
* @return              rotation angle.
*/
kVsFx(k64f)  kS3dVolumeCheckSparkAlg_SparkPolarAngle(kS3dVolumeCheckSparkAlg spark);

/**
*  Set rotation angle of spark covering plane in degree.
*
* @public              @memberof kS3dVolumeCheckSparkAlg
* @param   spark       kS3dVolumeCheckSparkAlg object.
* @param   polarAngle  rotation angle.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckSparkAlg_SetSparkPolarAngle(kS3dVolumeCheckSparkAlg spark, k64f polarAngle);

/**
*  Get slope angle of spark covering plane in degree.
*
* @public              @memberof kS3dVolumeCheckSparkAlg
* @param   spark       kS3dVolumeCheckSparkAlg object.
* @return              slope angle.
*/
kVsFx(k64f)  kS3dVolumeCheckSparkAlg_SparkSlopeAngle(kS3dVolumeCheckSparkAlg spark);

/**
*  Set slope angle of spark covering plane in degree.
*
* @public              @memberof kS3dVolumeCheckSparkAlg
* @param   spark       kS3dVolumeCheckSparkAlg object.
* @param   slopeAngle  slope angle.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckSparkAlg_SetSparkSlopeAngle(kS3dVolumeCheckSparkAlg spark, k64f slopeAngle);

/**
*  Get nominal diameter of spark hole in mm.
*
* @public              @memberof kS3dVolumeCheckSparkAlg
* @param   spark       kS3dVolumeCheckSparkAlg object.
* @return              nominal diameter.
*/
kVsFx(k64f)  kS3dVolumeCheckSparkAlg_SparkDiameter(kS3dVolumeCheckSparkAlg spark);

/**
*  Set nominal diameter of spark hole in mm.
*
* @public              @memberof kS3dVolumeCheckSparkAlg
* @param   spark       kS3dVolumeCheckSparkAlg object.
* @param   diameter    nominal diameter.
* @return              Operation status.
*/
kVsFx(kStatus)  kS3dVolumeCheckSparkAlg_SetSparkDiameter(kS3dVolumeCheckSparkAlg spark, k64f diameter);

/**
* Queries whether software cover for the spark is provided.
*
* @public              @memberof kS3dVolumeCheckSparkAlg
* @param   spark       kS3dVolumeCheckSparkAlg object.
* @return              is provided?
*/
kVsFx(kBool) kS3dVolumeCheckSparkAlg_SparkSoftCover(kS3dVolumeCheckSparkAlg spark);

/**
*  Set a boolean value to determine of the software cover for the spark.
*
* @public              @memberof kS3dVolumeCheckSparkAlg
* @param   spark       kS3dVolumeCheckSparkAlg object.
* @param   softCover   Set value.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckSparkAlg_SetSparkSoftCover(kS3dVolumeCheckSparkAlg spark, kBool softCover);

/**
*  Fill the spark hole in the output height map.
*
* @public              @memberof kS3dVolumeCheckSparkAlg
* @param   spark       kS3dVolumeCheckSparkAlg object.
* @param   output      output height map of kArray2< k32s >.
* @param   valid       valid map of kArray2 < k8u >. The surface is segmented into different parts.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckSparkAlg_AddSparkArea(kS3dVolumeCheckSparkAlg spark, kArray2 output, kArray2 valid, k64f zResolution, k64f xyScale);

#include <kVision/S3d/kS3dVolumeCheckSparkAlg.x.h>

#endif /* #ifndef KVISION_KS3D_VOLUMECHECKSPARK_H */
