/**
* @file    kS3dVolumeCheckInputBlockInfo.h
* @brief   Declares the kS3dVolumeCheckInputBlock class.
*
* @internal
* Copyright (C) 2016-2022 by LMI Technologies Inc. All rights reserved.
*/

#ifndef KVISION_KS3D_VOLUMECHECKINPUTBLOCKINFO_H
#define KVISION_KS3D_VOLUMECHECKINPUTBLOCKINFO_H

#include <kApi/kApiDef.h>
#include <kVision/Common/kVision.h>
#include <kVision/S3d/kS3dVolumeCheckDef.h>
#include <kVision/S3d/kS3dVolumeCheckUtilities.h>

#define kS3D_VOLUMCHECK_INPUTBLOCK_NAME                        "VolumeCheckInput"
#define kS3D_VOLUMCHECK_INPUTBLOCK_TYPE_NAME                   "VolumeCheckInput"
#define kS3D_VOLUMCHECK_INPUTBLOCK_IN_IMAGE_NAME               "InImage"
#define kS3D_VOLUMCHECK_INPUTBLOCK_OUTPUT_SURFACE_NAME         "OutSurface"
#define kS3D_VOLUMCHECK_INPUTBLOCK_OUTPUT_ADDITIONAL_NAME      "OutAdditional"

typedef enum kS3dVolumeCheckInputBlockPort
{
    kS3D_VOLUMCHECK_INPUTBLOCK_IN_IMAGE=0,
    kS3D_VOLUMCHECK_INPUTBLOCK_OUTPUT_SURFACE = 1,
    kS3D_VOLUMCHECK_INPUTBLOCK_OUTPUT_ADDITIONAL = 2
} kS3dVolumeCheckInputBlockPort;

typedef struct kS3dVolumeCheckInputBlockOutput
{
    k64f chamberType;
    k64f decision;
    kPoint3d64f scale;
    kPoint3d64f position;
    kPoint3d64f orientation;
} kS3dVolumeCheckInputBlockOutput;

/**
* @class       kS3dVolumeCheckInputBlockInfo
* @ingroup     kVision-S3d
* @brief       Settings class for kS3dVolumeCheckInputBlock.
*/
typedef kObject kS3dVolumeCheckInputBlockInfo;


/**
* Constructs a kS3dVolumeCheckInputBlockInfo object
*
* @public              @memberof kS3dVolumeCheckInputBlockInfo
* @param   info        Destination for the constructed object handle.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckInputBlockInfo_Construct(kS3dVolumeCheckInputBlockInfo* info, kAlloc allocator);

/**
* Constructs a kS3dVolumeCheckInputBlockInfo object by reading the content from a xml object
*
* @public              @memberof kS3dVolumeCheckInputBlockInfo
* @param   info        Destination for the constructed object handle.
* @param   xml         kXml object. 
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckInputBlockInfo_Load(kS3dVolumeCheckInputBlockInfo* info, kXml xml, kAlloc allocator);

/**
* Constructs a kXml object and uses it to store the content of this kS3dVolumeCheckInputBlockInfo instance.
*
* @public              @memberof kS3dVolumeCheckInputBlockInfo
* @param   info        kS3dVolumeCheckInputBlockInfo object.
* @param   xml         Receives constructed kXml object.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckInputBlockInfo_Store(kS3dVolumeCheckInputBlockInfo info, kXml* xml);

/**
* Reads kS3dVolumeCheckInputBlockInfo content from a kXml object.
*
* @public              @memberof kStartActionInfo
* @param   info        Info object.
* @param   xml         XML object.
* @param   item        XML item.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckInputBlockInfo_Read(kS3dVolumeCheckInputBlockInfo info, kXml xml, kXmlItem item);

/**
* Writes kS3dVolumeCheckInputBlockInfo content to a kXml object.
*
* @public              @memberof kStartActionInfo
* @param   info        Info object.
* @param   xml         XML object.
* @param   item        XML item.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckInputBlockInfo_Write(kS3dVolumeCheckInputBlockInfo info, kXml xml, kXmlItem item);


/**
* Get type of combustion chamber, defined in kS3dVolumeCheckDefaultChamberType.  
*
* @public              @memberof kS3dVolumeCheckInputBlockInfo
* @param   info        kS3dVolumeCheckInputBlockInfo object.
* @return              Type of the combustion chamber.
*/
kVsFx(k32u) kS3dVolumeCheckInputBlockInfo_ChamberType(kS3dVolumeCheckInputBlockInfo info);

/**
* Set type of combustion chamber, defined in kS3dVolumeCheckDefaultChamberType.  
*
* @public               @memberof kS3dVolumeCheckInputBlockInfo
* @param   info         kS3dVolumeCheckInputBlockInfo object.
* @param   chamberType  Type of the combustion chamber.
* @return               Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckInputBlockInfo_SetChamberType(kS3dVolumeCheckInputBlockInfo info, k32u chamberType);

/**
* Get basic file name of the simulated input file.  
*
* @public              @memberof kS3dVolumeCheckInputBlockInfo
* @param   info        kS3dVolumeCheckInputBlockInfo object.
* @return              Basic file name.
*/
kVsFx(kChar*) kS3dVolumeCheckInputBlockInfo_FileName(kS3dVolumeCheckInputBlockInfo info);

/**
* Set basic file name of the simulated input file.  
*
* @public              @memberof kS3dVolumeCheckInputBlockInfo
* @param   info        kS3dVolumeCheckInputBlockInfo object.
* @param   fileName    Basic input file name.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckInputBlockInfo_SetFileName(kS3dVolumeCheckInputBlockInfo info, kChar* fileName);

/**
* Get format of the simulated input data defined in kS3dVolumeCheckInputFormatType.  
*
* @public              @memberof kS3dVolumeCheckInputBlockInfo
* @param   info        kS3dVolumeCheckInputBlockInfo object.
* @return              Basic input file name.
*/
kVsFx(k32s) kS3dVolumeCheckInputBlockInfo_DataFormat(kS3dVolumeCheckInputBlockInfo info);

/**
* Set format of the simulated input data defined in kS3dVolumeCheckInputFormatType.  
*
* @public              @memberof kS3dVolumeCheckInputBlockInfo
* @param   info        kS3dVolumeCheckInputBlockInfo object.
* @param   format      Format of the simulated input data
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckInputBlockInfo_SetDataFormat(kS3dVolumeCheckInputBlockInfo info, k32s format);

/**
* Get available file count of the simulattion for a given data format.
*
* @public              @memberof kS3dVolumeCheckInputBlockInfo
* @param   info        kS3dVolumeCheckInputBlockInfo object.
* @return              Available file count.
*/
kVsFx(k32s) kS3dVolumeCheckInputBlockInfo_FileCount(kS3dVolumeCheckInputBlockInfo info);

/**
* Set available file count of the simulattion for a given data format.  
*
* @public              @memberof kS3dVolumeCheckInputBlockInfo
* @param   info        kS3dVolumeCheckInputBlockInfo object.
* @param   count       Available file count.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckInputBlockInfo_SetFileCount(kS3dVolumeCheckInputBlockInfo info, k32s count);

/**
* Get start index for several measurements. e.g. index=0: corresponds to input0.omc.
*
* @public              @memberof kS3dVolumeCheckInputBlockInfo
* @param   info        kS3dVolumeCheckInputBlockInfo object.
* @return              Start index for several measurements.
*/
kVsFx(k32s) kS3dVolumeCheckInputBlockInfo_StartIndex(kS3dVolumeCheckInputBlockInfo info);

/**
* Set start index for several measurements. e.g. index=0: corresponds to input0.omc.
*
* @public              @memberof kS3dVolumeCheckInputBlockInfo
* @param   info        kS3dVolumeCheckInputBlockInfo object.
* @param   startIndex  Start index for several measurements
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckInputBlockInfo_SetStartIndex(kS3dVolumeCheckInputBlockInfo info, k32s startIndex);

/**
* Get measurement count for several measurements. If the count is set more than available file count, the file is loaded into the loop.
*
* @public              @memberof kS3dVolumeCheckInputBlockInfo
* @param   info        kS3dVolumeCheckInputBlockInfo object.
* @return              Measurement count.
*/
kVsFx(k32s) kS3dVolumeCheckInputBlockInfo_MeasurementCount(kS3dVolumeCheckInputBlockInfo info);

/**
* Set measurement count for several measurements. If the count is set more than available file count, the file is loaded into the loop.
*
* @public              @memberof kS3dVolumeCheckInputBlockInfo
* @param   info        kS3dVolumeCheckInputBlockInfo object.
* @param   count       Measurement count. 
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckInputBlockInfo_SetMeasurementCount(kS3dVolumeCheckInputBlockInfo info, k32s count);

/**
* Get start position of the simulated transformation of the surface.
*
* @public              @memberof kS3dVolumeCheckInputBlockInfo
* @param   info        kS3dVolumeCheckInputBlockInfo object.
* @return              Start position.
*/
kVsFx(kPoint3d64f) kS3dVolumeCheckInputBlockInfo_StartPosition(kS3dVolumeCheckInputBlockInfo info);

/**
* Set start position of the simulated transformation of the surface.
*
* @public              @memberof kS3dVolumeCheckInputBlockInfo
* @param   info        kS3dVolumeCheckInputBlockInfo object.
* @param   position    Start position. 
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckInputBlockInfo_SetStartPosition(kS3dVolumeCheckInputBlockInfo info, kPoint3d64f position);

/**
* Get start orientation of the simulated transformation of the surface.
*
* @public              @memberof kS3dVolumeCheckInputBlockInfo
* @param   info        kS3dVolumeCheckInputBlockInfo object.
* @return              Start orientation.
*/
kVsFx(kPoint3d64f) kS3dVolumeCheckInputBlockInfo_StartOrientation(kS3dVolumeCheckInputBlockInfo info);

/**
* Set start orientation of the simulated transformation of the surface.
*
* @public              @memberof kS3dVolumeCheckInputBlockInfo
* @param   info        kS3dVolumeCheckInputBlockInfo object.
* @param   orientation Start orientation.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckInputBlockInfo_SetStartOrientation(kS3dVolumeCheckInputBlockInfo info, kPoint3d64f orientation);

/**
* Queries whether the transformation of the surface with the random generator is performed.
*
* @public              @memberof kS3dVolumeCheckInputBlockInfo
* @param   info        kS3dVolumeCheckInputBlockInfo object.
* @return              Random transformation will be performed?
*/
kVsFx(kBool) kS3dVolumeCheckInputBlockInfo_RandomPositioning(kS3dVolumeCheckInputBlockInfo info);

/**
* Set a boolean value to determine whether the transformation of the surface with the random generator should be performed.
*
* @public              @memberof kS3dVolumeCheckInputBlockInfo
* @param   info        kS3dVolumeCheckInputBlockInfo object.
* @param   randomUsed  kTRUE = will be performed
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckInputBlockInfo_SetRandomPositioning(kS3dVolumeCheckInputBlockInfo info, kBool randomUsed);

/**
* Get displacement amplitudes of the random generator.
*
* @public              @memberof kS3dVolumeCheckInputBlockInfo
* @param   info        kS3dVolumeCheckInputBlockInfo object.
* @return              Displacement amplitudes in x, y and directions. 
*/
kVsFx(kPoint3d64f) kS3dVolumeCheckInputBlockInfo_RandomPosition(kS3dVolumeCheckInputBlockInfo info);

/**
* Set displacement amplitudes of the random generator.
*
* @public                  @memberof kS3dVolumeCheckInputBlockInfo
* @param   info            kS3dVolumeCheckInputBlockInfo object.
* @param   randomPosition  Displacement amplitudes in x, y and directions. 
* @return                  Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckInputBlockInfo_SetRandomPosition(kS3dVolumeCheckInputBlockInfo info, kPoint3d64f randomPosition);

/**
* Get rotation amplitudes of the random generator.
*
* @public              @memberof kS3dVolumeCheckInputBlockInfo
* @param   info        kS3dVolumeCheckInputBlockInfo object.
* @return              Rotation amplitudes of the 3 angles.
*/
kVsFx(kPoint3d64f) kS3dVolumeCheckInputBlockInfo_RandomOrientation(kS3dVolumeCheckInputBlockInfo info);

/**
* Set rotation amplitudes of the random generator.
*
* @public                     @memberof kS3dVolumeCheckInputBlockInfo
* @param   info               kS3dVolumeCheckInputBlockInfo object.
* @param   randomOrientation  Rotation amplitudes of the 3 angles.
* @return                     Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckInputBlockInfo_SetRandomOrientation(kS3dVolumeCheckInputBlockInfo info, kPoint3d64f randomOrientation);

/**
* Queries whether the loaded omc file should be stored as kdat file used for further simulation.
*
* @public              @memberof kS3dVolumeCheckInputBlockInfo
* @param   info        kS3dVolumeCheckInputBlockInfo object.
* @return              kTrue = kdat file will be stored.
*/
kVsFx(kBool) kS3dVolumeCheckInputBlockInfo_StoreInputData(kS3dVolumeCheckInputBlockInfo info);

/**
* Set a boolean value to determine whether loaded omc file should be stored as kdat file used for further simulation.
*
* @public                 @memberof kS3dVolumeCheckInputBlockInfo
* @param   info           kS3dVolumeCheckInputBlockInfo object.
* @param   saveSimulation kTrue = kdat file will be stored.
* @return                 Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckInputBlockInfo_SetStoreInputData(kS3dVolumeCheckInputBlockInfo info, kBool saveSimulation);

/**
* Get scaling of the specified input height map or point clouds of kArray2.
*
* @public              @memberof kS3dVolumeCheckInputBlockInfo
* @param   info        kS3dVolumeCheckInputBlockInfo object.
* @return              Scaling the pouint cloud or height map.
*/
kVsFx(kPoint3d64f) kS3dVolumeCheckInputBlockInfo_InputScale(kS3dVolumeCheckInputBlockInfo info);

/**
* Set scaling of the specified input height map or point clouds of kArray2.
*
* @public              @memberof kS3dVolumeCheckInputBlockInfo
* @param   info        kS3dVolumeCheckInputBlockInfo object.
* @param   scale       Scaling of the pouint cloud or height map.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckInputBlockInfo_SetInputScale(kS3dVolumeCheckInputBlockInfo info, kPoint3d64f inputScale);

#include <kVision/S3d/kS3dVolumeCheckInputBlockInfo.x.h>

#endif /* #ifndef KVISION_KS3D_VOLUMECHECKINPUTBLOCKINFO_H  */
