/**
* @file    kS3dSpeckleProcessor.x.h
*
* @internal
* Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
*/
#ifndef KS3D_SPECKLE_PROCESSOR_X_H
#define KS3D_SPECKLE_PROCESSOR_X_H

#include <kVision/S3d/kS3dRectifier.h>
#include <kVision/S3d/kS3dSpeckleMatchAlg.h>
#include <kVision/S3d/kS3dSgbm.h>
#include <kVision/S3d/kS3dBm.h>
#include <kVision/S3d/kS3dSpecklePrefilter.h>
#include <kVision/S3d/kS3dSpeckleDisparityFilter.h>
#include <kVision/Vs/kVsJobQueue.h>

#define kS3D_SPECKLE_PROCESSOR_MAX_PARALLEL_COUNT                           (5)
#define kS3D_SPECKLE_PROCESSOR_DISPARITY_BIT_DEPTH                          (6)    // This limits the maximum disparity range to 1024 rectified px. 
                                                                                   // Can implement a dynamic system if this limitation is ever exercised 
#define kS3D_SPECKLE_PROCESSOR_DEFAULT_PARALLEL_COUNT                       (1)
#define kS3D_SPECKLE_PROCESSOR_DEFAULT_RECTIFICATION_RESOLUTION             (0.5) 

#define kS3D_SPECKLE_PROCESSOR_DEFAULT_SCALE_MULTIPLIER                     (4)                                
#define kS3D_SPECKLE_PROCESSOR_DEFAULT_MATCH_TYPE                           (kS3D_SPECKLE_PROCESSOR_MATCH_TYPE_SGBM)
#define kS3D_SPECKLE_PROCESSOR_DEFAULT_MATCH_WINDOW                         (5)
#define kS3D_SPECKLE_PROCESSOR_DEFAULT_DISPARITY_RANGE                      (128)
#define kS3D_SPECKLE_PROCESSOR_DEFAULT_DISPARITY_CENTRE                     (0)
#define kS3D_SPECKLE_PROCESSOR_DEFAULT_LR_THRESHOLD                         (2)
#define kS3D_SPECKLE_PROCESSOR_DEFAULT_TEXTURE_THRESHOLD                    (5)
#define kS3D_SPECKLE_PROCESSOR_DEFAULT_UNIQUENESS_RATIO                     (0.1)
#define kS3D_SPECKLE_PROCESSOR_DEFAULT_CONTINUITY_WEIGHT                    (1.0)
#define kS3D_SPECKLE_PROCESSOR_DEFAULT_OCCLUSION_WEIGHT                     (4.0)
#define kS3D_SPECKLE_PROCESSOR_DEFAULT_FILTER_CAP                           (127)

#define kS3D_SPECKLE_PROCESSOR_DEFAULT_FILTER_WINDOW                        (15)
#define kS3D_SPECKLE_PROCESSOR_DEFAULT_DISPARITY_REGION_MIN_SIZE            (100)
#define kS3D_SPECKLE_PROCESSOR_DEFAULT_DISPARITY_REGION_STEP_THRESHOLD      (5)
#define kS3D_SPECKLE_PROCESSOR_DEFAULT_DISPARITY_TEMPLATE_SMOOTH_WINDOW     (15)
#define kS3D_SPECKLE_PROCESSOR_DEFAULT_DISPARITY_TEMPLATE_SMOOTH_FILL       (0.1)

#define kS3D_SPECKLE_PROCESSOR_MAX_PARALLEL_COUNT                           (5)

typedef struct kS3dSpeckleProcessorLevelParams
{
    // Parameters
    k32u scale;
    kS3dSpeckleProcessorMatchAlgType matchType;
    k32u matchWindow;
    k32s disparityCentre; // Non-zero only for the coarsest step. Calculated from the offset between the two images provided by the rectifier
    k32u disparityRange;
    k64f uniquenessRatio;
    k32u disparityLrThreshold;
    k64f continuityWeight;
    k64f occlusionWeight;
    k32u filterCap;
    k32u filterWindow;
    k32u textureThreshold;
    k32u disparityRegionMinSize;
    k64f disparityRegionStepThreshold;
    k32u disparityTemplateSmoothWindow;

    // Data
    kS3dSpecklePrefilter prefilters[2]; // one for each camera to allow for parallel processing
    kS3dSpeckleMatchAlg matchAlgs[kS3D_SPECKLE_PROCESSOR_MAX_PARALLEL_COUNT];
    kS3dSpeckleDisparityFilter disparityFilter;

    kImage rectImages[3];       // rectified images sampled to the current level. Third image receives the remapped version of the second image
    kImage normImages[2];       // rectified images after normalization

    kImage normImageParts0[kS3D_SPECKLE_PROCESSOR_MAX_PARALLEL_COUNT];
    kImage normImageParts1[kS3D_SPECKLE_PROCESSOR_MAX_PARALLEL_COUNT];
    kArray2 disparityParts[kS3D_SPECKLE_PROCESSOR_MAX_PARALLEL_COUNT];
    
    kArray2 disparity;          // local disparity map after matching
    kArray2 adjustedDisparity;  // disparity map after adding the template from the previous step
    kArray2 nextTemplate;

    k32u scaleToNext;
    kSize imageWidth;
    kSize imageHeight;

    kSize partHeight;
    kSize partHeightMargin;

} kS3dSpeckleProcessorLevelData;

typedef struct kS3dSpeckleProcessorClass
{
    kObjectClass base;

    kS3dStereoCal cal;
    kS3dStereoProfiler profiler;
    kVsJobQueue jobQueue;

    // Shared parameters
    kBool sharedConfigured;
    kBool levelsConfigured;

    kSize levelCount;
    k64f projectionResolution;
    k64f rectificationResolution;
    k32u parallelCount;
    k32u bitDepth;
    k32u disparitySubpixScale;

    // Shared data
    kS3dRectifier rectifier;

    // Level data and parameters
    kArray1 levelData;
    kImage rawBuffer[2];

} kS3dSpeckleProcessorClass;

kDeclareClassEx(kVs, kS3dSpeckleProcessor, kObject)

//semi-private exported functions (virtual override methods)

kVsFx(kStatus) kS3dSpeckleProcessor_VInitClone(kS3dSpeckleProcessor processor, kS3dSpeckleProcessor source, kAlloc allocator);
kVsFx(kStatus) kS3dSpeckleProcessor_VRelease(kS3dSpeckleProcessor processor);

//non-exported (private) methods
kStatus kS3dSpeckleProcessor_RunSingle(kS3dSpeckleProcessor processor, kImage image0, kImage image1, kArray2 output);
kStatus kS3dSpeckleProcessor_RunParallel(kS3dSpeckleProcessor processor, kImage image0, kImage image1, kArray2 output);

kStatus kS3dSpeckleProcessor_Init(kS3dSpeckleProcessor processor, kS3dStereoCal cal, kS3dStereoProfiler profiler, k32u levelCount, kAlloc allocator);
kStatus kS3dSpeckleProcessor_ClearLevel(kS3dSpeckleProcessor processor, k32u level);
kStatus kS3dSpeckleProcessor_SetupLevel(kS3dSpeckleProcessor processor, k32u level);

kStatus kS3dSpeckleProcessor_DownsampleImage(kS3dSpeckleProcessor processor, kImage src, k32u scale, kImage dst);
kStatus kS3dSpeckleProcessor_ComputeTemplate(kS3dSpeckleProcessor processor, kS3dSpeckleDisparityFilter dispFilter, k32u scale, kArray2 disparity, kArray2 output);
kStatus kS3dSpeckleProcessor_ShiftImage(kS3dSpeckleProcessor processor, kImage image, kArray2 shiftMap, k32u scale, k8u nullValue, kImage output);
kStatus kS3dSpeckleProcessor_ShiftDisparity(kS3dSpeckleProcessor processor, kArray2 disparity, k32u scale, kArray2 nextTemplate, kArray2 output);
kStatus kS3dSpeckleProcessor_EvaluateDisparity(kS3dSpeckleProcessor processor, kArray2 disparity, kArray2 output);
kStatus kCall kS3dSpeckleProcessor_RemapFx(kPointer context, k32s viewIndex, k32s x, k32s y, k32s* xr);

//parallel processing helper functions

k64u kS3dSpeckleProcessor_MakeJobId(kSize levelIndex, kSize viewIndex, kSize sliceIndex);
kStatus kS3dSpeckleProcessor_ParseJobId(k64u id, kSize* levelIndex, kSize* viewIndex, kSize* sliceIndex);

kStatus kCall kS3dSpeckleProcessor_RectifyParallel(kS3dSpeckleProcessor processor, k64u id);
kStatus kCall kS3dSpeckleProcessor_DownsampleParallel(kS3dSpeckleProcessor processor, k64u id);
kStatus kCall kS3dSpeckleProcessor_AdaptiveNormParallel(kS3dSpeckleProcessor processor, k64u id);
kStatus kCall kS3dSpeckleProcessor_MatchParallel(kS3dSpeckleProcessor processor, k64u id);
kStatus kS3dSpeckleProcessor_CombineMatchParts(kS3dSpeckleProcessor processor, k32u level);
kStatus kS3dSpeckleProcessor_SplitMatchParts(kS3dSpeckleProcessor processor, k32u level);


//cast macro

#endif  /* #ifndef KS3D_SPECKLE_PROCESSOR_X_H */
