/**
* @file    kS3dSpecklePrefilter.h
* @brief   Declares the kS3dSpecklePrefilter class.
*
* @internal
* Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
*/
#ifndef KS3D_SPECKLE_PRE_FILTER_H
#define KS3D_SPECKLE_PRE_FILTER_H

#include <kVision/Common/kVision.h>
#include <kVision/S3d/kS3dCommon.h>

/**
* @class       kS3dSpecklePrefilter
* @extends     kObject
* @ingroup     kVision-S3d
* @brief       Provides a set of utility functions to perform pre-filtering on rectified images intended for stereo block matching.
*              User should apply both histogram re-scaling and adaptive normalization to the images before passing the result into 
*              kS3dSgbm, kS3dBm or any other class deriving from kS3dSpeckleMatchAlg
*/
typedef kObject kS3dSpecklePrefilter;

/**
* Constructs a kS3dSpecklePrefilter object
*
* @public              @memberof kS3dSpecklePrefilter
* @param   filter      Destination for the constructed object handle.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kS3dSpecklePrefilter_Construct(kS3dSpecklePrefilter* filter, kAlloc alloc);

/**
* Updates working buffers of the implementation to reflect current set of algorithm parameters.
* Calling this function is optional, as this validation step is also performed during each execution
* of the algorithm (kS3dSpecklePrefilter_AdaptiveNorm, kS3dSpecklePrefilter_StretchHistogram).
* However, the initialization time may be non-negligible, which would affect the first execution of the algorithm.
* To eliminate the added delay from the first algorithm execution, the user should call kS3dSpecklePrefilter_Setup
* after all of the parameters have been configured.
*
* @public              @memberof kS3dSpecklePrefilter
* @param   filter      Filter object
* @return              Operation status.
*/
kVsFx(kStatus) kS3dSpecklePrefilter_Setup(kS3dSpecklePrefilter filter);

/**
* Measures the intensity histogram of two stereo images and rescales both to eliminate gaps in intensity. 
*
* @public              @memberof kS3dSpecklePrefilter
* @param   filter      Filter object
* @param   image0      First input image
* @param   image1      Second input image
* @param   output0     Output for the first image (can be the same as image0)
* @param   output1     Output for the second image (can be the same as image1)
* @param   output      Output image (can be the same as the input)
* @return              Operation status.
*/
kVsFx(kStatus) kS3dSpecklePrefilter_StretchHistogram(kS3dSpecklePrefilter filter, kImage image0, kImage image1, kImage output0, kImage output1);

/**
* Applies adaptive normalization to an image. Local intensity values are scaled to the full greyscale range (0..255) and
* clamped to +/- filter cap (kS3dSpecklePrefilter_SetFilterCap). The result is then output as an unsigned 8-bit value 
* offset by filter cap. If intensity within the normalization window do not produce a range that exceeds the texture 
* threshold (kS3dSpecklePrefilter_SetTextureThreshold), they are 'zeroed' by being set to the filter cap value
*
* The algorithm does not operate in-place. If input and output images point to the same object, the class will use an
* internal buffer for execution and copy the result into the output image at the end. 
*
* @public              @memberof kS3dSpecklePrefilter
* @param   filter      Filter object
* @param   input       Input image
* @param   output      Output image (can be the same as the input)
* @return              Operation status.
*/
kVsFx(kStatus) kS3dSpecklePrefilter_AdaptiveNorm(kS3dSpecklePrefilter filter, kImage input, kImage output);

/**
* Sets the size of the expected input images.
*
* @public              @memberof kS3dSpecklePrefilter
* @param   filter      Filter object
* @param   imageWidth  Width of the input images
* @param   imageHeight Height of the input images
* @return              Operation status.
*/
kVsFx(kStatus) kS3dSpecklePrefilter_SetImageSize(kS3dSpecklePrefilter filter, kSize imageWidth, kSize imageHeight);

/**
* Gets the size of input images
*
* @public              @memberof kS3dSpecklePrefilter
* @param   filter      Filter object
* @param   imageWidth  Width of the input images
* @param   imageHeight Height of the input images
* @return              Operation status.
*/
kVsFx(kStatus) kS3dSpecklePrefilter_ImageSize(kS3dSpecklePrefilter filter, kSize* imageWidth, kSize* imageHeight);

/**
* Set the size of the filtering window. For adaptive normalization the window is used to determine the average local intensity
* and contrast. During algorithm execution, the average intensity is subtracted from the pixel intensity values and the range
* is re-scaled to full 8-bit range. The value must be greater or equal to 1 and odd
*
* @public              @memberof kS3dSpecklePrefilter
* @param   filter      Filter object
* @param   window      Filter window size (px)
* @return              Operation status.
*/
kVsFx(kStatus) kS3dSpecklePrefilter_SetWindow(kS3dSpecklePrefilter filter, k32u window);

/**
*  Get matching window size
*
* @public              @memberof kS3dSpecklePrefilter
* @param   filter      Filter object
* @return              Block matching window size (px)
*/
kVsFx(k32u) kS3dSpecklePrefilter_Window(kS3dSpecklePrefilter filter);

/**
* Set the intensity truncation value for the image filter output (from 1 to 127). Variation in image intensity between
* the two camera images will generally be proportional to the absolute magnitude. Truncating the filter output can help
* improve the robustness of the cost calculation by clipping higher intensities to a fixed value (which would be the
* same for both images). For images where the intensity variation between the two cameras is expected to be low, the
* truncation should be set high or disabled (using maximum of 127 counts). The field is set to maximum by default.
*
* @public               @memberof kS3dSpecklePrefilter
* @param   filter       Filter object
* @param   filterCap    Image filter truncation value
* @return               Operation status.
*/
kVsFx(kStatus) kS3dSpecklePrefilter_SetFilterCap(kS3dSpecklePrefilter filter, k32u filterCap);
/**
*  Get the image filter truncation value
*
* @public              @memberof kS3dSpecklePrefilter
* @param   filter      Filter object
* @return              Image filter truncation value
*/
kVsFx(k32u) kS3dSpecklePrefilter_FilterCap(kS3dSpecklePrefilter filter);

/**
* Set minimum amount of contrast within the filtering window to consider a patch valid. Intensity range lower than that results in the pixel centered
* at that window to be invalidated (set to filter cap)
*
* @public               @memberof kS3dSpecklePrefilter
* @param   filter       Filter object
* @param   threshold    Texture threshold
* @return               Operation status.
*/
kVsFx(kStatus) kS3dSpecklePrefilter_SetTextureThreshold(kS3dSpecklePrefilter filter, k32u threshold);

/**
*  Get the texture threshold
*
* @public              @memberof kS3dSpecklePrefilter
* @param   filter      Filter object
* @return              Texture threshold value
*/
kVsFx(k32u) kS3dSpecklePrefilter_TextureThreshold(kS3dSpecklePrefilter filter);

#include <kVision/S3d/kS3dSpecklePrefilter.x.h>

#endif  /* KS3D_SPECKLE_PRE_FILTER_H */
