/**
* @file    kS3dPhaseSampler.h
* @brief   Declares the kS3dPhaseSampler class.
*
* @internal
* Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
*/
#ifndef kS3D_PHASE_SAMPLER_H
#define kS3D_PHASE_SAMPLER_H

#include <kApi/Data/kArray1.h>
#include <kApi/Data/kArray2.h>
#include <kApi/Data/kArrayList.h>
#include <kVision/Common/kVision.h>
#include <kVision/S3d/kS3dCommon.h>
#include <kVision/S3d/kS3dStereoProfiler.h>
#include <kVision/S3d/kS3dPhaseCombiner.h>
#include <kVision/Vs/kVsJobQueue.h>

#include <kpp/Common/kppPhasePixel2.h>

/**
* @class       kS3dPhaseSampler
* @extends     kObject
* @ingroup     kVision-S3d
* @brief       Rectifies and resamples camera phase maps to a uniform step in projector space and uniform step in rectified Y space. For a 
*              a pair of phase maps acquired from a stereo camera pair, this sampling ensures that corresponding locations in the output buffers 
*              form matching stereo points
*/
typedef kObject kS3dPhaseSampler;

/**
* Constructs a kS3dPhaseSampler object
*
* @public              @memberof kS3dPhaseSampler
* @param   sampler     Destination for the constructed object handle.
* @param   profiler    Profiler object to be used for rectification
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseSampler_Construct(kS3dPhaseSampler* sampler, kS3dStereoProfiler profiler, kAlloc alloc);

/**
* Updates working buffers of the implementation to reflect current set of algorithm parameters.
* Calling this function is optional, as this validation step is also performed during each execution
* of the algorithm (kS3dPhaseSampler_Begin, kS3dPhaseSampler_Update).
* However, the initialization time may be non-negligible, which would affect the first execution of the algorithm.
* To eliminate the added delay from the first algorithm execution, the user should call kS3dPhaseSampler_Setup
* after all of the parameters have been configured.
*
* @public              @memberof kS3dPhaseSampler
* @param   sampler     Sampler object
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseSampler_Setup(kS3dPhaseSampler sampler);

/**
*  Rectify and resample the results of stripe and phase decoding to uniform stereo correspondence spacing. The output is 
*  a 2D array of rectified X coordinates spaced by uniform phase step (along rows) and uniform Y rectified step (along columns). The expected 
*  size of the output buffer is determined by inputs to kS3dPhaseSampler_SetPhaseRoi and kS3dPhaseSampler_SetYProjectionRoi functions. Optionally
*  an output buffer for matching intensity data can also be provided. 
* 
*
* @public                   @memberof kS3dPhaseSampler
* @param   sampler          Sampler object
* @param   viewIndex        Index of the view from which the stripe/phase sequence was acquired
* @param   stripePhaseMap   Result of stripe and phase decoding for the scan (kArray2<kPhasePixel2>)
* @param   outputXProj      Rectified X coordinates sampled to uniform phase and rectified Y positions (kArray2<k16s> of height specified by 
*                           kS3dPhaseSampler_SetPhaseRoi and width specified by kS3dPhaseSampler_SetYProjectionRoi)
* @param   outputIntensity  Optional buffer for output of intensity data corresponding to the rectified X coordinate output (kArray2<k8u> 
*                           of the same size as outputXProj)
* @param   outputTexture    Optional output buffer for texture coordinates (kArray2<kPoint32s> of height specified by
*                           kS3dPhaseSampler_SetPhaseRoi and width specified by kS3dPhaseSampler_SetYProjectionRoi). Texture pixel coordinates
*                           are returned with 8-bit fixed point precision
* @return                   Operation status.
*/
kVsFx(kStatus) kS3dPhaseSampler_Run(kS3dPhaseSampler sampler, kSize viewIndex, kArray2 stripePhaseMap, kArray2 outputXProj, kArray2 outputIntensity, kArray2 outputTexture);

//////////////////////////////////////////////////////////////////////////
// CUDA RUN
//////////////////////////////////////////////////////////////////////////

/**
*  Same as kS3dPhaseSampler_Run()
*  Executes the code on the device (GPU)
*  Assumes input output array are preallocated on device (GPU)
*
* @public                   @memberof kS3dPhaseSampler
* @param   sampler          Sampler object
* @param   viewIndex        Index of the view from which the stripe/phase sequence was acquired
* @param   stripePhaseMap   Result of stripe and phase decoding for the scan (kArray2<kPhasePixel2>)
* @param   outputXProj      Rectified X coordinates sampled to uniform phase and rectified Y positions (kArray2<k16s> of height specified by
*                           kS3dPhaseSampler_SetPhaseRoi and width specified by kS3dPhaseSampler_SetYProjectionRoi)
* @param   outputIntensity  Optional buffer for output of intensity data corresponding to the rectified X coordinate output (kArray2<k8u>
*                           of the same size as outputXProj)
* @return                   Operation status.
*/

#if defined (K_HAVE_CUDA)
kVsFx(kStatus) kS3dPhaseSampler_RunCuda(kS3dPhaseSampler sampler, kSize viewIndex, kArray2 inStripePhaseMap, kArray2 outXProj, kArray2 outIntensity);
#else
kInlineFx(kStatus) kS3dPhaseSampler_RunCuda(kS3dPhaseSampler sampler, kSize viewIndex, kArray2 inStripePhaseMap, kArray2 outXProj, kArray2 outIntensity) { return kERROR; }
#endif

/**
* Sets cuda stream for cuda parallel executions.
*
* @public              @memberof kS3dPhaseSampler
* @param   sampler     Sampler object
* @param   cudaStream  kCudaStream class
* @return              Operation status.
*/

kVsFx(kStatus) kS3dPhaseSampler_SetCudaStream(kS3dPhaseSampler sampler, kCudaStream cudaStream);

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

/**
* Sets the size of the expected input images.
*
* @public              @memberof kS3dPhaseSampler
* @param   sampler      Sampler object
* @param   imageWidth  Width of the input images
* @param   imageHeight Height of the input images
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseSampler_SetImageSize(kS3dPhaseSampler sampler, kSize imageWidth, kSize imageHeight);

/**
* Gets the size of input images
*
* @public              @memberof kS3dPhaseSampler
* @param   sampler     Sampler object
* @param   imageWidth  Width of the input images
* @param   imageHeight Height of the input images
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseSampler_ImageSize(kS3dPhaseSampler sampler, kSize* imageWidth, kSize* imageHeight);

/**
* Sets the scale of the phase period. The phase values expected by the kS3dPhaseSampler_Run function must be scaled such that 
* 0 to 2Pi phase offset is mapped to 0 to period
*
* @public              @memberof kS3dPhaseSampler
* @param   sampler     Sampler object
* @param   period      Phase period scale
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseSampler_SetPhasePeriod(kS3dPhaseSampler sampler, kSize period);

/**
* Gets the scale of the phase period.
*
* @public              @memberof kS3dPhaseSampler
* @param   sampler     Sampler object
* @return              Phase period scale
*/
kVsFx(kSize) kS3dPhaseSampler_PhasePeriod(kS3dPhaseSampler sampler);

/**
* Sets the parameters determining resampling region of interest along projection phase modulation axis (rows in both camera and projection images)
*
* @public              @memberof kS3dPhaseSampler
* @param   sampler     Sampler object
* @param   begin       Starting cumulative phase value (starting stripe * phasePeriod + periodic starting phase)   
* @param   stepShift   Shift value representing the phase increment between columns in the output buffers. Exact increment is calculated as
*                      (1 << stepShift). 
* @param   count       Number of points along the projection modulation axis (output buffer height)
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseSampler_SetPhaseRoi(kS3dPhaseSampler sampler, kSize begin, kSize stepShift, kSize count);

/**
* Gets the parameters determining resampling region of interest along projection phase modulation axis (rows in both camera and projection images)
*
* @public              @memberof kS3dPhaseSampler
* @param   sampler     Sampler object
* @param   begin       Output pointer to the starting cumulative phase value (starting stripe * phasePeriod + periodic starting phase)
* @param   stepShift   Output pointer to the shift value representing the phase increment between columns in the output buffers. Exact increment is calculated as
*                      (1 << stepShift).
* @param   count       Output pointer to the number of points along the projection modulation axis (output buffer height)
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseSampler_PhaseRoi(kS3dPhaseSampler sampler, kSize* begin, kSize* stepShift, kSize* count);

/**
* Sets the parameters determining resampling region of interest along the rectified Y axis (columns in both camera and projection images)
*
* @public              @memberof kS3dPhaseSampler
* @param   sampler     Sampler object
* @param   begin       Starting rectified Y value expressed in calibration object projection resolution units 
*                      (begin * kS3dStereoCal_Parameters(cal)->projectionResolution converts to mm value)
* @param   step        Step size along the rectified Y axis (step * kS3dStereoCal_Parameters(cal)->projectionResolution converts to mm value)
*                      (1 << stepShift).
* @param   count       Number of points along the rectified Y axis (output buffer width)
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseSampler_SetYProjectionRoi(kS3dPhaseSampler sampler, kSSize begin, kSize step, kSize count);

/**
* Gets the parameters determining resampling region of interest along the rectified Y axis (columns in both camera and projection images)
*
* @public              @memberof kS3dPhaseSampler
* @param   sampler     Sampler object
* @param   begin       Output pointer to the starting rectified Y value expressed in calibration object projection resolution units
*                      (begin * kS3dStereoCal_Parameters(cal)->projectionResolution converts to mm value)
* @param   step        Output pointer to the step size along the rectified Y axis 
                       (step * kS3dStereoCal_Parameters(cal)->projectionResolution converts to mm value)
* @param   count       Output pointer to the number of points along the rectified Y axis (output buffer width)
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseSampler_YProjectionRoi(kS3dPhaseSampler sampler, kSSize* begin, kSize* step, kSize* count);

/**
* Output the result of the intermediate phase sampling step. The resulting plot includes raw image X coordinates and
* intensity data sampled along phase interval specified by kS3dPhaseSampler_SetPhaseRoi function. 
*
* @public              @memberof kS3dPhaseSampler
* @param   sampler     Sampler object
* @param   x           Optional output pointer to the resulting x coordinate map (kArray2<k32s> with width equal to 
*                      count parameter of kS3dPhaseSampler_PhaseRoi and height equal to image height)
* @param   intensity   Optional output pointer to the resulting intensity coordinate map (kArray2<k8u> with width equal 
*                      to count parameter of kS3dPhaseSampler_PhaseRoi and
*                      height equal to image height)
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseSampler_ImagePhaseSamples(kS3dPhaseSampler sampler, kArray2* x, kArray2* intensity);

#include <kVision/S3d/kS3dPhaseSampler.x.h>

#endif  /* kS3D_PHASE_SAMPLER_H */
