/**
* @file    kS3dPhaseProcessor.h
* @brief   Declares the kS3dPhaseProcessor class.
*
* @internal
* Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
*/
#ifndef kS3D_PHASE_PROCESSOR_H
#define kS3D_PHASE_PROCESSOR_H

#include <kVision/S3d/kS3dPhaseViewDecompressor.h>
#include <kVision/S3d/kS3dPhaseViewProcessor.h>
#include <kVision/S3d/kS3dRangeFilter.h>
#include <kVision/S3d/kS3dSurfaceGenerator.h>

/**
* Sample code: https://docs.google.com/document/d/1rFWbk50PfrkcAe5IHGnkNDYgrNdNGYKUPKwElGUDGzE/edit#bookmark=id.uw25ur5j3npg
*
* @class       kS3dPhaseProcessor
* @extends     kObject
* @ingroup     kVision-S3d
* @brief       Rectifies and resamples camera phase maps to a uniform step in projector space and uniform step in rectified Y space. For a
*              a pair of phase maps acquired from a stereo camera pair, this sampling ensures that corresponding locations in the output buffers
*              form matching stereo points
*/
typedef kObject kS3dPhaseProcessor;

/**
* Constructs a kS3dPhaseProcessor object
*
* @public              @memberof kS3dPhaseProcessor
* @param   processor     Destination for the constructed object handle.
* @param   profiler    Profiler object to be used for rectification
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseProcessor_Construct(kS3dPhaseProcessor* processor, kS3dStereoProfiler profiler, kAlloc alloc);

/**
* Updates working buffers of the implementation to reflect current set of algorithm parameters.
* Calling this function is optional, as this validation step is also performed during each execution
* of the algorithm (kS3dPhaseProcessor_Begin, kS3dPhaseProcessor_Update).
* However, the initialization time may be non-negligible, which would affect the first execution of the algorithm.
* To eliminate the added delay from the first algorithm execution, the user should call kS3dPhaseProcessor_Setup
* after all of the parameters have been configured.
*
* @public              @memberof kS3dPhaseProcessor
* @param   processor   Processor object
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseProcessor_Setup(kS3dPhaseProcessor processor);

/**
* Initiates image processing sequence and initializes the output buffers. Should be called prior to calling
* kS3dPhaseProcessor_UpdateImageInput or kS3dPhaseProcessor_UpdateMapInput.  The output buffer handles are retained
* by the processor object and are used to populate the output rectified coordinate and intensity maps.
*
* Upon completion of the sequence (kS3dPhaseProcessor_IsComplete returns kTRUE)
* each entry in the output x projection buffer 'outputXProj' is populated with rectified X coordinates of the points sampled at
* resolution specified via kS3dPhaseProcessor_SetPhaseRoi and kS3dPhaseProcessor_SetYProjectionRoi
* Invalid points are set to k16S_NULL
*
* Optional parameter 'outputIntensity' provides a buffer for intensity output for the points corresponding to the X projection
* coordinates at the same locations. Invalid values are set to zero.
*
*
* @public                   @memberof kS3dPhaseProcessor
* @param   processor        Processor object
* @param   outputRanges     Output buffer for the triangulated range data. Expects kArray2<kPoint3d16s> with width determined by
*                           kS3dPhaseProcessor_SetYProjectionRoi 'count' parameter and height determined by
*                           kS3dPhaseProcessor_SetPhaseRoi 'sampleCount' parameter
* @param   outputIntensity  Output intensity buffer. Can be set to kNULL if intensity output isn't needed. Expect kArray2<k8u> with
*                           dimensions equal to those of 'outputRanges'
* @return                   Operation status.
*/
kVsFx(kStatus) kS3dPhaseProcessor_Begin(kS3dPhaseProcessor processor, kArray2 outputRanges, kArray2 outputIntensity);

/**
* Returns true when all image or map sequences have been processed and the outputs supplied to kS3dPhaseProcessor_Begin
* have been filled
*
* @public              @memberof kS3dPhaseProcessor
* @param   processor   Processor object
* @return              Operation status.
*/
kVsFx(kBool) kS3dPhaseProcessor_IsComplete(kS3dPhaseProcessor processor);

/**
* TODO: Rename all kS3dPhaseProcessor_UpdateXXX() functions to kS3dPhaseProcessor_Update()
*
* Update the processor with a single input image in the processing sequence. The images are expected in order of
* reference, stripe, phase with the exact number of images in each sub-sequence controlled via
* kS3dPhaseProcessor_SetSequenceLengthInfo
* The total number of sequences must match the count specified via kS3dPhaseProcessor_SetSequenceCount
* function
*
* Prior to beginning image processor the user should enable image input via kS3dPhaseProcessor_EnableImageInput
* and call kS3dPhaseProcessor_Begin to supply output buffers. The function kS3dPhaseProcessor_IsComplete will
* return true once all of the expected input images have been received
*
* @public              @memberof kS3dPhaseProcessor
* @param   processor   Processor object
* @param   viewIndex   View index
* @param   image       Image input. Expect kImage<k8u> with size determined by kS3dPhaseProcessor_SetImageSize call
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseProcessor_UpdateImage(kS3dPhaseProcessor processor, kSize viewIndex, kImage image);

/**
* Update the processor with multiple input images from a single view. Equivalent to calling kS3dPhaseProcessor_UpdateImageInput
* 'count' number of times
*
* @public              @memberof kS3dPhaseProcessor
* @param   processor   Processor object
* @param   viewIndex   View index
* @param   images      Pointer to the buffer of input image handles
* @param   count       Number of images in the input buffer
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseProcessor_UpdateImageList(kS3dPhaseProcessor processor, kSize viewIndex, const kImage* images, kSize count);

/**
* Update the processor with a subframe of a previously decoded and compressed phase map split into subframes.
* The total number of sequences must match the count specified via kS3dPhaseProcessor_SetSequenceCount() function.
*
* A call kS3dPhaseProcessor_Begin() to supply output buffers is required prior to calling this function.
* The function kS3dPhaseProcessor_IsComplete() will return true once all of the expected sequence inputs have been processed.
*
* @public              @memberof kS3dPhaseProcessor
* @param   processor   Processor object
* @param   viewIndex   View index
* @param   subframe    Decoded and compressed subframe of the phase map.
* @return              Operation status.
*/

kVsFx(kStatus) kS3dPhaseProcessor_UpdateCompression(kS3dPhaseProcessor processor, kSize viewIndex, kCompressedPhase subframe);

/**
* Update the processor with a previously decoded phase map.
* The total number of sequences must match the count specified via kS3dPhaseProcessor_SetSequenceCount() function.
*
* A call kS3dPhaseProcessor_Begin() to supply output buffers is required prior to calling this function.
* The function kS3dPhaseProcessor_IsComplete() will return true once all of the expected sequence inputs have been processed.
*
* @public              @memberof kS3dPhaseProcessor
* @param   processor   Processor object
* @param   viewIndex   View index
* @param   phasePixel2 Decoded phase map. Expect kArray2<kPhasePixel2 or k32s> with size determined by kS3dPhaseProcessor_SetImageSize() call.
* @return              Operation status.
*/

kVsFx(kStatus) kS3dPhaseProcessor_UpdatePixel2(kS3dPhaseProcessor processor, kSize viewIndex, kArray2 phasePixel2);

/**
* Plot stripe, phase, intensity and contrast data resulting from the decoding stage of the pipeline
*
* @public              @memberof kS3dPhaseProcessor
* @param   processor   Processor object
* @param   viewIndex   Index of the view from which the plot should be rendered
* @param   rawImage    Optional handle to a raw image to be copied as a top layer to the plot. The class does not
*                      take ownership of the image
* @param   plot        Output pointer to the resulting plot object
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseProcessor_DecodePlot(kS3dPhaseProcessor processor, kSize viewIndex, kImage rawImage, kPlot* plot);

/**
* Plot stripe and phase data as well as image coordinates sampled along the projection axis
*
* @public              @memberof kS3dPhaseProcessor
* @param   processor   Processor object
* @param   viewIndex   Index of the view from which the plot should be rendered
* @param   plot        Output pointer to the resulting plot object
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseProcessor_PhaseSamplePlot(kS3dPhaseProcessor processor, kSize viewIndex, kPlot* plot);

/**
* Outputs a text description of the last execution times and current memory usage
*
* @public              @memberof kS3dPhaseProcessor
* @param   processor   Processor object
* @param   report      Output pointer to the resulting text object
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseProcessor_ExecutionReport(kS3dPhaseProcessor processor, kString* report);

/**
* Get a handle to the internal buffer containing rectified X coordinates for a specified view
*
* @public              @memberof kS3dPhaseProcessor
* @param   processor   Processor object
* @param   viewIndex   View index of the rectified X map
* @return              Rectified x coordinates spaced according to phase and y rectified step specified by
*                      kS3dPhaseProcessor_SetPhaseRoi and kS3dPhaseProcessor_SetYProjectionRoi
*/
kVsFx(kArray2) kS3dPhaseProcessor_RectifiedXMap(kS3dPhaseProcessor processor, kSize viewIndex);

/**
* Sets the maximum number of threads allowed to be used for processing. Set to one for blocking single
* threaded operation
*
* Single threaded operation is configured as the default.
*
* @public              @memberof kS3dPhaseProcessor
* @param   processor   Processor object
* @param   count       Maximum number of threads allowed for processing.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseProcessor_SetParallelCount(kS3dPhaseProcessor processor, kSize count);

/**
* Gets the maximum number of threads allowed to be used for processing.
*
* @public              @memberof kS3dPhaseProcessor
* @param   processor   Processor object
* @return              Maximum number of threads allowed for processing.
*/
kVsFx(kSize) kS3dPhaseProcessor_ParallelCount(kS3dPhaseProcessor processor);

/**
* Enables image input. Must be set to kTRUE prior to using kS3dPhaseProcessor_UpdateImageInput function.
* If disabled, only direct map input is possible (kS3dPhaseProcessor_UpdateMapInput)
*
* @public              @memberof kS3dPhaseProcessor
* @param   processor   Processor object
* @param   enable      Flag specifying whether image input should be enabled
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseProcessor_EnableImageInput(kS3dPhaseProcessor processor, kBool enable);

/**
* Gets the value of the flag specifying whether raw image input is enabled.
*
* @public              @memberof kS3dPhaseProcessor
* @param   processor   Processor object
* @return              Flag specifying whether image input is enabled
*/
kVsFx(kBool) kS3dPhaseProcessor_ImageInputEnabled(kS3dPhaseProcessor processor);

/**
* Sets the lengths of the Gray code and phase subsequences. At most one reference image is supported.
* If no stripe sequence is used (phase period occupies full projection)
* the number of reference images must also be zero. Total number of expected images is the sum of the individual
* sub-sequence counts
*
* @public                        @memberof kS3dPhaseProcessor
* @param   processor             Processor object
* @param   referenceImageCount   Number of reference images in the image sequence (<= 1)
* @param   stripeImageCount      Number of stripe images in the image sequence (<= 7)
* @param   phaseImageCount       Number of phase images in the sequence
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseProcessor_SetStripeSequenceInfo(kS3dPhaseProcessor processor, kSize referenceImageCount, kSize stripeImageCount, kSize phaseImageCount);

/**
* Gets the lengths of the Gray code and phase subsequences.
*
* @public                        @memberof kS3dPhaseProcessor
* @param   processor             Processor object
* @param   referenceImageCount   Output pointer to the number of reference images in the image sequence
* @param   stripeImageCount      Output pointer to the number of stripe images in the image sequence
* @param   phaseImageCount       Output pointer to the number of phase images in the sequence
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseProcessor_StripeSequenceInfo(kS3dPhaseProcessor processor, kSize* referenceImageCount, kSize* stripeImageCount, kSize* phaseImageCount);

/**
* Sets period coefficients and the number of images per period.
* Coefficients are used to calculate the phase periods per set of patterns used.
*
* @public                     @memberof kS3dPhaseProcessor
* @param   processor          Processor object
* @param   coefficients       Array of coefficients describing embedded periods (k64f).
* @param   stepCounts         Array specifying number of images per period to use (kSize same size as "coefficients" and == coefficientCount).
* @param   coefficientCount   Size of the "coefficients" and "stepCounts" arrays.
* @return                     Operation status.
*/
kVsFx(kStatus) kS3dPhaseProcessor_SetEmbeddedPhaseSequenceInfo(kS3dPhaseProcessor processor, const k64f* coefficients, const kSize* stepCounts, kSize coefficientCount);

/**
* Gets period coefficients and the number of images per period.
* Coefficients are used to calculate the phase periods per set of patterns used.
*
* @public                        @memberof kS3dPhaseProcessor
* @param   processor             Processor object
* @param   outCoefficients       Array of coefficients describing embedded periods (k64f).
* @param   outStepCounts         Array specifying number of images per period to use (kSize same size as "coefficients" and == coefficientCount).
* @param   outCoefficientCount   Number of values in the "coefficients" and "stepCounts" arrays returned
* @param   capacity              Size of "coefficients" and "stepCounts" arrays passed in (capacity should be >= coefficientCount).
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseProcessor_EmbeddedPhaseSequenceInfo(kS3dPhaseProcessor processor, k64f* outCoefficients, kSize* outStepCounts, kSize* outCoefficientCount, kSize capacity);

/**
* Gets the number of images in the whole sequence aka total number of images used.
* Supports Stripe and EPS.
*
* @public              @memberof kS3dPhaseProcessor
* @param   processor   Processor object
* @return              Number of images in the sequence
*/
kVsFx(kSize) kS3dPhaseProcessor_SequenceLength(kS3dPhaseProcessor processor);

/**
* Sets the image encoding (decoding) type
*
* @public                        @memberof kS3dPhaseProcessor
* @param   processor             Processor object
* @param   type                  One of: kS3D_ENCODE_TYPE_STRIPE, kS3D_ENCODE_TYPE_EMBEDDED_PHASE
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseProcessor_SetEncodingType(kS3dPhaseProcessor processor, kS3dEncodeType type);

/**
* Gets the image encoding (decoding) type
*
* @public                        @memberof kS3dPhaseProcessor
* @param   processor             Processor object
* @return                        One of: kS3D_ENCODE_TYPE_STRIPE, kS3D_ENCODE_TYPE_EMBEDDED_PHASE
*/
kVsFx(kS3dEncodeType) kS3dPhaseProcessor_EncodingType(kS3dPhaseProcessor processor);

/**
* Sets the total number of decoding sequences.
*
* @public                @memberof kS3dPhaseProcessor
* @param   processor     Processor object
* @param   sequenceCount Number of expected decoding sequences (default is 1)
* @return                Operation status.
*/
kVsFx(kStatus) kS3dPhaseProcessor_SetSequenceCount(kS3dPhaseProcessor processor, kSize count);

/**
* Gets the total number of decoding sequences
*
* @public                @memberof kS3dPhaseProcessor
* @param   processor     Processor object
* @return                Number of expected decoding sequences
*/
kVsFx(kSize) kS3dPhaseProcessor_SequenceCount(kS3dPhaseProcessor processor);

/**
* Selects whether intensity data should be taken from first camera, second camera or a combination of the two.
*
* @public                @memberof kS3dPhaseProcessor
* @param   processor     Processor object
* @param   intensityType Intensity output type
* @return                Operation status.
*/
kVsFx(kStatus) kS3dPhaseProcessor_SetIntensityType(kS3dPhaseProcessor processor, kS3dStripeIntensityType intensityType);

/**
* Gets the intensity output type
*
* @public              @memberof kS3dPhaseProcessor
* @param   processor   Processor object
* @return              Intensity output type
*/
kVsFx(kS3dStripeIntensityType) kS3dPhaseProcessor_IntensityType(kS3dPhaseProcessor processor);

/**
* Sets the index of the sequence to be used for intensity output
*
* @public                @memberof kS3dPhaseProcessor
* @param   processor     Processor object
* @param   index         Index of the sequence to be used for intensity output
* @return                Operation status.
*/
kVsFx(kStatus) kS3dPhaseProcessor_SetIntensitySequenceIndex(kS3dPhaseProcessor processor, kSize index);

/**
* Gets the index of the sequence to be used for intensity output
*
* @public              @memberof kS3dPhaseProcessor
* @param   processor   Processor object
* @return              Index of the sequence to be used for intensity output
*/
kVsFx(kSize) kS3dPhaseProcessor_IntensitySequenceIndex(kS3dPhaseProcessor processor);

/**
* Sets contrast threshold for stripe decoding. The difference between stripe image pixel intensity and reference pixel intensity
* must be higher or equal to this value in order for the decoding at that pixel to be valid. Note that the value set by this function
* has no effect if map input is used
*
* @public              @memberof kS3dPhaseProcessor
* @param   processor   Processor object
* @param   threshold   Contrast threshold for stripe decoding
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseProcessor_SetStripeContrastThreshold(kS3dPhaseProcessor processor, kSize threshold);

/**
* Gets contrast threshold for stripe decoding
*
* @public              @memberof kS3dPhaseProcessor
* @param   processor   Processor object
* @return              Contrast threshold for stripe decoding
*/
kVsFx(kSize) kS3dPhaseProcessor_StripeContrastThreshold(kS3dPhaseProcessor processor);

/**
* Sets contrast threshold for phase decoding. The range of intensities for a given pixel location among all phase
* images must exceed this threshold value in order for the phase decoding to produce a valid result.
*
* @public              @memberof kS3dPhaseProcessor
* @param   processor   Processor object
* @param   threshold   Contrast threshold for phase decoding
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseProcessor_SetPhaseContrastThreshold(kS3dPhaseProcessor processor, kSize threshold);

/**
* Gets contrast threshold for phase decoding
*
* @public              @memberof kS3dPhaseProcessor
* @param   processor   Processor object
* @return              Contrast threshold for phase decoding
*/
kVsFx(kSize) kS3dPhaseProcessor_PhaseContrastThreshold(kS3dPhaseProcessor processor);

/**
* Sets the size of the expected input images.
*
* @public              @memberof kS3dPhaseProcessor
* @param   processor      Processor object
* @param   imageWidth  Width of the input images
* @param   imageHeight Height of the input images
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseProcessor_SetImageSize(kS3dPhaseProcessor processor, kSize imageWidth, kSize imageHeight);

/**
* Gets the size of input images
*
* @public              @memberof kS3dPhaseProcessor
* @param   processor     Processor object
* @param   imageWidth  Width of the input images
* @param   imageHeight Height of the input images
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseProcessor_ImageSize(kS3dPhaseProcessor processor, kSize* imageWidth, kSize* imageHeight);

/**
* Sets the parameters determining resampling region of interest along projection phase modulation axis (rows in both camera and projection images)
*
* @public                         @memberof kS3dPhaseProcessor
* @param   processor              Processor object
* @param   phasePeriod            Phase period (max phase) as calculated using: kS3dPhaseProcessor_CalculatePhasePeriod()
* @param   periodSampleCount      Number of samples per phase repetition
* @param   beginSample            First sample position to output.  Calculated as (fractional stripe begin) * periodSampleCount
* @param   sampleCount            Total number of projection samples. Calculated as (fractional stripe span) * periodSampleCount.
*                                 The value determines output range and intensity buffer height
* @return                         Operation status.
*/
kVsFx(kStatus) kS3dPhaseProcessor_SetPhaseRoi(kS3dPhaseProcessor processor, kSize phasePeriod, kSize periodSampleCount, kSize beginSample, kSize sampleCount);

/**
* Gets the parameters determining resampling region of interest along projection phase modulation axis (rows in both camera and projection images)
*
* @public                         @memberof kS3dPhaseProcessor
* @param   processor              Processor object
* @param   periodPeriod           Output pointer to the phase period value
* @param   periodSampleCount      Output pointer to the number of samples per phase repetition
* @param   beginSample            Output pointer to the first sample position to output. Calculated as fractional stripe begin * periodSampleCount
* @param   sampleCount            Output pointer to the total number of projection samples. Calculated as fractional stripe span * periodSampleCount.
*                                 The value determines output range and intensity buffer height
* @return                         Operation status.
*/
kVsFx(kStatus) kS3dPhaseProcessor_PhaseRoi(kS3dPhaseProcessor processor, kSize* phasePeriod, kSize* periodSampleCount, kSize* beginSample, kSize* sampleCount);

/**
* Sets the parameters determining resampling region of interest along the rectified Y axis (columns in both camera and projection images)
*
* @public              @memberof kS3dPhaseProcessor
* @param   processor   Processor object
* @param   begin       Starting rectified Y value expressed in mm.(must be a multiple of kS3dStereoCal_Parameters(cal)->projectionResolution
*                      to ensure that the full expected ROI is generated)
* @param   step        Step size along the rectified Y axis in mm (must be a multiple of kS3dStereoCal_Parameters(cal)->projectionResolution
*                      to ensure that the full expected ROI is generated)
* @param   count       Number of points along the rectified Y axis (output buffer width)
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseProcessor_SetYProjectionRoi(kS3dPhaseProcessor processor, k64f begin, k64f step, kSize count);

/**
* Gets the parameters determining resampling region of interest along the rectified Y axis (columns in both camera and projection images)
*
* @public              @memberof kS3dPhaseProcessor
* @param   processor   Processor object
* @param   begin       Output pointer to the starting rectified Y value (mm)
* @param   step        Output pointer to the step size along the rectified Y axis (mm)
* @param   count       Output pointer to the number of points along the rectified Y axis (output buffer width)
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseProcessor_YProjectionRoi(kS3dPhaseProcessor processor, k64f* begin, k64f* step, kSize* count);

/**
* Sets the flag for retain phase map input. This will allow phase combiner to save phase map input buffer which used 
* to generate decode plot
*
* @public                        @memberof kS3dPhaseProcessor
* @param   processor             Processor object
* @param   enable                Flag to retain phase map input
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseProcessor_EnableRetainInput(kS3dPhaseProcessor processor, kBool enable);

/**
* Gets the flag for retain phase map input.
*
* @public                        @memberof kS3dPhaseProcessor
* @param   processor             Processor object
* @return                        retain input flag state.
*/
kVsFx(kBool) kS3dPhaseProcessor_RetainInputEnabled(kS3dPhaseProcessor processor);

/**
* Enable outlier removal of the output range data
*
* @public              @memberof kS3dPhaseProcessor
* @param   processor   Processor object
* @param   enable      Flag specifying whether outlier removal should be enabled
* @return              Operation status
*/
kVsFx(kStatus) kS3dPhaseProcessor_EnableRangeFilter(kS3dPhaseProcessor processor, kBool enable);

/**
* Query whether outlier removal of the output range data is enabled
*
* @public              @memberof kS3dPhaseProcessor
* @param   processor   Processor object
* @return              Flag specifying whether outlier removal is enabled
*/
kVsFx(kBool) kS3dPhaseProcessor_RangeFilterEnabled(kS3dPhaseProcessor processor);

/**
* Set the type of correction to be applied to phase map data. Supported options:
* kS3D_PHASE_CORRECTION_TYPE_NONE - do not perform any correction (default when processor class is constructed)
* kS3D_PHASE_CORRECTION_TYPE_MULTI_REFLECTION - correct phase from multi reflection e.g.Combustion chamber.
* kS3D_PHASE_CORRECTION_TYPE_TRANSLUCENCY - correct phase for transparent surface, e.g. skin surface measurement.
*
* @public                  @memberof kS3dPhaseProcessor
* @param   processor       Processor object
* @param   correctionType  Phase correction type
* @return                  Operation status.
*/
kVsFx(kStatus) kS3dPhaseProcessor_SetPhaseCorrectionType(kS3dPhaseProcessor processor, kS3dPhaseCorrectionType correctionType);

/**
* Gets the type of correction to be applied to phase map data.
*
* @public              @memberof kS3dPhaseProcessor
* @param   processor   Processor object
* @return              Phase correction type
*/
kVsFx(kS3dPhaseCorrectionType) kS3dPhaseProcessor_PhaseCorrectionType(kS3dPhaseProcessor processor);

/**
* Set the margin around valid phase boundary to erode. This operation can help clean up spikes and transitions around object edges caused by lens blur.
* The value is specified in pixels and is zero by default
*
* @public                  @memberof kS3dPhaseProcessor
* @param   processor       Processor object
* @param   erosionMargin   Erosion margin (px)
* @return                  Operation status.
*/
kVsFx(kStatus) kS3dPhaseProcessor_SetErosionMargin(kS3dPhaseProcessor processor, kSize erosionMargin);

/**
* Gets the erosion margin around valid phase boundary.
*
* @public                  @memberof kS3dPhaseProcessor
* @param   processor       Processor object
* @return                  Erosion margin (px)
*/
kVsFx(kSize) kS3dPhaseProcessor_ErosionMargin(kS3dPhaseProcessor processor);

/**
* Get the raw stripe output as kImage<k8u>.
*
* @public                  @memberof kS3dPhaseViewProcessor
* @param   processor       Processor object
* @param   viewIndex       The index of the view (0 or 1)
* @param   stripe          Pointer to a kImage that will be filled with the output.
* @return                  Operation status.
*/
kVsFx(kStatus) kS3dPhaseProcessor_ViewStripe(kS3dPhaseProcessor processor, kSize viewIndex, kImage* stripe);

/**
* Get the raw phase output as kImage<k16s>.
*
* @public                  @memberof kS3dPhaseViewProcessor
* @param   processor       Processor object
* @param   viewIndex       The index of the view (0 or 1)
* @param   phase           Pointer to a kImage that will be filled with the output.
* @return                  Operation status.
*/
kVsFx(kStatus) kS3dPhaseProcessor_ViewPhase(kS3dPhaseProcessor processor, kSize viewIndex, kImage* phase);

/**
* Get the intensity output as kImage<k8u>.
*
* @public                  @memberof kS3dPhaseViewProcessor
* @param   processor       Processor object
* @param   viewIndex       The index of the view (0 or 1)
* @param   intensity       Pointer to a kImage that will be filled with the output.
* @return                  Operation status.
*/
kVsFx(kStatus) kS3dPhaseProcessor_ViewIntensity(kS3dPhaseProcessor processor, kSize viewIndex, kImage* intensity);

/**
* Get the unwrapped (stripe+phase) output as a kImage<k64f>. Unwrapped phase value is StripeValue * PhasePeriod + PhaseValue
* providing a continuous index across the phase pattern, combining the stripe and phase data.
*
* @public                  @memberof kS3dPhaseViewProcessor
* @param   processor       Processor object
* @param   viewIndex       The index of the view (0 or 1)
* @param   unwrappedPhase  Pointer to a kImage that will be filled with the output.
* @return                  Operation status.
*/
kVsFx(kStatus) kS3dPhaseProcessor_ViewUnwrappedPhase(kS3dPhaseProcessor processor, kSize viewIndex, kImage* unwrappedPhase);

/**
* Get the high frequency unwrapped phase output as a kImage<k64f>. High frequency phase is the difference between the
* unwrapped phase and a 5x5 pixel moving average in X. In other words, a high-pass filter is applied. This method produces
* reliable results on contiguous targets only, since the moving average presumes continuity.
*
* @public                       @memberof kS3dPhaseViewProcessor
* @param   processor            Processor object
* @param   viewIndex            The index of the view (0 or 1)
* @param   highFrequencyPhase   Pointer to a kImage that will be filled with the output.
* @return                       Operation status.
*/
kVsFx(kStatus) kS3dPhaseProcessor_ViewHighFrequencyPhase(kS3dPhaseProcessor processor, kSize viewIndex, kImage* highFrequencyPhase);

/**
* Export all scan and diagnostic data available from the most recent scan
*
* @public                       @memberof kS3dPhaseViewProcessor
* @param   processor            Processor object
* @param   output               A pointer to a kDataTree. Will be filled with a new kDataTree containing the view data
* @return                       Operation status.
*/
kVsFx(kStatus) kS3dPhaseProcessor_ExportViewData(kS3dPhaseProcessor processor, kDataTree* output);

///@param pixelXSpacing Mean distance between pixels (rays) in X world coords within ROI. Units mm.
///@param sampleXSpacing Mean distance between samples in X world coords within ROI. Units mm.
kVsFx(kStatus) kS3dPhaseProcessor_MeanPixelXSpacing(kS3dPhaseProcessor processor, kArray2 ranges64f, kArray2 texture0, kRect32s roi, k64f* pixelXSpacing, k64f* sampleXSpacing);

/**
* Enables compression. Must be set to kTRUE prior to using Setup() / Begin() / UpdateMapInputCompressed() functions.
*
* @public              @memberof kS3dPhaseProcessor
* @param   processor   Processor object
* @param   enable      Flag specifying whether compression should be enabled
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseProcessor_EnableCompression(kS3dPhaseProcessor processor, kBool enable);

/**
* Gets the value of the flag specifying whether compression input is enabled.
*
* @public              @memberof kS3dPhaseProcessor
* @param   processor   Processor object
* @return              Flag specifying whether compression is enabled
*/
kVsFx(kBool) kS3dPhaseProcessor_CompressionEnabled(kS3dPhaseProcessor processor);

/**
* Sets the number of subframes in the compression. (Number of parts)
*
* @public                        @memberof kS3dPhaseProcessor
* @param   processor             Processor object
* @param   subframeCount         Number of subframes
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseProcessor_SetCompressionSubframeCount(kS3dPhaseProcessor processor, kSize subframeCount);

/**
* Gets the number of subframes in the compression. (Number of parts)
*
* @public                        @memberof kS3dPhaseProcessor
* @param   processor             Processor object
* @return                        Number of subframes
*/
kVsFx(kSize) kS3dPhaseProcessor_CompressionSubframeCount(kS3dPhaseProcessor processor);

/**
* Calculates phase period based on the number of phase samples per phase period (repetition)
* and a hardcoded phase period bit depth kS3D_PHASE_PROCESSOR_PHASE_BIT_DEPTH.
*
* @public                          @memberof kS3dPhaseProcessor
* @param   phasePeriodSampleCount  Number of phase samples per phase period (repetition).
* @return                          Phase period (kSIZE_NULL in case the number of phase samples is bigger than 2^kS3D_PHASE_PROCESSOR_PHASE_BIT_DEPTH)
*/

kInlineFx(kSize) kS3dPhaseProcessor_CalculatePhasePeriod(kSize phasePeriodSampleCount);

/**
* Sets the flag to enable the use of cuda managed memory.
* Optional if we want to override the default behavior.
*   (Defaults to: kS3dPhaseProcessor_IsManagedMemoryEnabled())
*
* @public                        @memberof kS3dPhaseViewProcessor
* @param   processor             Processor object
* @param   cudaManagedMem        Flag to enable cuda managed memory
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseProcessor_EnableManagedMemory(kS3dPhaseProcessor processor, kBool cudaManagedMem);

/**
* Gets the flag for cuda managed memory.
*
* @public                        @memberof kS3dPhaseViewProcessor
* @param   processor             Processor object
* @return                        Cuda managed memory state.
*/
kVsFx(kBool) kS3dPhaseProcessor_ManagedMemoryEnabled(kS3dPhaseProcessor processor);

#include <kVision/S3d/kS3dPhaseProcessor.x.h>

#endif  /* kS3D_PHASE_PROCESSOR_H */
