/**
* @file    kS3dPhaseExpand.h
* @brief   Declares the kS3dPhaseExpand class. 
*
* @internal
* Copyright (C) 2015-2022 by LMI Technologies Inc. All rights reserved.
*/

#ifndef kS3D_PHASE_EXPAND_H
#define kS3D_PHASE_EXPAND_H


#include <kVision/Common/kVision.h>
#include <kVision/S3d/kS3dCommon.h>

/**
* @class       kS3dPhaseExpand
* @extends     kObject
* @ingroup     kVision-M3d
* @brief       Performs expanding operation on a phase map. 
*              The order of the segmented small area will be taken over 
*              from the neighboring points with the largest segment area in pixel step.
*/
typedef kObject kS3dPhaseExpand;

/**
* Constructs a kS3dPhaseExpand object
*
* @public              @memberof kS3dPhaseExpand
* @param   expand      Destination for the constructed object handle.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseExpand_Construct(kS3dPhaseExpand* expand, kAlloc allocator);

/**
* Runs the expanding operation.  
*
* @public              @memberof kS3dPhaseExpand
* @param   expand      kS3dPhaseExpand object.
* @param   data        Phase map of kArray2 <k32s>. The both width and height of the array should be larger than 8 pixel.
* @param   map         Segment map. Each point of the phase map is grouped with an index. kArray2 <k32s>
*                      The both width and height of the array should be identical to the phase map array.  
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseExpand_Run(kS3dPhaseExpand expand, kArray2 data, kArray2 map);

/**
* Set the repeat number of filtering
*
* @public              @memberof kS3dPhaseExpand
* @param   expand      kS3dPhaseExpand object.
* @param   repeat      Repeat number
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseExpand_SetRepeat(kS3dPhaseExpand expand, kSize repeat);

/**
* Get the repeat number of filtering
*
* @public              @memberof kS3dPhaseExpand
* @param   expand      kS3dPhaseExpand object.
* @return              Repeat number
*/
kVsFx(kSize) kS3dPhaseExpand_Repeat(kS3dPhaseExpand expand);

/**
* Set the digital count of one phase period.
*
* @public              @memberof kS3dPhaseExpand
* @param   expand      kS3dPhaseExpand object.
* @param   period      Digital count of one phase period.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseExpand_SetPeriod(kS3dPhaseExpand expand, kSize period);

/**
* Get the digital count of one phase period.
*
* @public              @memberof kS3dPhaseExpand
* @param   expand      kS3dPhaseExpand object.
* @return              Digital count of one phase period.
*/
kVsFx(kSize) kS3dPhaseExpand_Period(kS3dPhaseExpand expand);

/**
* Set the segment index. All points in the phase map with this index are isolated with very small area.
*
* @public              @memberof kS3dPhaseExpand
* @param   expand      kS3dPhaseExpand object.
* @param   limitIndex  Segment index with isolated points. This value was determined by segmenting process. 
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseExpand_SetSegmentLimitIndex(kS3dPhaseExpand expand, kSize limitIndex);

/**
* Get the segment index. All points in the phase map with this index are isolated with very small area.
*
* @public              @memberof kS3dPhaseExpand
* @param   expand      kS3dPhaseExpand object.
* @return              Segment index with isolated points.
*/
kVsFx(kSize) kS3dPhaseExpand_SegmentLimitIndex(kS3dPhaseExpand expand);

/**
* Set the View index. For storing intermediate results  if necessary.
*
* @public              @memberof kS3dPhaseExpand
* @param   expand      kS3dPhaseExpand object.
* @param   index       View index: 0 or 1.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseExpand_SetViewIndex(kS3dPhaseExpand expand, kSize index);

#include <kVision/S3d/kS3dPhaseExpand.x.h>

#endif
