/**
* @file    kS3dPhaseDecoder.x.h
*
* @internal
* Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
*/
#ifndef kS3D_PHASE_DECODER_X_H
#define kS3D_PHASE_DECODER_X_H

#define kS3D_PHASE_DECODER_DEFAULT_CONTRAST_THRESHOLD     (10)

#define kS3D_PHASE_DECODER_ATAN_LUT_SHIFT                 (16)
#define kS3D_PHASE_DECODER_ATAN_LUT_MAX                   (1 << kS3D_PHASE_DECODER_ATAN_LUT_SHIFT)
#define kS3D_PHASE_DECODER_INTENSITY_SHIFT                (8)

#define kS3D_PHASE_DECODER_EVAL_PHASE_EXACT(OBJ, DX, DY, LUT, OUT)\
do\
{\
k64f phaseF = atan2(DY, DX) / (2*kMATH_PI) * (OBJ)->lutPhasePeriod;\
*(OUT) = kMath_Round16s_(phaseF);\
if((*OUT) < 0) (*OUT) += (k16s)(OBJ)->lutPhasePeriod;\
}\
while(0);\


/* Should be converted to an inline function and performance verified */
#define kS3D_PHASE_DECODER_EVAL_PHASE(OBJ, DX, DY, TANLUT, OUT)\
do\
{\
    k32s dyAbs = kAbs_(DY);\
    k32s dxAbs = kAbs_(DX);\
    k32s ySign = (DY) > 0 ? 1 : -1;\
    k32s angleTan;\
    k32s outValue = k16S_NULL;\
    if(dyAbs < dxAbs && (dxAbs >> 12) != 0)\
        {\
        angleTan = (dyAbs << 4) / (dxAbs >> 12);\
        angleTan = kMin_(angleTan, kS3D_PHASE_DECODER_ATAN_LUT_MAX);\
        outValue = (DX) > 0 ? ySign*TANLUT[angleTan] : ySign*((OBJ)->lutPhaseHPeriod - TANLUT[angleTan]);\
        if(outValue < 0) outValue += (OBJ)->lutPhasePeriod;\
        }\
        else if((dyAbs >> 12 ) != 0)\
    {\
        angleTan = (dxAbs << 4) / (dyAbs >> 12);\
        angleTan = kMin_(angleTan, kS3D_PHASE_DECODER_ATAN_LUT_MAX);\
        outValue = (DX) > 0 ? ySign*((OBJ)->lutPhaseQPeriod - TANLUT[angleTan]) : ySign*((OBJ)->lutPhaseQPeriod + TANLUT[angleTan]);\
        if(outValue < 0) outValue += (OBJ)->lutPhasePeriod;\
    }\
    *(OUT) = (k16s)outValue;\
}\
while(0);\


typedef struct kS3dPhaseDecoderAccum 
{
    k32s dx;
    k32s dy;

    k8u minValue;
    k8u maxValue;
    k16u sum;

} kS3dPhaseDecoderAccum;

typedef struct kS3dPhaseDecoderClass
{
    kObjectClass base;

    // Algorithm parameters
    kBool useCuda;
    kSize imageWidth;
    kSize imageHeight;
    kSize phaseImageCount;
    kSize contrastThreshold;
    kSize phasePeriod;
    kBool atanEnabled;

    kSize acquiredMask;
    kArray2 output;
    kArray2 accum;

    kArray1 atanLut;
    kArray1 sinCoeffLut;
    kArray1 cosCoeffLut;

    k32s lutPhaseImageCount;
    k32s lutPhasePeriod;
    k32s lutPhaseHPeriod;
    k32s lutPhaseQPeriod;

    kCudaStream cudaStream;

} kS3dPhaseDecoderClass;

kDeclareClassEx(kVs, kS3dPhaseDecoder, kObject)

//semi-private exported functions (virtual override methods)

kVsFx(kStatus) xkS3dPhaseDecoder_VInitClone(kS3dPhaseDecoder decoder, kS3dPhaseDecoder source, kAlloc allocator);
kVsFx(kStatus) xkS3dPhaseDecoder_VRelease(kS3dPhaseDecoder decoder);
kVsFx(kSize)   xkS3dPhaseDecoder_VSize(kS3dPhaseDecoder decoder);

//non-exported (private) methods
kStatus xkS3dPhaseDecoder_Init(kS3dPhaseDecoder decoder, kBool useCuda, kAlloc allocator);

kVsFx(kStatus) xkS3dPhaseDecoder_UpdateHost(kS3dPhaseDecoder decoder, kSize index, kImage image);
kVsFx(kStatus) xkS3dPhaseDecoder_UpdateCuda(kS3dPhaseDecoder decoder, kSize index, kImage image);

//cast macro

#endif  /* #ifndef kS3D_PHASE_DECODER_X_H */
