/**
 * @file    ${filename_header_public}}
 * @brief   Declares the ${classname} class.
 *
 * @internal
 * Copyright (C) 2016-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef kS3D_MONO_MERGE_RIGID_ALG_H
#define kS3D_MONO_MERGE_RIGID_ALG_H

#include <kVision/Vs/kVsJobQueue.h>
#include <kVision/S3d/kS3dStereoProfiler.h>

/**
 * @class                    kS3dMonoMergeRigidAlg
 * @extends                  kObject
 * @ingroup                  kVision - S3d
 * @brief                    Algorithm class for merging mono and stereo meshes
 */

typedef kObject kS3dMonoMergeRigidAlg;

/**
 * Constructs a kS3dMonoMergeRigidAlg object.
 *
 * @public                   @memberof kS3dMonoMergeRigidAlg
 * @param   alg              Destination for the constructed object handle.
 * @param   allocator        Memory allocator (or kNULL for default).
 * @return                   Operation status.
 */

kVsFx(kStatus) kS3dMonoMergeRigidAlg_Construct(kS3dMonoMergeRigidAlg* alg, kS3dStereoProfiler profiler, kAlloc allocator); 

/**
 * Optional function to preallocate some of the internal buffers.
 * Requires kS3dMonoMergeRigidAlg_SetHeight() and kS3dMonoMergeRigidAlg_SetWidth() to be set.
 *
 * @public                   @memberof kS3dMonoMergeRigidAlg
 * @param   alg              Alg object
 * @return                   Operation status.
 */

kVsFx(kStatus) kS3dMonoMergeRigidAlg_Setup(kS3dMonoMergeRigidAlg alg);

/**
 * Optional function to set the input output array dimensions.
 * Only required if kS3dMonoMergeRigidAlg_Setup() is used for preallocation.
 *
 * @public                   @memberof kS3dMonoMergeRigidAlg
 * @param   alg              Alg object
 * @param   height           Input/output arrays height.
 * @return                   Operation status.
 */

kVsFx(kStatus) kS3dMonoMergeRigidAlg_SetHeight(kS3dMonoMergeRigidAlg alg, kSize height);

/**
 * Optional function to set the input output array dimensions.
 * Only required if kS3dMonoMergeRigidAlg_Setup() is used for preallocation.
 *
 * @public                   @memberof kS3dMonoMergeRigidAlg
 * @param   alg              Alg object
 * @param   width            Input/output arrays width.
 * @return                   Operation status.
 */

kVsFx(kStatus) kS3dMonoMergeRigidAlg_SetWidth(kS3dMonoMergeRigidAlg alg, kSize width);

/**
 * Set outlier tolerance for the merging algorithms.
 * Used for outlier removal as an average multiplier during the sampling stage.
 * Tolerance = Sqr(tol) * average
 *
 * @public                   @memberof kS3dMonoMergeAlg
 * @param   alg              Alg object
 * @param   tol              Tolerance for what can be considered an outlier.
 */

kVsFx(kStatus) kS3dMonoMergeRigidAlg_SetOutlierTolerance(kS3dMonoMergeRigidAlg alg, k64f tol);

/**
 * Get outlier tolerance for the merging algorithms.
 * Used for outlier removal as an average multiplier during the sampling stage.
 * Tolerance = Sqr(tol) * average
 *
 * @public                   @memberof kS3dMonoMergeAlg
 * @param   alg              Alg object
 * @param   tol              Tolerance for what can be considered an outlier.
 */

kVsFx(kStatus) kS3dMonoMergeRigidAlg_OutlierTolerance(kS3dMonoMergeRigidAlg alg, k64f* tol);
/**
 * Set the job queue for parallel processing.
 *
 * @public                   @memberof kS3dMonoMergeAlg
 * @param   alg              Alg object
 * @param   queue            Job queue for parallel processing.
 * @return                   Operation status.
 */

kVsFx(kStatus) kS3dMonoMergeRigidAlg_SetJobQueue(kS3dMonoMergeRigidAlg alg, kVsJobQueue queue);

/**
* Executes the rigid mono merge by filling in the missing data in the provided stereo array. 
*
* @public                   @memberof kS3dMonoMergeRigidAlg
* @param   alg              Alg object
* @param   stereo           An array of stereo point to fill with missing data from monos
* @param   yBegin           Sample start
* @param   yStep            Sample step
* @param   x0               Resampled phase map 0
* @param   x1               Resampled phase map 1
* @param   phaseTable       Phase table
* @return                   Operation status
*/

kVsFx(kStatus) kS3dMonoMergeRigidAlg_Run(kS3dMonoMergeRigidAlg alg, kArray2 stereo, kSSize yBegin, kSize yStep, kArray2 x0, kArray2 x1, kArray1 phaseTable);

/**
* Executes the rigid mono merge by filling in the missing data in the provided stereo array.
* Same as kS3dMonoMergeRigidAlg_Run() only executes in Cuda and assumes the data is allocated on device.
*
* @public                   @memberof kS3dMonoMergeRigidAlg
* @param   alg              Alg object
* @param   stereo           An array of stereo point to fill with missing data from monos
* @param   yBegin           Sample start
* @param   yStep            Sample step
* @param   x0               Resampled phase map 0
* @param   x1               Resampled phase map 1
* @param   phaseTable       Phase table
* @return                   Operation status
*/

#if defined (K_HAVE_CUDA)
kVsFx(kStatus) kS3dMonoMergeRigidAlg_RunCuda(kS3dMonoMergeRigidAlg alg, kArray2 stereo, kSSize yBegin, kSize yStep, kArray2 x0, kArray2 x1, kArray1 phaseTable);
#else
kInlineFx(kStatus) kS3dMonoMergeRigidAlg_RunCuda(kS3dMonoMergeRigidAlg alg, kArray2 stereo, kSSize yBegin, kSize yStep, kArray2 x0, kArray2 x1, kArray1 phaseTable) { return kERROR; }
#endif

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

#include <kVision/S3d/kS3dMonoMergeRigidAlg.x.h>

#endif  /* kS3D_MONO_MERGE_RIGID_ALG_H */
