#ifndef kS3D_ENCODING_MATH_H
#define kS3D_ENCODING_MATH_H

#include <kVision/Common/kVision.h>
#include <kApi/Data/kMath.h>

//////////////////////////////////////////////////////////////////////////
// Set of functions and constants used in varoous 
// decoding and pattern generations algs
// 
// kS3dPhaseProcessor
// kS3dPhaseViewProcessor
// kS3dEmbeddedPhaseDecoder
// kS3dPatternSequence
//////////////////////////////////////////////////////////////////////////

#define kS3D_PHASE_STEP_MAX_COUNT            (30)

//////////////////////////////////////////////////////////////////////////
// All the trigonometric functions below are in radians
//////////////////////////////////////////////////////////////////////////

/**
* Used in conjunction with kS3dEncodingMath_PhaseSteps() for generating step (shift) angles for the phase image sequence
*
* Trims the [0, 2*PI] range to [0, 2*PI - trim]
* This way phase step images don't correspond to trivial angles like PI/2, PI, 3*PI/4 eliminating degeneracies where cosine or sine is 0
* 
* Example:
*    const k64f stepsTrim = kS3dEncodingMath_PhaseStepsTrim(count);
*    k64f phaseSteps[kS3D_MAX_PHASE_STEP_COUNT];
*
*    kCheck(kS3dEncodingMath_PhaseSteps(6, stepsTrim, phaseSteps, kS3D_MAX_PHASE_STEP_COUNT));
*
* 
* @public              @memberof kS3dEncodingMath
* @param   stepCount   Number of steps (images) per phase sequence
* @return              Trim value in radians
*/
kInlineFx(k64f) kS3dEncodingMath_PhaseStepsTrim(kSize stepCount);

/**
* Generates step (shift) angles for the phase image sequence based on the number of images to be used
*
* Example:
*    const k64f stepsTrim = kS3dEncodingMath_PhaseStepsTrim(count);
*    k64f phaseSteps[kS3D_MAX_PHASE_STEP_COUNT];
*
*    kCheck(kS3dEncodingMath_PhaseSteps(6, stepsTrim, phaseSteps, kS3D_MAX_PHASE_STEP_COUNT));
*
* @public              @memberof kS3dEncodingMath
* @param   outSteps    Out array of steps (shift) angles per image (in radians)
* @param   capacity    Size of the outSteps >= stepCount
* @return              Operation status.
*/
kInlineFx(kStatus) kS3dEncodingMath_PhaseSteps(kSize stepCount, k64f trim, k64f* outSteps, kSize capacity);

//////////////////////////////////////////////////////////////////////////
// Set of functions to convert embedded phase coefficients to actual pattern periods
// Generally we store a set coefficients for EPS decoding 
// The encoding/decoding alg can derive a set of periods to be used 
// See kS3dEmbeddedPhaseDecoder.h for details 
//
// coefficients -> embeddedPeriods -> patternPeriods 
//
//////////////////////////////////////////////////////////////////////////

/**
* Convert coefficients to embedded periods.
*
* @public                   @memberof kS3dEncodingMath
* @param   coefficients     Array of input coefficients
* @param   count            Size of "coefficients" array
* @param   embeddedPeriods  Out array of embedded periods
* @param   capacity         Size of "embeddedPeriods" array
* @return                   Operation status.
*/
kInlineFx(kStatus) kS3dEncodingMath_EmbeddedPhasePeriods(const k64f* coefficients, kSize count, k64f* embeddedPeriods, kSize capacity);

/**
* Convert embedded periods to actual patterns periods (projected images).
*
* @public                   @memberof kS3dEncodingMath
* @param   embeddedPeriods  Array of input embedded periods
* @param   count            Size of "embeddedPeriods" array
* @param   patternPeriods   Out array of pattern periods (projected images)
* @param   capacity         Size of "patternPeriods" array
* @return                   Operation status.
*/
kInlineFx(kStatus) kS3dEncodingMath_PatternPhasePeriods(const k64f* embeddedPeriods, kSize count, k64f* patternPeriods, kSize capacity);

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

#include <kVision/S3d/kS3dEncodingMath.x.h>

#endif /* #ifndef kS3D_ENCODING_MATH_H */
