/**
 * @file    kS3dEmbeddedPhaseDecoderInt.x.h
 *
 * @internal
 * Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef kS3D_EMBEDDED_PHASE_DECODER_INT_X_H
#define kS3D_EMBEDDED_PHASE_DECODER_INT_X_H

kBeginHeader()
//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

#define kS3D_EPD_INT_KUWAHARA_AVERAGE_SHIFT         (6)                                                 // Increase precision of ave^2

// sumAve = 4 * ave * w; // 4 moving sums
// sumAve = 2^2 * 2^16 * w;
// if maxW is 32: 2^32 = 2^18 * 2^x; // w = 2^x
// x = 14 max
// if maxW is 64: 2^64 = 2^18 * 2^x; // w = 2^x
// x = 46 max

//#define kS3D_EPD_INT_KUWAHARA_WEIGHT_SHIFT          (24)                                                // const k64u maxW = ((k64u)1 << 24);
#define kS3D_EPD_INT_KUWAHARA_WEIGHT_SHIFT          (18)                                                // should be 24 but had to drop it to 18 due to PL limitations

// window max 15
// 15 * 15 = 225 <= 2^8
// sum = 2^16 * 2^8 = 2^24
// sumSqr = 2^16 * 2^16 * 2^8 = 2^40
// 40 - 32 = 8

#define kS3D_EPD_INT_MOVING_SUM_SHIFT               (8)

//////////////////////////////////////////////////////////////////////////
// Micro Phase Shifting http://www.cs.columbia.edu/CAVE/projects/micro_phase_shifting/
// solving for M*U = R same as in MPS paper (better described than in EPS paper)
//
// Embedded Phase Shifting http://mesh.brown.edu/eps/
// Paper SVN:  https://svn1.lmi-tech.com/sensors/Gocator/Documents/3xxx/Papers/Moreno_Embedded_Phase_Shifting_2015_CVPR_paper.pdf
//////////////////////////////////////////////////////////////////////////

typedef struct
{
    k8u minValue;
    k8u maxValue;
    k16u sum;
} kS3dEmbeddedPhaseDecoderIntPrimeAccum;

typedef struct
{
    k32u sum;
    k32u sumSqr;
    kSize count;

//#define kS3D_EPD_INT_DEB_KUWAHARA
#ifdef kS3D_EPD_INT_DEB_KUWAHARA
    // DEB //
    k64s sumDEB;
    k64s sumSqrDEB;
#endif

} MovingSumInt;

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

typedef struct
{
    kObjectClass base;

    // params //
    kSize imageWidth;
    kSize imageHeight;

    kArray1 coefficients;    // k64f
    kArray1 embeddedPeriods; // k64f

    kArray1 stepCounts;      // kSize, images per period

    kSize phaseScale;        // [0,2pi] rescaled to [0, phaseScale] so phaseScale is 2pi in k16s
                             // also phase is [0, phaseScale]

    kSize contrastThreshold;

    // internals //
    kSize periodCount;       // num frequencies
    kSize imageCount;        // sequence length
    kSize pixelCount;        // imageHeight * imageWidth

    kSize filterWindow;

    kSize updatedImageCount; // how many images already in

    kBool updateSystem;
    kBool updateImageSize;

    kS3dEmbeddedPhaseDecoderOrientation orientation;

    // Using paper notations here
    // M*U = R
    // Mt*M*U = Mt*R
    // U = inv(Mt*M)*Mt*R
    // A = inv(Mt*M)*Mt
    // U = A*R
    //
    // R - inputImages intensities vector
    // A - shiftMatrix mangled as M is a true shift matrix
    // U - output cos, sin matrix
    // see kS3dEmbeddedPhaseDecoderInt_CalculatePhases()

    kArray2 shiftMatrix; // kArray2<k32s>

    // accumulates all the images as rows
    kArray2 inputImages;  // kArray2<k8u>(imageCount, pixelCount)

    kArray2 primeAccum; // kArray2<kS3dEmbeddedPhaseDecoderIntPrimeAccum>
    kArray1 phaseArray; // kArray1<kArray2<k16s>> per period

    kArray2 tmpPhase;       // temp buffer

    kArray2 outputBaseMap;  // k8u
    kArray2 outputPhaseMap; // kPhasePixel

    kArrayList inputPhaseArrayState;  // unit test report kArrayList<kArray2<k16s/k32f>> [n alg steps][coefficientCount]
    kArrayList outputPhaseArrayState; // unit test report kArrayList<kArray2<k16s>> [n alg steps][coefficientCount]

    // moving sum array used by kS3dEmbeddedPhaseDecoderInt_SmoothKuwahara()
    kArray2 movingSum;

} kS3dEmbeddedPhaseDecoderIntClass;

kDeclareClass(kVs, kS3dEmbeddedPhaseDecoderInt, kObject)

//////////////////////////////////////////////////////////////////////////
//semi-private exported functions (virtual override methods)
//////////////////////////////////////////////////////////////////////////

kVsFx(kStatus) kS3dEmbeddedPhaseDecoderInt_VRelease(kS3dEmbeddedPhaseDecoderInt decoder);
kVsFx(kSize)   kS3dEmbeddedPhaseDecoderInt_VSize(kS3dEmbeddedPhaseDecoderInt decoder);

//////////////////////////////////////////////////////////////////////////
//non-exported (private) methods
//////////////////////////////////////////////////////////////////////////

kStatus kS3dEmbeddedPhaseDecoderInt_Init(kS3dEmbeddedPhaseDecoderInt decoder, kAlloc allocator);

kStatus kS3dEmbeddedPhaseDecoderInt_MatrixConvert(kArray2 mIn, kArray2 mOut, k32s value);
kStatus kS3dEmbeddedPhaseDecoderInt_UpdateSystem(kS3dEmbeddedPhaseDecoderInt decoder);
kStatus kS3dEmbeddedPhaseDecoderInt_UpdateAccumulators(kS3dEmbeddedPhaseDecoderInt decoder);

kStatus kS3dEmbeddedPhaseDecoderInt_UpdateFinishFN(kS3dEmbeddedPhaseDecoderInt decoder);

kStatus kS3dEmbeddedPhaseDecoderInt_CalculatePrime(kS3dEmbeddedPhaseDecoderInt decoder);

kStatus kS3dEmbeddedPhaseDecoderInt_CalculatePhases(kS3dEmbeddedPhaseDecoderInt decoder);
kStatus kS3dEmbeddedPhaseDecoderInt_CalculatePhase(kS3dEmbeddedPhaseDecoderInt decoder, kArray2 phaseU, kSize period, kArray2 phase);

kStatus kS3dEmbeddedPhaseDecoderInt_SubtractPhase(kS3dEmbeddedPhaseDecoderInt decoder, kArray2 fN, kArray2 f1);
kStatus kS3dEmbeddedPhaseDecoderInt_SubtractPhaseBase(kS3dEmbeddedPhaseDecoderInt decoder, kArray2 fN, kArray2 f1);

kStatus kS3dEmbeddedPhaseDecoderInt_SmoothKuwahara(kS3dEmbeddedPhaseDecoderInt decoder, kSize window, kArray2 inPhase, kArray2 outPhase);

kStatus kS3dEmbeddedPhaseDecoderInt_BaseRamp(kS3dEmbeddedPhaseDecoderInt decoder, k32s primePeriod, kArray2 primePhase, k32s basePeriod, kArray2 basePhase);

kStatus kS3dEmbeddedPhaseDecoderInt_Unwrap(kS3dEmbeddedPhaseDecoderInt decoder, k32s phasePeriod, kArray2 phase, k32s basePeriod, kArray2 base);

kStatus kS3dEmbeddedPhaseDecoderInt_Export(kS3dEmbeddedPhaseDecoderInt decoder, kArray2 base);

//semi-private methods (API can change any time)
kStatus kS3dEmbeddedPhaseDecoderInt_SetJobQueue(kS3dEmbeddedPhaseDecoderInt decoder, kVsJobQueue queue);

//////////////////////////////////////////////////////////////////////////
//cast macro, virtual table macro
//////////////////////////////////////////////////////////////////////////

#define kS3dEmbeddedPhaseDecoderInt_Cast_(ID)         kCastClass_(kS3dEmbeddedPhaseDecoderInt, ID)

//////////////////////////////////////////////////////////////////////////
// Alg state functions for unit tests 
//////////////////////////////////////////////////////////////////////////

kInlineFx(kStatus) kS3dEmbeddedPhaseDecoderInt_CopyPhase(kS3dEmbeddedPhaseDecoderInt decoder, kArray2 phaseSource, kArray2 phaseTarget, kBool rampFlag)
{
    kS3dEmbeddedPhaseDecoderIntClass* obj = kS3dEmbeddedPhaseDecoderInt_Cast_(decoder);

    kCheckArgs(kArray2_Count(phaseSource) == kArray2_Count(phaseTarget));   

    for (kSize h = 0; h < kArray2_Length(phaseSource, 0); ++h)
    {
        for (kSize w = 0; w < kArray2_Length(phaseSource, 1); ++w)
        {
            if (kArray2_ItemType(phaseSource) == kTypeOf(k16s))
            {
                kArray2_AsT(phaseTarget, h, w, k16s) = kArray2_AsT(phaseSource, h, w, k16s);
            }
            else if (kArray2_ItemType(phaseSource) == kTypeOf(k32f))
            {
                k32f phase = kArray2_AsT(phaseSource, h, w, k32f);

                if (phase == k32F_NULL)
                {
                    kArray2_AsT(phaseTarget, h, w, k16s) = k16S_NULL;
                }
                else if (rampFlag)
                {
                    kArray2_AsT(phaseTarget, h, w, k16s) = (k16s)(phase); // ramp no need to scale
                }
                else
                {
                    kArray2_AsT(phaseTarget, h, w, k16s) = (k16s)(phase * obj->phaseScale);
                }
            }
        }
    }

    return kOK;
}

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

kInlineFx(kStatus) kS3dEmbeddedPhaseDecoderInt_UpdatePhaseState(kS3dEmbeddedPhaseDecoderInt decoder, kBool rampFlag = kFALSE)
{
    kS3dEmbeddedPhaseDecoderIntClass* obj = kS3dEmbeddedPhaseDecoderInt_Cast_(decoder);

    if (kIsNull(obj->outputPhaseArrayState)) return kOK;

    kStatus status;
    kAlloc alloc = kObject_Alloc(decoder);
    kArray1 outState = kNULL;

    kTry;
    {
        // output //
        kTest(kObject_Clone(&outState, obj->phaseArray, alloc));

        kTest(kArrayList_Add(obj->outputPhaseArrayState, &outState));

        // input //
        if (!kIsNull(obj->inputPhaseArrayState))
        {
            kSize statePos = kArrayList_Count(obj->outputPhaseArrayState) - 1;

            kTestArgs(kArrayList_Count(obj->inputPhaseArrayState) > statePos);

            kArray1 inState = kArrayList_AsT(obj->inputPhaseArrayState, statePos, kArray1);

            kSize phaseCount = kArray1_Count(inState);

            for (kSize fi = 0; fi < phaseCount; ++fi)
            {
                kArray2 inPhase = kArray1_AsT(inState, fi, kArray2);
                kArray2 outPhase = kArray1_AsT(obj->phaseArray, fi, kArray2);

                kTest(kS3dEmbeddedPhaseDecoderInt_CopyPhase(decoder, inPhase, outPhase, rampFlag && fi == phaseCount - 1)); // F3 is a ramp flag
            }
        }
    }
    kCatch(&status);
    {
        kCheck(kDisposeRef(&outState));

        kEndCatch(status);
    }

    return kOK;
}

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////


kEndHeader()
#endif  /* #ifndef kS3D_EMBEDDED_PHASE_DECODER_INT_X_H */
