/**
* @file    kS3dDecompressor.x.h
*
* @internal
* Copyright (C) 2018-2022 by LMI Technologies Inc.  All rights reserved.
*/
#ifndef K_VISION_S3D_DECOMPRESSOR_X_H
#define K_VISION_S3D_DECOMPRESSOR_X_H

#include <kApi/Threads/kThreadPool.h>

#define kS3D_DECOMPRESSOR_IS_CONCURRENT                (kTRUE)      //can be disabled for testing

//////////////////////////////////////////////////////////////////////////
//
// kS3dDecompressorTile
//
//////////////////////////////////////////////////////////////////////////

/**
* @class       kS3dPhaseDecoderTile
* @extends     kObject
* @ingroup     kVision-S3d
* @brief       Performs decoding activities for phase or intensity in one subframe.
*/
typedef kObject kS3dDecompressorTile;

typedef struct kS3dDecompressorTileClass
{
    kObjectClass base;

    kBool isIntensity;                                  //is this an intensity tile (vs phase)?

    kCompressedPhase subframe;                          //compressed input data

    kUnpackedCompressedPhase unpacked;                  //unpacked lists (delta, fail1, fail2, null)
    kObject unpackContext;                              //reusable context object for unpacking

    kArray1 decompressed;                               //attaches to slice within user output structure

    kPointer deltaCompressor;                           //decoding engine (DeltaCompressor<T>)
    kPointer deltaCompressorEncoding;                   //temporary data structure for decoding engine (DeltaCompresssor::Encoding<T>)

} kS3dDecompressorTileClass;

kDeclareClassEx(kVs, kS3dDecompressorTile, kObject)

kVsFx(kStatus) xkS3dDecompressorTile_Construct(kS3dDecompressorTile* tile, kS3dDecompressor decompressor, kBool isIntensity, kAlloc allocator);
kVsFx(kStatus) xkS3dDecompressorTile_Init(kS3dDecompressorTile tile, kType type, kS3dDecompressor decompressor, kBool isIntensity, kAlloc alloc);
kVsFx(kStatus) xkS3dDecompressorTile_VRelease(kS3dDecompressorTile tile);

kVsFx(kStatus) xkS3dDecompressorTile_PrepareDecompress(kS3dDecompressorTile tile, kCompressedPhase subframe, kPointer subframeOutputSlice);
kVsFx(kStatus) xkS3dDecompressorTile_Decompress(kS3dDecompressorTile tile);
kVsFx(kStatus) xkS3dDecompressorTile_Unpack(kS3dDecompressorTile tile);
kVsFx(kStatus) xkS3dDecompressorTile_Decode(kS3dDecompressorTile tile);

//////////////////////////////////////////////////////////////////////////
//
// kS3dDecompressorTiles
// A simple pair for passing to a thread pool as one parameter to decompress a subframe.
// Note:
//   When decompressing into kPhasePixel2 we cant decode intensity and phase tiles for 
//   the same subframe simultaneously as they overwrite each other (pixel2 stores both intensity and phase)
//   Therefore we split the work into pairs of tiles (one pair of phase and intensity for each subframe)
//
//////////////////////////////////////////////////////////////////////////

/**
* @class       kS3dDecompressorSubframeTiles
* @extends     kObject
* @ingroup     kVision-S3d
* @brief       Performs decoding activities for one subframe.
*/
typedef kObject kS3dDecompressorSubframeTiles;

typedef struct kS3dDecompressorSubframeTilesClass
{
    kObjectClass base;

    kS3dDecompressorTile phase;
    kS3dDecompressorTile intensity;

} kS3dDecompressorSubframeTilesClass;

kDeclareClassEx(kVs, kS3dDecompressorSubframeTiles, kObject)

kVsFx(kStatus) xkS3dDecompressorSubframeTiles_Construct(kS3dDecompressorSubframeTiles* tiles, kAlloc allocator);
kVsFx(kStatus) xkS3dDecompressorSubframeTiles_Init(kS3dDecompressorSubframeTiles tiles, kType type, kAlloc alloc);
kVsFx(kStatus) xkS3dDecompressorSubframeTiles_VRelease(kS3dDecompressorSubframeTiles tiles);

kVsFx(kStatus) xkS3dDecompressorSubframeTiles_Decompress(kS3dDecompressorSubframeTiles tiles); // kThreadFx type function

kInlineFx(kStatus) xkS3dDecompressorSubframeTiles_SetPhase(kS3dDecompressorSubframeTiles tiles, kS3dDecompressorTile tile)
{
    kObj(kS3dDecompressorSubframeTiles, tiles);
    obj->phase = tile;
    return kOK;
}

kInlineFx(kS3dDecompressorTile) xkS3dDecompressorSubframeTiles_Phase(kS3dDecompressorSubframeTiles tiles)
{
    kObj(kS3dDecompressorSubframeTiles, tiles);
    return obj->phase;
}

kInlineFx(kStatus) xkS3dDecompressorSubframeTiles_SetIntensity(kS3dDecompressorSubframeTiles tiles, kS3dDecompressorTile tile)
{
    kObj(kS3dDecompressorSubframeTiles, tiles);
    obj->intensity = tile;
    return kOK;
}

kInlineFx(kS3dDecompressorTile) xkS3dDecompressorSubframeTiles_Intensity(kS3dDecompressorSubframeTiles tiles)
{
    kObj(kS3dDecompressorSubframeTiles, tiles);
    return obj->intensity;
}

//////////////////////////////////////////////////////////////////////////
//
// kS3dDecompressor
//
//////////////////////////////////////////////////////////////////////////

typedef struct kS3dDecompressorClass
{
    kObjectClass base;

    kBool isIntensity;
    kBool isPhase;
    kSize expectedFrameSize;
    kSize expectedSubframeCapacity;

    kQueue freePhaseTiles;                  //queue of phase tiles available for reuse
    kQueue phaseTiles;                      //active phase tiles

    kQueue freeIntensityTiles;              //queue of intensity tiles available for reuse
    kQueue intensityTiles;                  //active intensity tiles

    kQueue subframeTiles;                   //kQueue<kS3dDecompressorSubframeTiles> of tile pairs for thread pool

    kArray2 output;                         //reference to user output container for decompressed data

    kSize subframeCount;                    //count of received subframes in the current frame
    kSize subframeOffset;                   //offset to start decompressed data for current subframe, in pixels
    kBool lastFrameProvided;                //has the final subframe in the frame been observed?

    kThreadPool threadPool;                 //thread pool for decoding operations
    kQueue transactions;                    //list of all outstanding thread pool transactions

} kS3dDecompressorClass;

kDeclareClassEx(kVs, kS3dDecompressor, kObject)

kVsFx(kStatus) xkS3dDecompressor_Init(kS3dDecompressor decompressor, kType type, kAlloc alloc);
kVsFx(kStatus) xkS3dDecompressor_VRelease(kS3dDecompressor decompressor);
kVsFx(kSize)   xkS3dDecompressor_VSize(kS3dDecompressor processor);

kVsFx(kStatus) xkS3dDecompressor_BeginDecompress(kS3dDecompressor decompressor, kS3dDecompressorSubframeTiles tiles, kCompressedPhase subframe, kSize subframeOffset);
kVsFx(kStatus) xkS3dDecompressor_AllocatePhaseTile(kS3dDecompressor decompressor, kS3dDecompressorTile* tile);
kVsFx(kStatus) xkS3dDecompressor_AllocateIntensityTile(kS3dDecompressor decompressor, kS3dDecompressorTile* tile);
kVsFx(kStatus) xkS3dDecompressor_AllocateSubframeTiles(kS3dDecompressor decompressor, kS3dDecompressorSubframeTiles* tiles);

// Inline //

kInlineFx(kStatus) kS3dDecompressor_EnableIntensity(kS3dDecompressor decompressor, kBool intensity)
{
    kObj(kS3dDecompressor, decompressor);

    obj->isIntensity = intensity;

    return kOK;
}

kInlineFx(kBool) kS3dDecompressor_IsIntensity(kS3dDecompressor decompressor)
{
    kObj(kS3dDecompressor, decompressor);

    return obj->isIntensity;
}

kInlineFx(kStatus) kS3dDecompressor_EnablePhase(kS3dDecompressor decompressor, kBool phase)
{
    kObj(kS3dDecompressor, decompressor);

    obj->isPhase = phase;

    return kOK;
}

kInlineFx(kBool) kS3dDecompressor_IsPhase(kS3dDecompressor decompressor)
{
    kObj(kS3dDecompressor, decompressor);

    return obj->isPhase;
}

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

#endif
