/**
* @file    kS3dCheckerBoard.h
* @brief   Declares the kS3dCheckerBoard Class.
*
* @internal
* Copyright (C) 2016-2022 by LMI Technologies Inc. All rights reserved.
*/

#ifndef KVISION_S3D_CHECKERBOARD_H
#define KVISION_S3D_CHECKERBOARD_H

#include <kApi/kApiDef.h>
#include <kApi/kAlloc.h>
#include <kVision/Common/kVision.h>
#include <kVision/G3d/kG3dPolynomMap.h>
#include <kApi/Data/kMath.h>
#include <kApi/Data/kArray1.h>
#include <kApi/Data/kArray2.h>
#include <kApi/Data/kArray3.h>
#include <kApi/Data/kArrayList.h>
#include <kApi/Data/kBytes.h>
#include <kApi/Data/kString.h>
#include <kApi/Io/kPath.h>
#include <kVision/L3d/kL3dTransform3d.h>
#include <kApi/Data/kImage.h>
#include <math.h>
#include <time.h>
#include <kFireSync/Data/kPlot.h>
#include <kFireSync/Data/kGraphic.h>
#include <kApi/Io/kDirectory.h>
#include <kApi/Io/kFile.h>
#include <stdio.h>
#include <kVision/L3d/kL3dLattice2d.h>

/**
* @class       kS3dCheckerBoard
* @extends     kObject
* @ingroup     kVision-M3d
* @brief       Calculates the chessboard image into 4 polynomial phase functions with Hough transformation method.
*
* Methodology
*
* 1. The evaluation of the edge lines is performed by deriving the input camera image in the X and Y directions separately.
* 2. It is assumed that the lines can be considered with a few of curvature in the first approximation as straight lines.
* 3. It is assumed that the maximum of the partial derivative is corresponding to the edge line.
* 4. It is further assumed that the distance between the adjacent lines changes smoothly and continuously in the entire
*    image from the camera view.
* 5. That is, it exists functional relationships between the camera and Chessboard coordinates
*    that can be described one another with 4 polynomial functions:
*
*        Xcamera=polynomial1(Xchess, Ychess)
*        Ycamera=polynomial2(Xchess, Ychess)
*        Xchess=polynomial3(Xcamera, Ycamera)
*        Ychess=polynomial4(Xcamera, Ycamera)

* 6. The sorting of lines is performed by Hough transformation
* 7. Calculation of polynomial function using the value of partial derivative as a weight factor for each point.
* 8. Search and determination of the center-ellipse is also performed by using the partial derivative.
*
* Limitations
*
* 1. The camera image acquisition should prefer to avoid  the strong saturation at white areas (Blooming effect).
* 2. It was assumed that the chessboard will never be rotated greater than +/- 7 degrees in relation to the camera orientation.
* 3. It was assumed that the chessboard will never be tilted greater than 25 degrees in relation to the optical axis of camera.
* 4. The grid spacing must be at least 40 pixels in camera view.
* 5. The chessboard should as much as possible to cover the entire camera image.
* 6.The detectable edge lines shall be not less than 7 in the two directions.
*
*/
typedef kObject kS3dCheckerBoard;

/**
* Constructs a kS3dCheckerBoard object
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard   Destination for the constructed object handle.
* @param   allocator    Memory allocator (or kNULL for default).
* @return               Operation status.
*/
kVsFx(kStatus) kS3dCheckerBoard_Construct(kS3dCheckerBoard* checkerBoard, kAlloc allocator);

/**
* Calculates the chessboard image into 4 polynomial phase functions.
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard   kS3dCheckerBoard object.
* @param   input        Input picture of chessboard.   
* @return               Operation status.
*/
kVsFx(kStatus) kS3dCheckerBoard_Run(kS3dCheckerBoard checkerBoard, kImage input);

/**
* Get a kArray2 object contains the corner points of the chess board. Unit is pixel.
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard   kS3dCheckerBoard object.
* @param   corner       Destination for the kArray2 <kPoint64f> object handle. Conner points. should be destroyed after use.
*                       The mark is in the center:  k64f xm = gridWidth / 2., k64f ym = gridHeight /2. ;
* @param   gridHeight   Count of the chess board in x direction, should always be an odd number.
* @param   gridWidth    Count of the chess board in y direction, should always be an odd number.
* @return               Operation status.
*/
kVsFx(kStatus) kS3dCheckerBoard_CornerMatrix(kS3dCheckerBoard checkerBoard, kArray2* corner, k32s gridHeight, k32s gridWidth);

/**
* Get corrdinate of the mark in camera view. Unit is pixel.
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard   kS3dCheckerBoard object.
* @return               Mark coordinate.
*/
kVsFx(const kPoint64f*) kS3dCheckerBoard_MarkerPosition(kS3dCheckerBoard checkerBoard);

/**
* Create a kArray1 object to plot the mark ellipse. Unit is pixel.
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard   kS3dCheckerBoard object.
* @param   ellipse      Destination for the kArrayList <kPoint32f> object handle. Ellipse curve. should be destroyed after use.
* @param   count        Determines the curve fineness. Default: 360 (points) 
* @return               Operation status.
*/
kVsFx(kStatus) kS3dCheckerBoard_EllipseLine(kS3dCheckerBoard checkerBoard, kArray1* ellipse, k32s count);

/**
* Get the minimum line index of the chessboard in x direction.
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard   kS3dCheckerBoard object.
* @return               Minimum line index.
*/
kVsFx(k32s) kS3dCheckerBoard_MinimumIndexX(kS3dCheckerBoard checkerBoard);

/**
* Get the maximum line index of the chessboard in x direction.
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard   kS3dCheckerBoard object.
* @return               Maximum line index.
*/
kVsFx(k32s) kS3dCheckerBoard_MaximumIndexX(kS3dCheckerBoard checkerBoard);

/**
* Get the minimum line index of the chessboard in y direction.
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard   kS3dCheckerBoard object.
* @return               Minimum line index.
*/
kVsFx(k32s) kS3dCheckerBoard_MinimumIndexY(kS3dCheckerBoard checkerBoard);

/**
* Get the maximum line index of the chessboard in y direction.
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard   kS3dCheckerBoard object.
* @return               Maximum line index.
*/
kVsFx(k32s) kS3dCheckerBoard_MaximumIndexY(kS3dCheckerBoard checkerBoard);

/**
* Get two kArrayList objects to plot the edge lines in x and y directions.
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard   kS3dCheckerBoard object.
* @param   xLines       Destination for the kArrayList <kPoint32f> object handle. Edge lines in x directuion. should be destroyed after use.
* @param   yLines       Destination for the kArrayList <kPoint32f> object handle. Edge lines in y directuion. should be destroyed after use.
* @return               Operation status.
*/
kVsFx(kStatus) kS3dCheckerBoard_BorderLines(kS3dCheckerBoard checkerBoard, kArrayList* xLines, kArrayList* yLines);

/**
* Get polynomial coefficients. Chess_X = polynomial(camera_X,camera_Y). 
*
* Conner coordinate X and Y are always -11.5, -0.5, 0.5, 1.5, 11.5 ... with the circle mark as center (0, 0)
* camera_X = (IndexX - ImageSizeX / 2) * scale;   camera_Y = (IndexY - ImageSizeY / 2) * scale;
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard   kS3dCheckerBoard object.
* @return               polynomial coefficients.
*/
kVsFx(k64f*) kS3dCheckerBoard_PolynCameraXyToChessX(kS3dCheckerBoard checkerBoard);

/**
* Get polynomial coefficients. Chess_Y = polynomial(camera_X,camera_Y).  
*
* Conner coordinate X and Y are always -11.5, -0.5, 0.5, 1.5, 11.5 ... with the circle mark as center (0, 0)
* camera_X = (IndexX - ImageSizeX / 2) * scale;   camera_Y = (IndexY - ImageSizeY / 2) * scale; 
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard   kS3dCheckerBoard object.
* @return               polynomial coefficients.
*/
kVsFx(k64f*) kS3dCheckerBoard_PolynCameraXyToChessY(kS3dCheckerBoard checkerBoard);

/**
* Get polynomial coefficients. Camera_X = polynomial(chess_X,chess_Y).  
* Conner coordinate X and Y are always -11.5, -0.5, 0.5, 1.5, 11.5 ... with the circle mark as center (0, 0)
* camera_X = (IndexX - ImageSizeX / 2) * scale;   camera_Y = (IndexY - ImageSizeY / 2) * scale;
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard   kS3dCheckerBoard object.
* @return               polynomial coefficients.
*/
kVsFx(k64f*) kS3dCheckerBoard_PolynChessXyToCameraX(kS3dCheckerBoard checkerBoard);

/**
* Get polynomial coefficients. Camera_Y = polynomial(chess_X,chess_Y).
* Conner coordinate X and Y are always -11.5, -0.5, 0.5, 1.5, 11.5 ... with the circle mark as center (0, 0)
* camera_X = (IndexX - ImageSizeX / 2) * scale;   camera_Y = (IndexY - ImageSizeY / 2) * scale;
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard   kS3dCheckerBoard object.
* @return               polynomial coefficients.
*/
kVsFx(k64f*) kS3dCheckerBoard_PolynChessXyToCameraY(kS3dCheckerBoard checkerBoard);

/**
* Get the width of the input camera image 
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard   kS3dCheckerBoard object.
* @return               Operation status.
*/
kVsFx(k32s) kS3dCheckerBoard_ImageSizeX(kS3dCheckerBoard checkerBoard);

/**
* Get the height of the input camera image 
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard   kS3dCheckerBoard object.
* @return               Operation status.
*/
kVsFx(k32s) kS3dCheckerBoard_ImageSizeY(kS3dCheckerBoard checkerBoard);

/**
* Get the order of the polynomial function. 
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard   kS3dCheckerBoard object.
* @return               order of the polynomial function.
*/
kVsFx(k32s) kS3dCheckerBoard_Order(kS3dCheckerBoard checkerBoard);

/**
* Get the physical camera pixel pitch in mm. 
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard   kS3dCheckerBoard object.
* @return               Camera pixel pitch.
*/
kVsFx(k64f) kS3dCheckerBoard_Scale(kS3dCheckerBoard checkerBoard);

/**
* Get the internal calculating pixel step to speed up the calculation. Default = 4
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard   kS3dCheckerBoard object.
* @return               Calculating pixel step.
*/
kVsFx(k32s) kS3dCheckerBoard_Step(kS3dCheckerBoard checkerBoard);

/**
* Set the width of the input camera image 
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard   kS3dCheckerBoard object.
* @param   imageSizeX   Width of camera image
* @return               Operation status.
*/
kVsFx(kStatus) kS3dCheckerBoard_SetImageSizeX(kS3dCheckerBoard checkerBoard, k32s imageSizeX);

/**
* Set the heighth of the input camera image 
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard   kS3dCheckerBoard object.
* @param   imageSizeY   Height of camera image
* @return               Operation status.
*/
kVsFx(kStatus) kS3dCheckerBoard_SetImageSizeY(kS3dCheckerBoard checkerBoard, k32s imageSizeY);

/**
* Set the order of the polynomial function. 
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard   kS3dCheckerBoard object.
* @param   order        Order of the polynomial function.
* @return               Operation status.
*/
kVsFx(kStatus) kS3dCheckerBoard_SetOrder(kS3dCheckerBoard checkerBoard, k32s order);

/**
* Set the physical camera pixel pitch in mm. 
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard   kS3dCheckerBoard object.
* @param   scale        Physical camera pixel pitch in mm.
* @return               Operation status.
*/
kVsFx(kStatus) kS3dCheckerBoard_SetScale(kS3dCheckerBoard checkerBoard, k64f scale);

/**
* Set the internal calculating pixel step to speed up the calculation. Default = 6
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard   kS3dCheckerBoard object.
* @param   step         Internal calculating pixel step
* @return               Operation status.
*/
kVsFx(kStatus) kS3dCheckerBoard_SetStep(kS3dCheckerBoard checkerBoard, k32s step);

/**
* Set grid width of checkerboard. 
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard kS3dCheckerBoard object.
* @param   gridWidth    grid width of checkerboard.  Default = 60
* @return               Operation status.
*/
kVsFx(kStatus) kS3dCheckerBoard_SetGridWidth(kS3dCheckerBoard checkerBoard, k32s gridWidth);

/**
* Set grid height of checkerboard.
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard kS3dCheckerBoard object.
* @param   gridHeight   grid height of checkerboard.  Default = 60
* @return               Operation status.
*/
kVsFx(kStatus) kS3dCheckerBoard_SetGridHeight(kS3dCheckerBoard checkerBoard, k32s gridHeight);

/**
* Get lattice map obect.
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard kS3dCheckerBoard object.
* @return               Lattice map obect
*/
kVsFx(kL3dLattice2d) kS3dCheckerBoard_Lattice(kS3dCheckerBoard checkerBoard);

/**
* Get raw camera image.
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard kS3dCheckerBoard object.
* @return               Raw camera image.
*/
kVsFx(kImage) kS3dCheckerBoard_RawImage(kS3dCheckerBoard checkerBoard);

/**
* Set reference point.
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard kS3dCheckerBoard object.
* @param   reference     I don't know, Why do we need this point.
* @return               Operation status.
*/
kVsFx(kStatus) kS3dCheckerBoard_SetReferencePoint(kS3dCheckerBoard checkerBoard, kPoint32s reference);

/**
* Set reference point.
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard kS3dCheckerBoard object.
* @param   world        The rect object is used for Lattice map obect.
* @return               Operation status.
*/
kVsFx(kStatus) kS3dCheckerBoard_SetWorldRect(kS3dCheckerBoard checkerBoard, kRect64f world);

/**
* Get output display image.
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard kS3dCheckerBoard object.
* @param   plot         Output object handle of kPlot.
* @return               Operation status.
*/
kVsFx(kStatus) kS3dCheckerBoard_Plot(kS3dCheckerBoard checkerBoard, kPlot* plot);

/**
* Get reference kL3dLattice2d object. This function is only used for verification.
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard kS3dCheckerBoard object.
* @return               Reference lattice map obect.
*/
kVsFx(kL3dLattice2d) kS3dCheckerBoard_ReferenceLattice(kS3dCheckerBoard checkerBoard);

/**
* Set reference kL3dLattice2d object. This function is only used for verification to display the sorted corner points.
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard kS3dCheckerBoard object.
* @param   refLattice   Reference lattice object.
* @return               Operation status.
*/
kVsFx(kStatus) kS3dCheckerBoard_SetReferenceLattice(kS3dCheckerBoard checkerBoard, kL3dLattice2d  refLattice);

/**
* Get reference corner object. This function is only used for verification.
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard kS3dCheckerBoard object.
* @return               Reference Corner object.
*/
kVsFx(kArrayList) kS3dCheckerBoard_ReferenceCorners(kS3dCheckerBoard checkerBoard);

/**
* Set reference corner object. This function is only used for verification to display the unsorted corner points.
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard kS3dCheckerBoard object.
* @param   refCorners   Reference corner object of kArrayList < kPoint64f >.
* @return               Operation status.
*/
kVsFx(kStatus) kS3dCheckerBoard_SetReferenceCorners(kS3dCheckerBoard checkerBoard, kArrayList refCorners);

/**
* Get reference marker position. This function is only used for verification.
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard kS3dCheckerBoard object.
* @return               Reference marker position.
*/
kVsFx(kPoint64f) kS3dCheckerBoard_ReferenceCenter(kS3dCheckerBoard checkerBoard);

/**
* Set reference marker position. This function is only used for verification to display the marker.
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard kS3dCheckerBoard object.
* @param   center       Reference marker position.
* @return               Operation status.
*/
kVsFx(kStatus) kS3dCheckerBoard_SetReferenceCenter(kS3dCheckerBoard checkerBoard, kPoint64f   center);

/**
* Get Output polynomial function in x direction: phaseX(camX, camY).
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard kS3dCheckerBoard object.
* @return               Output polynomial for  phaseX(camX, camY).
*/
kVsFx(kG3dPolynomial) kS3dCheckerBoard_PolynMCX(kS3dCheckerBoard checkerBoard);

/**
* Get Output polynomial function in y direction: phaseY(camX, camY).
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard kS3dCheckerBoard object.
* @return               Output polynomial for  phaseX(camX, camY).
*/
kVsFx(kG3dPolynomial) kS3dCheckerBoard_PolynMCY(kS3dCheckerBoard checkerBoard);

/**
* Get Output polynomial inverse function in x direction: CamX(phaseX, phaseY).
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard kS3dCheckerBoard object.
* @return               Output inverse polynomial for CamX(phaseX, phaseY).
*/
kVsFx(kG3dPolynomial) kS3dCheckerBoard_PolynCMX(kS3dCheckerBoard checkerBoard);

/**
* Get Output polynomial inverse function in y direction: CamY(phaseX, phaseY).
*
* @public               @memberof kS3dCheckerBoard
* @param   checkerBoard kS3dCheckerBoard object.
* @return               Output inverse polynomial for CamY(phaseX, phaseY).
*/
kVsFx(kG3dPolynomial) kS3dCheckerBoard_PolynCMY(kS3dCheckerBoard checkerBoard);

#include <kVision/S3d/kS3dCheckerBoard.x.h>

#endif /* #ifndef KVISION_S3D_CHECKERBOARD_H */
