/**
* @file kMp3dGeoCalUtilities.h
*
* @internal
* Copyright (C) 2017-2022 by LMI Technologies Inc.  All rights reserved.
*/

#ifndef LMITECH_KVISION_MP3D_GEOCALUTILITIES_H_INCLUDED
#define LMITECH_KVISION_MP3D_GEOCALUTILITIES_H_INCLUDED

#include <kVision/Common/kNullMath.h>
#include <kVision/Mp3d/kMp3dGeoCal.h>

/**
* Applies X resolution to Z fit from GeoCal object to obtain X resolution at given Z.
* The X and Y resolution (mm/pixel) of an imager's pixels varies with the range. When resampling images,
* it is necessary to know the native resolution of each pixel. Typically, this is only needed for color/texture output.
* The X resolution at a given Z range (mm) is determined by:
*
*       X resolution = coefficients[0] + coefficients[1]*Z + ... + coefficients[N]*Z^N
*
* @public                           @memberof kMp3dGeoCal
* @param  cal                       kMp3dGeoCal calibration object.
* @param  cameraIndex               Camera index. kERROR_PARAMETER if invalid.
* @param  runWindowXSubsampling     X subsampling of run window
* @param  z                         Input range
* @return                           X resolution at given Z. k64F_NULL if cal contains no X res to Z fit.
*/
kVsFx(k64f) kMp3dGeoCalUtilities_ApplyXResolutionFit(kMp3dGeoCal cal, kSize cameraIndex, k32u runWindowXSubsampling, k64f z);

/**
* Applies X resolution to Z fit from GeoCal object on buffer of Z data to obtain buffer of X resolutions.
*
* @public                           @memberof kMp3dGeoCal
* @param  cal                       kMp3dGeoCal calibration object.
* @param  cameraIndex               Camera index. kERROR_PARAMETER if invalid.
* @param  runWindowXSubsampling     X subsampling of run window
* @param  z                         Data buffer of Z ranges
* @param  xRes                      Output buffer of X resolutions
* @param  count                     Count of Z ranges.
* @return                           Operation status.
*/
kVsFx(kStatus) kMp3dGeoCalUtilities_ApplyXResolutionFitN(kMp3dGeoCal cal, kSize cameraIndex, k32u runWindowXSubsampling, const k64f* z, k64f* xRes, kSize count);

/**
* Applies Y resolution to Z fit from GeoCal object to obtain X resolution at given Z.
* The X and Y resolution (mm/pixel) of an imager's pixels varies with the range. When resampling images,
* it is necessary to know the native resolution of each pixel. Typically, this is only needed for color/texture output.
* The Y resolution at a given Z range (mm) is determined by:
*
*       Y resolution = coefficients[0] + coefficients[1]*Z + ... + coefficients[N]*Z^N
*
* @public                           @memberof kMp3dGeoCal
* @param  cal                       kMp3dGeoCal calibration object.
* @param  cameraIndex               Camera index. kERROR_PARAMETER if invalid.
* @param  runWindowYSubsampling     Y subsampling of run window
* @param  z                         Input range
* @return                           Y resolution at given Z. k64F_NULL if cal contains no Y res to Z fit.
*/
kVsFx(k64f) kMp3dGeoCalUtilities_ApplyYResolutionFit(kMp3dGeoCal cal, kSize cameraIndex, k32u runWindowYSubsampling, k64f z);

/**
* Applies Y resolution to Z fit from GeoCal object on buffer of Z data to obtain buffer of Y resolutions.
*
* @public                           @memberof kMp3dGeoCal
* @param  cal                       kMp3dGeoCal calibration object.
* @param  cameraIndex               Camera index. kERROR_PARAMETER if invalid.
* @param  runWindowYSubsampling     Y subsampling of run window
* @param  z                         Data buffer of Z ranges
* @param  yRes                      Output buffer of Y resolutions
* @param  count                     Count of Z data.
* @return                           Operation status.
*/
kVsFx(kStatus) kMp3dGeoCalUtilities_ApplyYResolutionFitN(kMp3dGeoCal cal, kSize cameraIndex, k32u runWindowYSubsampling, const k64f* z, k64f* yRes, kSize count);

/**
* Applies X center to Z fit from GeoCal object to obtain X center at given Z.
* The world X center (mm) of an imager varies with the range if any yaw is present. It also represents the distance
* from the sensor reference 0 plane (usually mid-sensor). The X center at a given Z range (mm) can be determined by:
*
*       X center = coefficients[0] + coefficients[1]*Z + ... + coefficients[N]*Z^N
*
* @public                           @memberof kMp3dGeoCal
* @param  cal                       kMp3dGeoCal calibration object.
* @param  cameraIndex               Camera index. kERROR_PARAMETER if invalid.
* @param  z                         Input range
* @return                           X center at given Z. k64F_NULL if cal contains no X center to Z fit.
*/
kVsFx(k64f) kMp3dGeoCalUtilities_ApplyXCenterFit(kMp3dGeoCal cal, kSize cameraIndex, k64f z);

/**
* Applies X center to Z fit from GeoCal object on buffer of Z data to obtain buffer of X center.
*
* @public                           @memberof kMp3dGeoCal
* @param  cal                       kMp3dGeoCal calibration object.
* @param  cameraIndex               Camera index. kERROR_PARAMETER if invalid.
* @param  z                         Data buffer of Z ranges
* @param  xCenter                   Output buffer of X centers
* @param  count                     Count of Z data.
* @return                           Operation status.
*/
kVsFx(kStatus) kMp3dGeoCalUtilities_ApplyXCenterFitN(kMp3dGeoCal cal, kSize cameraIndex, const k64f* z, k64f* xCenter, kSize count);

/**
* Applies Y center to Z fit from GeoCal object to obtain Y center at given Z.
* The world Y center (mm) of an imager varies with the range if any pitch is present. It also represents the distance
* from the sensor reference 0 plane. The Y center at a given Z range (mm) can be determined by:
*
*       Y center = coefficients[0] + coefficients[1]*Z + ... + coefficients[N]*Z^N
*
* @public                           @memberof kMp3dGeoCal
* @param  cal                       kMp3dGeoCal calibration object.
* @param  cameraIndex               Camera index. kERROR_PARAMETER if invalid.
* @param  z                         Input range
* @return                           Y center at given Z. k64F_NULL if cal contains no Y center to Z fit.
*/
kVsFx(k64f) kMp3dGeoCalUtilities_ApplyYCenterFit(kMp3dGeoCal cal, kSize cameraIndex, k64f z);

/**
* Applies Y center to Z fit from GeoCal object on buffer of Z data to obtain buffer of Y center.
*
* @public                           @memberof kMp3dGeoCal
* @param  cal                       kMp3dGeoCal calibration object.
* @param  cameraIndex               Camera index. kERROR_PARAMETER if invalid.
* @param  z                         Data buffer of Z ranges
* @param  yCenter                   Output buffer of Y centers
* @param  count                     Count of Z data.
* @return                           Operation status.
*/
kVsFx(kStatus) kMp3dGeoCalUtilities_ApplyYCenterFitN(kMp3dGeoCal cal, kSize cameraIndex, const k64f* z, k64f* yCenter, kSize count);

// Private functions

k64f kMp3dGeoCalUtilities_Apply(k64f* coeffs, kSize coeffCount, k64f x);
kStatus kMp3dGeoCalUtilities_ApplyN(k64f* coeffs, kSize coeffCount, const k64f* x, k64f* f, kSize count);

#endif /* LMITECH_KVISION_MP3D_GEOCALUTILITIES_H_INCLUDED */
