

#include <kVision/Vs/kVsDef.h>
#include <kApi/kApi.h>
#include <kVision/L3d/kL3dUtilities.h>
#include <kApi/Io/kFile.h>
#include <kVision/L3d/kL3dTransform3d.h>
#include <math.h>

/**
* Converts a heightmap to an array of kPoint3d64f which contain the x,y, and z values for every point.  
* Requires the resolution in all 3 dimensions (in mm).  If the original surface comes from a surface message, these can be obtained use the function calls 
* GoSurfaceMsg_XResolution, GoSurfaceMsg_YResolution, and GoSurfaceMsg_ZResolution respectively.  
* Note that the z values in a height map are not in mm.  They are integer multiples of the z resolution.  
*
* @public               @memberof kL3dSystemAlignBarUtilities
* @param   source       The source heightmap
* @param   destination  The pre-allocated destination array.  This needs to have the same dimensions as the source, but contain kPoint3d64f structures instead.
* @param   resolution   The x,y, and z resolution.  All 3 are required for this function to work.
* @param   yOffset      The y offset to apply to all points in the converted surface.  For many cases this will be the encoder resolution multiplied by the encoder y value.
* @return               Operation status.
*/
kStatus kL3dSystemAlignBarUtilities_ConvertToSurface(kArray2 source, kArray2 destination, const kPoint3d64f resolution, k64s encoderY, k64f encoderResolution, k64f xOffset);

/**
* Converts a surface of x,y, and z values to a heightmap.
* Note that the z values in a height map are not in mm, they are integer multiples of the z resolution.
* Requires the z resolution (in mm) to convert the z values into this format.
*
* @public               @memberof kL3dSystemAlignBarUtilities
* @param   source       The source heightmap
* @param   destination  The pre-allocated destination array.  This needs to have the same dimensions as the source, but consist of k16s values instead.
* @param   resolution   Thez resolution.  This is required to convert the format of the z values.
* @return               Operation status.
*/
kStatus kL3dSystemAlignBarUtilities_ConvertToHeightmap(kArray2 source, kArray2 destination, k64f zResolution);

/**
* Converts a surface of x,y, and z values to a heightmap.
* Similar to the ConvertToHeightmap function except that the units will be in mm.
*
* @public               @memberof kL3dSystemAlignBarUtilities
* @param   source       The source heightmap
* @param   destination  The pre-allocated destination array.  This needs to have the same dimensions as the source, but consist of k64f values instead.
* @param   resolution   The x,y, and z resolution.  All 3 are required for this function to work.
* @return               Operation status.
*/
kStatus kL3dSystemAlignBarUtilities_ConvertToDistanceHeightmap(kArray2 source, kArray2 destination);

/**
* Converts a surface of x,y, and z values to a kImage.
* The values will be the same as a standard heightmap.  That is, in integer multiples of the z resolution.
*
* @public               @memberof kL3dSystemAlignBarUtilities
* @param   source       The source heightmap
* @param   destination  The pre-allocated kImage structure.  This needs to have the same dimensions as the source, but consist of k16s values instead.
* @param   zResolution  The z resolution.  This is required to convert the format of the z values.
* @param   zOffset      Return value - The z offset that was applied to avoid an overflow/underflow.
* @return               Operation status.
*/
kStatus kL3dSystemAlignBarUtilities_SurfaceToKImage(kArray2 source, kImage destination, k64f zResolution, k64f *zOffset);

/**
* Converts a surface of x,y, and z values to disk in CSV format.
* The values will be the same as a standard heightmap.  That is, in integer multiples of the z resolution.
*
* @public               @memberof kL3dSystemAlignBarUtilities
* @param   source       The source heightmap
* @param   resolution   The x,y, and z resolution.  All 3 are required for this function to work.
* @param   offsets      The x,y, and z offsets.  
* @param   fileName     The fileName of the csv file to save.
* @param   includeHeader If true, the header information for the CSV converter will be included.  If false, only the raw CSV values will be included.
* @return               Operation status.
*/
kStatus kL3dSystemAlignBarUtilities_SurfaceToCSV(kArray2 surface, kPoint3d64f resolution, const kChar *fileName, kBool includeHeader);

/**
* Creates a transformation matrix for with the given rotations and translations.  Currently it ignores the rotation about the x axis as this is not measured by SystemAlignBar.
* Note that due to the skew created by a rotation about the z axis, the reverse transform is not the same as the forward transform with all values set to the negative of the forward transform.
*
* @public               @memberof kL3dSystemAlignBarUtilities
* @param   rotations    The rotations about the x, y, and z axis.  
* @param   translation  The x, y, and z translations.
* @param   forward      True will provide the forward transform, false will provide the reverse transform.  
* @output  transform    The constructed transform.
* @return               Operation status.
*/
kVsFx(kStatus) kL3dSystemAlignUtilities_CreateAlignTransform(kPoint3d64f rotations, kPoint3d64f translations, kBool forward, kL3dTransform3d *transform);

/**
* Adds a correction to the z rotation and the x and y translations.  It calculates the average z rotation of all the sensors and assumes that this is the rotation of the bar.
* 
*
* @public               @memberof kL3dSystemAlignBarUtilities
* @param   rotations    The rotations about the x, y, and z axis.
* @return               Operation status.
*/
kVsFx(kStatus) kL3dSystemAlignBarUtilities_CorrectTransforms(kArrayList rotations, kArrayList translations);
