#ifndef LMITECH_KVISION_L3D_SURFACECOMBINER_H_INCLUDED
#define LMITECH_KVISION_L3D_SURFACECOMBINER_H_INCLUDED

#include <kApi/kApi.h>
#include <kVision/L3d/kL3dCommon.h>
#include <kVision/L3d/kL3dTransform3d.h>
#include <kVision/Vs/kVsDef.h>

/**
* @class       kL3dSurfaceCombiner
* @extends     kObject
* @ingroup     kVision-L3d
* @brief       Class object for the kL3dSurfaceCombiner class.
*/
typedef kObject kL3dSurfaceCombiner;


/**
* Constructs a kP3dCombinedSurfaces object.
*
* @public               @memberof kL3dSurfaceCombiner
* @param   object       Destination for the constructed object handle.
* @param   alloc        Memory allocator (or kNULL for default).
* @return               Operation status.
*/
kVsFx(kStatus) kL3dSurfaceCombiner_Construct(kL3dSurfaceCombiner* object, kAlloc alloc);

/**
* Destroys a kP3dSurfaceCombiner object.
*
* @public               @memberof kL3dSurfaceCombiner
* @param   object       Destination for the constructed object handle.
* @return               Operation status.
*/
kVsFx(kStatus) kL3dSurfaceCombiner_Destroy(kL3dSurfaceCombiner object);

/**
* Combines the surfaces together.  The combined surfaces will have the transforms applied.  Must have at least one surface added via AddSurface.
*
* @public                     @memberof kL3dSurfaceCombiner
* @return                     Operation status.
*/
kVsFx(kStatus) kL3dSurfaceCombiner_Stitch(kL3dSurfaceCombiner object);


//Input parameters

/**
* Input.  This sets the resolution of the encoder.  This will be constant for all sensors.  This is required to convert the heightmap into world coordinates.
* The encoder resolution multiplied by the encoder index of the first row of the surface will give you the world coordinates of the start of the surface (prior to the transform).
*
* @public                     @memberof kL3dSurfaceCombiner
* @parma   object             Handle to the surfaceCombiner object.
* @param   encoderResolution  The resolution of the encoder, in mm.
* @return                     Operation status.
*/
kVsFx(kStatus) kL3dSurfaceCombiner_SetEncoderResolution(kL3dSurfaceCombiner object, k64f encoderResolution);

/**
* Call this once for each surface.  It adds that surface into the list of surfaces that will be combined.
* Note that this uses the original surface and the transform calculated from GoSystemCalibration.
* The combined surface will have the transforms applied from each sensor.
*
* @public                     @memberof kL3dSurfaceCombiner
* @param   initialSurface     The surface message to be combined.
* @param   transform          The transform for this surface calculated from GoSystemCalibration
* @return                     Operation status.
*/
//kVsFx(kStatus) kL3dSurfaceCombiner_AddSurface(kL3dSurfaceCombiner object, kArray2 heightmap, const kL3dTransform3d forwardTransform, const kPoint3d64f resolution, k64s encoderY);
kVsFx(kStatus) kL3dSurfaceCombiner_AddSurface(kL3dSurfaceCombiner object, kArray2 heightmap, kPoint3d64f offsets, kPoint3d64f rotations, const kPoint3d64f resolution, k64s encoderY, k64f xOffset);

/**
* Optional input.  Sets the fov for the output surface.  If this is not called, it will automatically find the smallest FOV for the given surfaces.
* Warning: If the FOV does not include the entire combined surface region, the output will be cropped.  
* Any height value that is not defined will be k64F_NULL.
* The width and height cannot be 0.
*
* @public                     @memberof kL3dSurfaceCombiner
* @parma   object             Handle to the surfaceCombiner object.
* @param   fov                Rectangle describing the new field of view.
* @return                     Operation status.
*/
kVsFx(kStatus) kL3dSurfaceCombiner_SetCombinedFov(kL3dSurfaceCombiner object, kRect64f fov);

/**
* Returns the combined surface.  Must call GoCombineSurface_Stitch before calling this function.
* The resulting surface is a kArray2 where each element is a kPoint3d4f.
*
* @public                     @memberof kL3dSurfaceCombiner
* @param   surface            The surface to be returned.
* @return                     Operation status.
*/
kVsFx(kStatus) kL3dSurfaceCombiner_CombinedSurface(kL3dSurfaceCombiner object, kArray2 *surface);

/**
* Returns the combined heightmap.  Must call GoCombineSurface_Stitch before calling this function.
* The resulting surface is a kArray2 where each element is a k16s.
*
* @public                     @memberof kL3dSurfaceCombiner
* @param   surface            The surface to be returned.
* @return                     Operation status.
*/
kVsFx(kStatus) kL3dSurfaceCombiner_CombinedHeightmap(kL3dSurfaceCombiner object, kArray2 *combinedHeightmap);

/**
* Returns the field of view of the heightmap.
*
* @public                     @memberof kL3dSurfaceCombiner
* @param   surface            The surface to be returned.
* @return                     Operation status.
*/
kVsFx(kRect64f) kL3dSurfaceCombiner_HeightmapFov(kL3dSurfaceCombiner object);

/**
* Returns the number of surfaces that have been added to the class.
*
* @public                     @memberof kL3dSurfaceCombiner
* @return                     The number of surfaces added.
*/
kVsFx(k32u) kL3dSurfaceCombiner_SurfaceCount(kL3dSurfaceCombiner object);

/**
* Saves the combined surface to disk in CSV format.
*
* @public                     @memberof kL3dSurfaceCombiner
* @return                     The number of surfaces added.
*/
kVsFx(kStatus) kL3dSurfaceCombiner_SaveCsv(kL3dSurfaceCombiner object, const kChar *fileName);

/**
* Testing only.
*
* @public                     @memberof kL3dSurfaceCombiner
* @return                     Operation status.
*/
kVsFx(kStatus) kL3dSurfaceCombiner_SurfaceToHeightMapTest(kL3dSurfaceCombiner object, kArray2 surface, kArray2 *heightMap, kPoint64f resolution, kPoint64f dimensions, kL3dTransform3d forwardTransform);

#include <kVision/L3d/kL3dSurfaceCombiner.x.h>

#endif
