/**
* @file    kG3dPolynomial.h
* @brief   Declares the kG3dPolynomial class. 
*
* @internal
* Copyright (C) 2016-2022 by LMI Technologies Inc. All rights reserved.
*/

#ifndef KVISION_kG3D_POLYNOMIAL_H
#define KVISION_kG3D_POLYNOMIAL_H

#include <kVision/G3d/kG3dCommon.h>

/**
* @class       kG3dPolynomial
* @extends     kObject
* @ingroup     kMC-Filtering
* @brief       Algorithm class provides some utility functions for polynomial object. 
*              The polynomial order is limited to between 1 and 8.
*/
typedef kObject kG3dPolynomial;

/**
* Constructs a kG3dPolynomial object
*
* @public              @memberof kG3dPolynomial
* @param   polynomial  Destination for the constructed object handle.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kG3dPolynomial_Construct(kG3dPolynomial* polynomial, kAlloc allocator);

/**
* Get the polynomial value at x and y: z(x,y) = sum (a * x^i * y ^j)
*
* @public              @memberof kG3dPolynomial
* @param   polynomial  kG3dPolynomial object
* @param   x           Coordinate x.
* @param   y           Coordinate y.
* @return              Polynomial value at x and y.
*/
kVsFx(k64f) kG3dPolynomial_Value(kG3dPolynomial polynomial, k64f x, k64f y);

/**
* Get the polynomial value at x and y with coefficient a(0) == 0.
*
* @public              @memberof kG3dPolynomial
* @param   polynomial  kG3dPolynomial object
* @param   x           Coordinate x.
* @param   y           Coordinate y.
* @return              Polynomial value at x and y.
*/
kVsFx(k64f) kG3dPolynomial_ValueZero(kG3dPolynomial polynomial, k64f x, k64f y);

/**
* Get the partial derivative value of x at x and y: dz(x,y) / dx.
*
* @public              @memberof kG3dPolynomial
* @param   polynomial  kG3dPolynomial object
* @param   x           Coordinate x. 
* @param   y           Coordinate y.
* @return              Partial derivative value.
*/
kVsFx(k64f) kG3dPolynomial_DxValue(kG3dPolynomial polynomial, k64f x, k64f y);

/**
* Get the first coefficient a(0) of the polynomial.
*
* @public              @memberof kG3dPolynomial
* @param   polynomial  kG3dPolynomial object
* @return              First coefficient.
*/
kVsFx(k64f) kG3dPolynomial_Offset(kG3dPolynomial polynomial);

/**
* Set the first coefficient a(0) of the polynomial.
*
* @public              @memberof kG3dPolynomial
* @param   polynomial  kG3dPolynomial object
* @param   offset      Value to be set for the first coefficient a(0)
* @return              Operation status.
*/
kVsFx(kStatus) kG3dPolynomial_SetOffset(kG3dPolynomial polynomial, k64f offset);

/**
* Get the order of the polynomial.
*
* @public              @memberof kG3dPolynomial
* @param   polynomial  kG3dPolynomial object
* @return              Order of the polynomial.
*/
kVsFx(k16s) kG3dPolynomial_Order(kG3dPolynomial polynomial);

/**
* Set the order of the polynomial.
*
* @public              @memberof kG3dPolynomial
* @param   polynomial  kG3dPolynomial object
* @param   order       Order of the polynomial
* @return              Operation status.
*/
kVsFx(kStatus) kG3dPolynomial_SetOrder(kG3dPolynomial polynomial, k16s order);


/**
* Rescale the coefficients.
*
* @public              @memberof kG3dPolynomial
* @param   polynomial  kG3dPolynomial object
* @param   factor      Scaling factor.
* @return              Operation status.
*/
kVsFx(kStatus) kG3dPolynomial_ApplyScale(kG3dPolynomial polynomial, k64f factor);

/**
* Get the coefficient of the polynomial at the given position: a(index).
*
* @public              @memberof kG3dPolynomial
* @param   polynomial  kG3dPolynomial object.
* @param   Index       Position index.
* @return              Coefficient of the polynomial at the given position.
*/
kVsFx(k64f) kG3dPolynomial_CoefficientItem(kG3dPolynomial polynomial, k32s Index);

/**
* Set all coefficient of the polynomial to zero.
*
* @public              @memberof kG3dPolynomial
* @param   polynomial  Destination for the constructed object handle.
* @return              Operation status.
*/
kVsFx(kStatus) kG3dPolynomial_Zero(kG3dPolynomial polynomial);

/**
* Checking whether all coefficients of the polynomial are zero.
*
* @public              @memberof kG3dPolynomial
* @param   polynomial  kG3dPolynomial object.
* @return              Operation status.
*/
kVsFx(kBool) kG3dPolynomial_IsZero(kG3dPolynomial polynomial);

/**
* Copy contents of a polynomial
*
* @public              @memberof kG3dPolynomial
* @param   polynomial  Destination kG3dPolynomial object. 
* @param   source      Source kG3dPolynomial object.
* @return              Operation status.
*/
kVsFx(kStatus) kG3dPolynomial_Assign(kG3dPolynomial polynomial, kG3dPolynomial source);

/**
* Get polynomial coefficents
*
* @public              @memberof kG3dPolynomial
* @param   polynomial  kG3dPolynomial object.
* @return              Polynomial coefficents.
*/
kVsFx(k64f*) kG3dPolynomial_Coefficients(kG3dPolynomial polynomial);

/**
* Replace all the coefficients in the class
*
* @public              @memberof kG3dPolynomial
* @param   polynomial  kG3dPolynomial object.
* @param   coeffs      Handle of k64f array. All the coefficients in the class 
*                      will be replaced by the array.
* @return              Operation status.
*/
kVsFx(kStatus) kG3dPolynomial_SetCoefficients(kG3dPolynomial polynomial, const k64f* coeffs);

/**
* Add all coefficients of two polynomials
*
* @public              @memberof kG3dPolynomial
* @param   outPolyn    kG3dPolynomial object. outPolyn += addPolyn
* @param   addPolyn    Polynomial to be added
* @return              Operation status.
*/
kVsFx(kStatus) kG3dPolynomial_Add(kG3dPolynomial outPolyn, kG3dPolynomial addPolyn);

/**
* Add an offset to a coefficient at given position
*
* @public              @memberof kG3dPolynomial
* @param   polynomial  kG3dPolynomial object. a[Index] += offset
* @param   index       Coefficient position.  
* @param   offset      Value to be added
* @return              Operation status.
*/
kVsFx(kStatus) kG3dPolynomial_AddItem(kG3dPolynomial polynomial, k32s index, k64f offset);

#include <kVision/G3d/kG3dPolynomial.x.h>

#endif /* KVISION_kG3D_POLYNOMIAL_H */
