/**
* @file    kG3dContourFitLine.h
* @brief   Declares the kG3dContourFitLine class. 
*
* @internal
* Copyright (C) 2016-2022 by LMI Technologies Inc. All rights reserved.
*/

#ifndef KVISION_kG3D_CONTOURFITLINE_H
#define KVISION_kG3D_CONTOURFITLINE_H

#include <kVision/Common/kVision.h>
#include <kVision/S3d/kS3dCommon.h>
#include <kVision/G3d/kG3dContourTemplate.h>
#include <kVision/G3d/kG3dContourHistogram.h>

/**
* @class       kG3dContourFitLine
* @extends     kObject
* @ingroup     kVision-G3d
* @brief       Class for calculation of straight lines from the candidate point groups
*              by using Hough transform.
*/
typedef kObject kG3dContourFitLine;

/**
* Constructs a kG3dContourFitLine object
*
* @public              @memberof kG3dContourFitLine
* @param   fitLine     Destination for the constructed object handle.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kG3dContourFitLine_Construct(kG3dContourFitLine* fitLine, kAlloc allocator);

/**
* Performs the detection, sorting and calculation of straight lines from the candidate point 
* groupes with the best fit method.
*
* @public                 @memberof kG3dContourFitLine
* @param   fitLine        kG3dContourFitLine object.
* @param   contourPoints  candidate point groups for the determination of straight lines
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourFitLine_Run(kG3dContourFitLine fitLine, kArray1 contourPoints);

/**
* Set the number of the input candidate point groups
*
* @public                 @memberof kG3dContourFitLine
* @param   fitLine        kG3dContourFitLine object.
* @param   count          Number of the input candidate point groups.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourFitLine_SetLineCount(kG3dContourFitLine fitLine, kSize count);

/**
* Set the scaling coeffizients of the corresponding height map.
*
* @public                 @memberof kG3dContourFitLine
* @param   fitLine        kG3dContourFitLine object.
* @param   scale          scaling coeffizients of the height map in the 3 directions.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourFitLine_SetScale(kG3dContourFitLine fitLine, kPoint3d64f scale);

/**
* Set the width and height of the corresponding height map.
*
* @public                 @memberof kG3dContourFitLine
* @param   fitLine        kG3dContourFitLine object.
* @param   imageSizeX     Width of the height map.
* @param   imageSizeY     Height of the height map.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourFitLine_SetImageSize(kG3dContourFitLine fitLine, kSize imageSizeX, kSize imageSizeY);

/**
* Set data buffer for the output of calculated corner points
*
* @public                 @memberof kG3dContourFitLine
* @param   fitLine        kG3dContourFitLine object.
* @param   points         Data buffer for the output of calculated corner points
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourFitLine_BorderInterestingPoints(kG3dContourFitLine fitLine, kPoint3d64f* points);

/**
* Set a boolean value to determine, whether only border points should be involved in the calculation
*
* @public                 @memberof kG3dContourFitLine
* @param   fitLine        kG3dContourFitLine object.
* @param   borderOnly     Only border points should be used?
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourFitLine_SetBorderLineOnly(kG3dContourFitLine fitLine, kBool borderOnly);

/**
* Set a boolean value to determine, whether only not used points should be involved in the calculation
* Some points for example belongs to a circle or a closed contour should not be used in this case.
*
* @public                 @memberof kG3dContourFitLine
* @param   fitLine        kG3dContourFitLine object.
* @param   notUsedOnly    Only not used points should be used?
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourFitLine_SetNotUsedPointsOnly(kG3dContourFitLine fitLine, kBool notUsedOnly);

/**
* Set the number of additional interesting points along the long side of the border contour
*
* @public                 @memberof kG3dContourFitLine
* @param   fitLine        kG3dContourFitLine object.
* @param   count          Number of  additional interesting points.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourFitLine_SetLongSidePointCount(kG3dContourFitLine fitLine, kSize count);

/**
* Set the number of additional interesting points along the short side of the border contour
*
* @public                 @memberof kG3dContourFitLine
* @param   fitLine        kG3dContourFitLine object.
* @param   count          Number of additional interesting points.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourFitLine_SetShortSidePointCount(kG3dContourFitLine fitLine, kSize count);

/**
* Set a boolean value to determine, whether only the points close to center should
* be used for the determination of the geometry of short side straight line because of round corner.
*
* @public                 @memberof kG3dContourFitLine
* @param   fitLine        kG3dContourFitLine object.
* @param   used           Only the points close to center should be used?
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourFitLine_SetShortSideCenterAlgin(kG3dContourFitLine fitLine, kBool used);

/**
* Set a boolean value to determine, whether only the points close to center should
* be used for the determination of the geometry of long side straight line because of round corner.
*
* @public                 @memberof kG3dContourFitLine
* @param   fitLine        kG3dContourFitLine object.
* @param   used           Only the points close to center should be used?
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourFitLine_SetLongSideCenterAlgin(kG3dContourFitLine fitLine, kBool used);

/**
* Set a boolean value to determine, whether the short side of the border contour should be corrected
* perpendicular to the long side contour.
*
* @public                 @memberof kG3dContourFitLine
* @param   fitLine        kG3dContourFitLine object.
* @param   used           Short side contour should be corrected?
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourFitLine_SetShortSideRect(kG3dContourFitLine fitLine, kBool used);

#include <kVision/G3d/kG3dContourFitLine.x.h>

#endif  /* KVISION_kG3D_CONTOURFITLINE_H */
