/**
* @file    kG3dContourClosedLine.h
* @brief   Declares the kG3dContourClosedLine class. 
*
* @internal
* Copyright (C) 2016-2022 by LMI Technologies Inc. All rights reserved.
*/

#ifndef KVISION_kG3D_CONTOURCLOSEDLINE_H
#define KVISION_kG3D_CONTOURCLOSEDLINE_H

#include <kVision/Common/kVision.h>
#include <kVision/S3d/kS3dCommon.h>
#include <kVision/G3d/kG3dContourSegment.h>
#include <kVision/G3d/kG3dContourTemplate.h>

/**
* @class       kG3dContourClosedLine
* @extends     kObject
* @ingroup     kVision-G3d
* @brief       Class for the detection, eliminating, merging, sorting and smoothing the closed 
*              contours from the candidate point groups.
*/
typedef kObject kG3dContourClosedLine;

/**
* Constructs a kG3dContourClosedLine object
*
* @public               @memberof kG3dContourClosedLine
* @param   closedLine   Destination for the constructed object handle.
* @param   allocator    Memory allocator (or kNULL for default).
* @return               Operation status.
*/
kVsFx(kStatus) kG3dContourClosedLine_Construct(kG3dContourClosedLine* closedLine, kAlloc allocator);

/**
* Performs the evaluation of the detecting, sorting and smoothing the closed contours from the candidate point groupes
*
* @public              @memberof kG3dContourClosedLine
* @param   closedLine  Object of kG3dContourClosedLine.
* @param   index       Index of the output contour
* @return              Number of detected closed contours
*/
kVsFx(kStatus) kG3dContourClosedLine_Run(kG3dContourClosedLine closedLine, kArray1 contourPoints);

/**
* Set the data buffer for the final output contours
*
* @public              @memberof kG3dContourClosedLine
* @param   closedLine  Object of kG3dContourClosedLine.
* @param   profiles    Data buffer for the final output contours.
* @return              Operation status.
*/
kVsFx(kStatus) kG3dContourClosedLine_SetProfiles(kG3dContourClosedLine closedLine, kArrayList profiles);

/**
* Get the number of detected closed contours
*
* @public              @memberof kG3dContourClosedLine
* @param   closedLine  Object of kG3dContourClosedLine.
* @param   index       Index of the output contour
* @return              Number of detected closed contours
*/
kVsFx(kSize)   kG3dContourClosedLine_ContourCount(kG3dContourClosedLine closedLine);

/**
* Get an output structure including information about the detected closed contour for a given index  
*
* @public              @memberof kG3dContourClosedLine
* @param   closedLine  Object of kG3dContourClosedLine.
* @param   index       Index of the output contour
* @return              Output structure.
*/
kVsFx(kG3dClosedContourType*)   kG3dContourClosedLine_Contour(kG3dContourClosedLine closedLine, k32s index);

/**
* Set the number of the input candidate point groups
*
* @public              @memberof kG3dContourClosedLine
* @param   closedLine  Object of kG3dContourClosedLine.
* @param   count       Number of the input candidate point groupes
* @return              Operation status.
*/
kVsFx(kStatus) kG3dContourClosedLine_SetLineCount(kG3dContourClosedLine closedLine, kSize count);

/**
* Set the scaling coeffizients of the corresponding height map.
*
* @public              @memberof kG3dContourClosedLine
* @param   closedLine  Object of kG3dContourClosedLine.
* @param   scale       scaling coeffizients of the height map in the 3 directions.
* @return              Operation status.
*/
kVsFx(kStatus) kG3dContourClosedLine_SetScale(kG3dContourClosedLine closedLine, kPoint3d64f scale);

/**
* Set the width and height of the corresponding height map.
*
* @public              @memberof kG3dContourClosedLine
* @param   closedLine  Object of kG3dContourClosedLine.
* @param   imageSizeX  Width of the height map.
* @param   imageSizeY  Height of the height map.
* @return              Operation status.
*/
kVsFx(kStatus) kG3dContourClosedLine_SetImageSize(kG3dContourClosedLine closedLine, kSize imageSizeX, kSize imageSizeY);

/**
* Set histogram range. The value can be changed depending on the balance between 
* accuracy requirement and speed requirement.
*
* @public              @memberof kG3dContourClosedLine
* @param   closedLine  Object of kG3dContourClosedLine.
* @param   size        Histogram range
* @return              Operation status.
*/
kVsFx(kStatus) kG3dContourClosedLine_SetHistogramSize(kG3dContourClosedLine closedLine, kSize size);

/**
* Set sample index of a measurement series.This function is only used for debug purpose to store
* the intermediate results without overwriting.
*
* @public              @memberof kG3dContourClosedLine
* @param   closedLine  Object of kG3dContourClosedLine.
* @param   index       Sample index of a measurement series
* @return              Operation status.
*/
kVsFx(kStatus) kG3dContourClosedLine_SetSampleIndex(kG3dContourClosedLine closedLine, kSize index);

/**
* Set repeat index of a measurement series.This function is only used for debug purpose to store
* the intermediate results without overwriting.
*
* @public              @memberof kG3dContourClosedLine
* @param   closedLine  Object of kG3dContourClosedLine.
* @param   index       Repeat index of a measurement series
* @return              Operation status.
*/
kVsFx(kStatus) kG3dContourClosedLine_SetRepeatIndex(kG3dContourClosedLine closedLine, kSize index);


#include <kVision/G3d/kG3dContourClosedLine.x.h>

#endif  /* KVISION_kG3D_CONTOURCLOSEDLINE_H */
