/**
* @file    kG3dAffineTransform.h
* @brief   Declares the kG3dAffineTransform class. 
*
* @internal
* Copyright (C) 2016-2022 by LMI Technologies Inc. All rights reserved.
*/

#ifndef KVISION_kG3D_AFFINETRANSFORM_H
#define KVISION_kG3D_AFFINETRANSFORM_H

#include <kVision/Common/kVision.h>
#include <kVision/S3d/kS3dCommon.h>
#include <kVision/L3d/kL3dTransform3d.h>
#include <kVision/S3d/kS3dVolumeCheckUtilities.h>

typedef enum kG3dAffineTransformModeType
{
    kG3D_TRANSFORM2D_AFFINE = 0,
    kG3D_TRANSFORM2D_SCALING,
    kG3D_TRANSFORM2D_RIGID
} kG3dAffineTransformModeType;

/**
* @class       kG3dAffineTransform
* @extends     kObject
* @ingroup     kVision-S3d
* @brief       Creates an affine transformation matrix using given point pairs.
*              This class is mainly used for 2D perspective correction.
*              The perspective correction should be possible through an affine transformation. 
*              For this, it is necessary to detect at least 4 reliable point pairs.
*              The following two properties of the transformation are used in the algorithms:
*
*                 A straight line stays straight line
*                 A circle is turned into an ellipse
*
*              The affine transformation based on a semi perspective projection
*
*              x_out = a11 * x_in + a12 * y_in + a13 * x_in * y_in + a14;
*              y_out = a21 * x_in + a22 * y_in + a23 * x_in * y_in + a24;
*
*              For mode kG3D_TRANSFORM2D_SCALING: the coefficients a13 = a23 = 0;
*              
*              For mode  kG3D_TRANSFORM2D_RIGID
*
*              x_out =  (a11 * x_in + a12 * y_in + a13 * z_in + a14) / Q;
*              y_out =  (a21 * x_in + a22 * y_in + a23 * z_in + a24) / Q;
*              Q     =  (a31 * x_in + a32 * y_in + a33 * z_in + a34) / (-K);
*            
*              K is the camera constant
*
*/
typedef kObject kG3dAffineTransform;

/**
* Constructs a kG3dAffineTransform object
*
* @public               @memberof kG3dAffineTransform
* @param   affineTrans  Destination for the constructed object handle.
* @param   allocator    Memory allocator (or kNULL for default).
* @return               Operation status.
*/
kVsFx(kStatus) kG3dAffineTransform_Construct(kG3dAffineTransform* affineTrans, kAlloc allocator);


/**
* Set transform mode defined in kG3dAffineTransformModeType
*
* @public               @memberof kG3dAffineTransform
* @param   affineTrans  kG3dAffineTransform object
* @param   mode         transform mode defined in kG3dAffineTransformModeType
* @return               Operation status.
*/
kVsFx(kStatus) kG3dAffineTransform_SetMode(kG3dAffineTransform affineTrans, kG3dAffineTransformModeType mode);

/**
* Create a affine transformation matrixfrom the given 2d point pairs
*
* @public                 @memberof kG3dAffineTransform
* @param   affineTrans    kG3dAffineTransform object
* @param   refPoints      kArray1 < kPoint3d64f >  reference points
* @param   measurePoints  kArray1 < kPoint3d64f >  points should be transformed.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dAffineTransform_Run(kG3dAffineTransform affineTrans, kArray1 refPoints, kArray1 measurePoints, kL3dTransform3d* T, k64f tol);

#include <kVision/G3d/kG3dAffineTransform.x.h>

#endif  /* KVISION_kS3D_EMPTY_H */
