#ifndef LMITECH_kBOOTSTRAPFIT_H_INCLUDED
#define LMITECH_kBOOTSTRAPFIT_H_INCLUDED

#include <kVision/Common/kPolynomial.h>
#include <kVision/Common/kPolynomialFit.h>

typedef kObject kBootstrapFit; 


// IMPORTANT NOTE: PolynomialOrder < ceil(DataPointCount*0.63)  For example for a 3 order polynomial you'll need at least 6 points because ceil (6 * 0.63) = 4

/**
* @brief   Constructs a kBootstrapFit object.
*
* @public                       @memberof kBootstrapFit
* @param   fit                  Pointer to the object handle.
* @param   polynomialOrder      Order of the polynomial fit.
* @param   fitCount             Number of random fits to the data. The class will fit this number of different polynomials to the input data.
* @param   allocator            Memory allocator (or kNULL for default).
* @return                       Operation status.
*/
kVsFx(kStatus) kBootstrapFit_Construct(kBootstrapFit* fit, k32s polynomialOrder, kSize fitCount, kBool correlatedMode,kObject allocator);

/**
* @brief    Performs a series of polynomial fits to randomnly resampled versions of the input data. The resulting polynomials are stored internally within the class.
*
* @public               @memberof kBootstrapFit
* @param   fit          The kBootstrapFit object handle
* @param   x            Pointer to the start of input data block
* @param   f            Pointer to the start of desired output data block
* @param   count        Data point count
* @return               Operation status.
*/
kVsFx(kStatus) kBootstrapFit_Apply(kBootstrapFit fit, const k64f* x, const k64f* f, kSize count);

/**
* @brief    Returns the number of random fits as set by the constructor.
*
* @public               @memberof kBootstrapFit
* @param   fit          The kBootstrapFit object handle
* @return               The number of random fits to the data.
*/
kVsFx(kSize) kBootstrapFit_FitCount(kBootstrapFit fit);

/**
* @brief    Returns the number of polynomials fitted to randomnly resampled versions of the input data.
*
* @public               @memberof kBootstrapFit
* @param   fit          The kBootstrapFit object handle
* @return               The count.
*/
kVsFx(kSize) kBootstrapFit_PolynomialCount(kBootstrapFit fit);

/**
* @brief    Returns the ith polynomial fitted to a randomnly resampled version of the input data.
*
* @public               @memberof kBootstrapFit
* @param   fit          The kBootstrapFit object handle
* @param   i            Polynomial index
* @return               The polynomial handle.
*/
kVsFx(kPolynomial) kBootstrapFit_PolynomialAt(kBootstrapFit fit, kSize i);

/**
* @brief    Evaluates all the fitted polynomials at x and stores the results in the outBuffer
*
* @public               @memberof kBootstrapFit
* @param   fit          The kBootstrapFit object handle
* @param   x            The point at which the fitted polynomials will be evaluated
* @param   outBuffer    Pointer to output buffer
* @param   bufferSize   Output buffer size
* @return               Operation status.
*/
kVsFx(kStatus) kBootstrapFit_Evaluate(kBootstrapFit fit, k64f x, const k64f* outBuffer, const kSize bufferSize);

/**
* @brief    Evaluates all the fitted polynomials at x and outputs the value for one percentile, median, and ninty nine percentile of the evaluations 
*
* @public                   @memberof kBootstrapFit
* @param   fit              The kBootstrapFit object handle
* @param   x                The point at which the fitted polynomials will be evaluated
* @param   onePerc          Pointer to where 1 percentile of the evaluations will be stored.
* @param   median           Pointer to where median of the evaluations will be stored.
* @param   nintyNinePerc    Pointer to where 99 percentile of the evaluations will be stored.
* @return                   Operation status.
*/
kVsFx(kStatus) kBootstrapFit_EvaluateConfidenceLevels(kBootstrapFit fit, k64f x, k64f* onePerc, k64f* median, k64f* nintyNinePerc);

kVsFx(kStatus) kBootstrapFit_SetOrderPenalty(kBootstrapFit fit, k64f orderPenalty);
kVsFx(kStatus) kBootstrapFit_SetTargetError(kBootstrapFit fit, k64f targetError);
kVsFx(k32s) kBootstrapFit_Order(kBootstrapFit fit);
kVsFx(k64f) kBootstrapFit_OrderPenalty(kBootstrapFit fit);
kVsFx(k64f) kBootstrapFit_TargetError(kBootstrapFit fit);

#include <kVision/Common/kBootstrapFit.x.h>

#endif
