/** 
 * @file    kFsDef.h
 * @brief   Essential API declarations for the kFireSync library.
 *
 * @internal
 * Copyright (C) 2012-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_FIRESYNC_DEF_H
#define K_FIRESYNC_FIRESYNC_DEF_H

#include <kApi/Io/kNetwork.h>

#if defined (K_FS_EMIT)
#   define kFsFx(TYPE)    kExportFx(TYPE)               ///< kFireSync function declaration helper. 
#   define kFsDx(TYPE)    kExportDx(TYPE)               ///< kFireSync data declaration helper. 
#else
#   define kFsFx(TYPE)    kImportFx(TYPE)    
#   define kFsDx(TYPE)    kImportDx(TYPE)    
#endif

/**
* @struct  kPair32s
* @extends kValue
* @ingroup kFireSync
* @brief   Represents a pair of 32-bit values.
* 
* This type is not currently serializable; this support can be addded in future, if needed.
*/
typedef struct kPair32s
{
    k32s first;     ///< First value.
    k32s second;    ///< Second value.
} kPair32s;

/**
 * @struct  kEthernetLink
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents an Ethernet link speed. 
 */
typedef k32s kEthernetLink; 

/** @relates kEthernetLink @{ */
#define kETHERNET_LINK_NONE                     (0x0000)        ///< No link, or speed unknown. 
#define kETHERNET_LINK_10_HALF                  (0x0001)        ///< 10 Mb/s, half duplex. 
#define kETHERNET_LINK_10_FULL                  (0x0002)        ///< 10 Mb/s, full duplex. 
#define kETHERNET_LINK_100_HALF                 (0x0004)        ///< 100 Mb/s, half-duplex. 
#define kETHERNET_LINK_100_FULL                 (0x0008)        ///< 100 Mb/s, full-duplex. 
#define kETHERNET_LINK_1000_HALF                (0x0010)        ///< 1000 Mb/s, half-duplex. 
#define kETHERNET_LINK_1000_FULL                (0x0020)        ///< 1000 Mb/s, full-duplex. 
#define kETHERNET_LINK_AUTO                     (0x8000)        ///< Auto-negotiated. 
/** @} */

/** 
 * Formats a kEthernetLink value to a string buffer. 
 *
 * @public              @memberof kEthernetLink
 * @param   speed       Speed. 
 * @param   text        Receives formatted string. 
 * @param   capacity    Buffer capacity. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kEthernetLink_Format(kEthernetLink speed, kChar* text, kSize capacity); 

/** 
 * Based on the link speed, reports the maximum send/receive throughput (bytes/second). 
 *
 * @public              @memberof kEthernetLink
 * @param   speed       Speed. 
 * @return              Maximum throughput, in bytes/second.
 */
kFsFx(k64u) kEthernetLink_Capacity(kEthernetLink speed); 

/**
 * @struct  kUpgLogLevel
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents Upgrade Log Level. 
 */
typedef k32s kUpgLogLevel; 

/** @relates kUpgLogLevel @{ */
#define kFS_UPG_LOG_LEVEL_INFO               (0)           ///< Log level providing information. 
#define kFS_UPG_LOG_LEVEL_WARN               (1)           ///< Log level providing a warning. 
#define kFS_UPG_LOG_LEVEL_ERROR              (2)           ///< Log level providing an error. 

/**
 * @struct  kUpgLogMsg
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents an upgrade log message. 
 */
typedef struct kUpgLogMsg
{
    kChar buffer[512];
    kUpgLogLevel logLevel;
} kUpgLogMsg;

/**
* @struct  kNetworkSpeed
* @extends kValue
* @ingroup kFireSync
* @brief   Represents a network speed
*/
typedef k32s kNetworkSpeed;

/** @relates kNetworkSpeed @{ */
#define kNETWORK_SPEED_NULL                 (0x0000)        ///< Unknown. 
#define kNETWORK_SPEED_AUTO                 (0x0001)        ///< Auto negotiation. 
#define kNETWORK_SPEED_10                   (0x0002)        ///< 10 Mb/s. 
#define kNETWORK_SPEED_100                  (0x0004)        ///< 100 Mb/s. 
#define kNETWORK_SPEED_1000                 (0x0008)        ///< 1000 Mb/s. 
/** @} */

/**
* Formats a kNetworkSpeed value to a string buffer.
*
* @public              @memberof kNetworkSpeed
* @param   speed       Speed.
* @param   text        Receives formatted string.
* @param   capacity    Buffer capacity.
* @return              Operation status.
*/
kFsFx(kStatus) kNetworkSpeed_Format(kNetworkSpeed speed, kChar* text, kSize capacity);

/**
 * @struct  kIpNegotiation
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents IP auto configuration options. 
 */
typedef k32s kIpNegotiation; 

/** @relates kIpNegotiation @{ */
#define kIP_NEGOTIATION_STATIC                  (0)           ///< Static IP configuration. 
#define kIP_NEGOTIATION_DHCP                    (1)           ///< DHCP, using 169.254.x.x/16 as fallback (aka Zero Configuration). 

/** @} */

/** 
 * Formats a kIpNegotiation value to a string buffer. 
 *
 * @public                  @memberof kIpNegotiation
 * @param   ipNegotiation   IP auto configuration setting. 
 * @param   text            Receives formatted string. 
 * @param   capacity        Buffer capacity. 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kIpNegotiation_Format(kIpNegotiation ipNegotiation, kChar* text, kSize capacity);

/**
 * @struct  kIpConfig
 * @extends kValue
 * @ingroup kFireSync 
 * @brief   Represents address configuration for an Internet Protocol adapter. 
 */
typedef struct kIpConfig
{
    kIpAddress address;             ///< IP address. 
    k32u prefixLength;              ///< Subnet prefix length. 
    kIpAddress gateway;             ///< Default gateway address. 
    kIpNegotiation ipNegotiation;   ///< IP auto configuration setting.
} kIpConfig; 

/** 
 * Converts a numeric subnet prefix length to an IPv4 subnet mask. 
 *
 * @public          @memberof kIpConfig
 * @param   prefix  Prefix length. 
 * @return          IPv4 Subnet mask. 
 */
kFsFx(k32u) kIpConfig_SubnetPrefixToMask(k32u prefix); 

/** 
 * Converts an IPv4 subnet mask to a numeric subnet prefix length. 
 *
 * @public          @memberof kIpConfig
 * @param   mask    IPv4 Subnet mask.
 * @return          Prefix length. 
 */
kFsFx(k32u) kIpConfig_SubnetMaskToPrefix(k32u mask); 

/**
* @struct  kRegisterBits
* @extends kValue
* @ingroup kFireSync
* @brief   Represents address/bits data register pair.
*/
typedef struct kRegisterBits
{
    k16u address;       ///< Address value.
    k16u startBit;      ///< Start bit.
    k16u endBit;        ///< End bit.
    k16u data;          ///< Data value.
} kRegisterBits;

/**
 * @struct  kRegister16u
 * @extends kValue
 * @ingroup kFireSync 
 * @brief   Represents 16-bit address/data register pair. 
 */
typedef struct kRegister16u
{
    k16u address;       ///< Address value.
    k16u data;          ///< Data value.
} kRegister16u; 

/**
* @struct  kRegister32u
* @extends kValue
* @ingroup kFireSync
* @brief   Represents 32-bit address/data register pair.
*/
typedef struct kRegister32u
{
    k32u address;       ///< Address value.
    k32u data;          ///< Data value.
} kRegister32u;

/**
 * @struct  kVisualizerType
 * @extends kValue
 * @ingroup kFireSync-Data
 * @brief   Represents an enumeration of data visualization types.
 */
typedef k32s kVisualizerType;

/** @relates kVisualizerType @{ */
#define kVISUALIZER_MESH                (0)     ///< Accepts kTerrainModel. 
#define kVISUALIZER_PROPERTY_TABLE      (1)     ///< Accepts kArrayList of kArrayList of kStrings, first item in string indicates whether it's for the name field or the value field ("name" or "value"). 
#define kVISUALIZER_TEXT                (2)     ///< Accepts kString.
#define kVISUALIZER_2D                  (3)     ///< Accepts kImage, kArray2. 
/** @} */

/**
 * @struct  kAlignment
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents alignment within a 3x3 grid. 
 */
typedef k32s kAlignment; 

/** @relates kAlignment @{ */
#define kALIGNMENT_CENTRE               (0x00)                                 ///< Centre alignment. 
#define kALIGNMENT_LEFT                 (0x01)                                 ///< Left alignment. 
#define kALIGNMENT_RIGHT                (0x02)                                 ///< Right alignment. 
#define kALIGNMENT_TOP                  (0x04)                                 ///< Top alignment. 
#define kALIGNMENT_BOTTOM               (0x08)                                 ///< Bottom alignment. 
#define kALIGNMENT_TOP_LEFT             (kALIGNMENT_TOP | kALIGNMENT_LEFT)     ///< Top left alignment. 
#define kALIGNMENT_TOP_RIGHT            (kALIGNMENT_TOP | kALIGNMENT_RIGHT)    ///< Top right alignment. 
#define kALIGNMENT_BOTTOM_LEFT          (kALIGNMENT_BOTTOM | kALIGNMENT_LEFT)  ///< Bottom left alignment. 
#define kALIGNMENT_BOTTOM_RIGHT         (kALIGNMENT_BOTTOM | kALIGNMENT_RIGHT) ///< Bottom right alignment. 
/** @} */

/**
 * @struct  kFontStyle
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents font style values.  
 */
typedef k32s kFontStyle; 

/** @relates kFontStyle @{ */
#define kFONT_STYLE_NORMAL              (0x00)     ///< Normal font. 
#define kFONT_STYLE_BOLD                (0x01)     ///< Bold font.
#define kFONT_STYLE_ITALIC              (0x02)     ///< Italicized font.
#define kFONT_STYLE_UNDERLINE           (0x04)     ///< Underlined font.
/** @} */

/**
 * @struct  kFontFamily
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents font family values. 
 */
typedef k32s kFontFamily;

/** @relates kFontFamily @{ */
#define kFONT_FAMILY_SERIF              (0)         ///< Serif font family. 
#define kFONT_FAMILY_SANS_SERIF         (1)         ///< Sans serif font family. 
#define kFONT_FAMILY_MONOSPACE          (2)         ///< Monospace font family. 
/** @} */

/**
* @struct  kMarkerShape
* @extends kValue
* @ingroup kFireSync
* @brief   Graphic marker shape enumeration.
*/
typedef k32s kMarkerShape; 

/** @relates kMarkerShape @{ */
#define kMARKER_SHAPE_CROSS             (0)        ///<  Cross marker shape. 
#define kMARKER_SHAPE_PLUS              (1)        ///<  Plus marker shape. 
#define kMARKER_SHAPE_CIRCLE            (2)        ///<  Circle marker shape. 
#define kMARKER_SHAPE_SQUARE            (3)        ///<  Square marker shape. 
#define kMARKER_SHAPE_TRIANGLE          (4)        ///<  Triangle marker shape. 
#define kMARKER_SHAPE_DIAMOND           (5)        ///<  Diamond marker shape. 
/** @} */

/**
* @struct  kDemosaicStyle
* @extends kValue
* @ingroup kFireSync
* @brief   Bayer demosaic algorithm style.
*/
typedef k32s kDemosaicStyle;

#define kDEMOSAIC_STYLE_REDUCE        (0)           ///< Use direct sampling to generate one color pixel for each 2x2 bayer mask group. 
#define kDEMOSAIC_STYLE_BILINEAR      (1)           ///< Use binlinear interpolation to generate one color pixel for each greyscale pixel.
#define kDEMOSAIC_STYLE_GRADIENT      (2)           ///< Use gradient-aware algorithm to generate one color pixel for each greyscale pixel. 

/**
* @struct  kAxis
* @extends kValue
* @ingroup kFireSync
* @brief   Coordinate axis dimension.
*/

typedef k32s kAxis; 

#define kAXIS_X     (0x1)           ///< X Axis.
#define kAXIS_Y     (0x2)           ///< Y Axis.
#define kAXIS_Z     (0x4)           ///< Z Axis. 

/**
 * @class   kCompressedPhaseStreamId
 * @extends kValue
 * @ingroup kFireSync-Data
 * @brief   Stream type identifier for a kCompressedPhase stream.
 *
 * Note: enumerators are currently assigned to match PL implementation; exercise caution if changing. 
 */
typedef k32s kCompressedPhaseStreamId; 

/** @relates kCompressedPhaseStreamId @{ */
#define kCOMPRESSED_PHASE_STREAM_ID_PHASE_DELTA          (0)          ///< Phase delta values. 
#define kCOMPRESSED_PHASE_STREAM_ID_PHASE_FAIL_2         (1)          ///< Phase "fail 2nd" index/value pairs.
#define kCOMPRESSED_PHASE_STREAM_ID_PHASE_NULL           (2)          ///< Phase null index/count pairs.
#define kCOMPRESSED_PHASE_STREAM_ID_PHASE_FAIL_1         (3)          ///< Phase "fail 1st" index/value pairs.
#define kCOMPRESSED_PHASE_STREAM_ID_INTENSITY_DELTA      (4)          ///< Intensity delta values. 
#define kCOMPRESSED_PHASE_STREAM_ID_INTENSITY_FAIL_2     (5)          ///< Intensity "fail 2nd" index/value pairs.
#define kCOMPRESSED_PHASE_STREAM_ID_INTENSITY_NULL       (6)          ///< Intensity null index/count pairs.
#define kCOMPRESSED_PHASE_STREAM_ID_INTENSITY_FAIL_1     (7)          ///< Intensity "fail 1st" index/value pairs (note: unused at time of writing).
/** @} */

/** 
 * Reports whether the specified stream identifier represents a delta stream.
 * 
 * @public              @memberof kCompressedPhaseStreamId
 * @param   streamId    Stream identifier.  
 * @return              kTRUE if the stream identifier represents a delta stream.
 */
kInlineFx(kBool) kCompressedPhaseStreamId_IsDelta(kCompressedPhaseStreamId streamId)
{
    switch (streamId)
    {
        case kCOMPRESSED_PHASE_STREAM_ID_PHASE_DELTA:
        case kCOMPRESSED_PHASE_STREAM_ID_INTENSITY_DELTA:
            return kTRUE; 
        default:
            return kFALSE;
    }
}

/** 
 * Reports whether the specified stream identifier represents a "fail" stream.
 * 
 * @public              @memberof kCompressedPhaseStreamId
 * @param   streamId    Stream identifier.  
 * @return              kTRUE if the stream identifier represents a "fail" stream.
 */
kInlineFx(kBool) kCompressedPhaseStreamId_IsFail(kCompressedPhaseStreamId streamId)
{
    switch (streamId)
    {
        case kCOMPRESSED_PHASE_STREAM_ID_PHASE_FAIL_1:
        case kCOMPRESSED_PHASE_STREAM_ID_PHASE_FAIL_2:
        case kCOMPRESSED_PHASE_STREAM_ID_INTENSITY_FAIL_1:
        case kCOMPRESSED_PHASE_STREAM_ID_INTENSITY_FAIL_2:
            return kTRUE; 
        default:
            return kFALSE;
    }
}

/** 
 * Reports whether the specified stream identifier represents a null stream.
 * 
 * @public              @memberof kCompressedPhaseStreamId
 * @param   streamId    Stream identifier.  
 * @return              kTRUE if the stream identifier represents a null stream.
 */
kInlineFx(kBool) kCompressedPhaseStreamId_IsNull(kCompressedPhaseStreamId streamId)
{
    switch (streamId)
    {
        case kCOMPRESSED_PHASE_STREAM_ID_PHASE_NULL:
        case kCOMPRESSED_PHASE_STREAM_ID_INTENSITY_NULL:
            return kTRUE; 
        default:
            return kFALSE;
    }
}

/** 
 * Reports whether the specified stream identifier represents a phase stream.
 * 
 * @public              @memberof kCompressedPhaseStreamId
 * @param   streamId    Stream identifier.  
 * @return              kTRUE if the stream identifier represents a phase stream.
 */
kInlineFx(kBool) kCompressedPhaseStreamId_IsPhase(kCompressedPhaseStreamId streamId)
{
    return streamId < kCOMPRESSED_PHASE_STREAM_ID_INTENSITY_DELTA;
}

/**
 * @class   kCompressedPhaseEncoding
 * @extends kValue
 * @ingroup kFireSync-Data
 * @brief   Represents an encoding style for a kCompressedPhase stream.
 */
typedef k32s kCompressedPhaseEncoding; 

/** @relates kCompressedPhaseEncoding @{ */
#define kCOMPRESSED_PHASE_ENCODING_4B_DELTA              (1)          ///< 4-bit signed delta encoding.
#define kCOMPRESSED_PHASE_ENCODING_6B_DELTA              (2)          ///< 6-bit signed delta encoding.
#define kCOMPRESSED_PHASE_ENCODING_8B_DELTA              (3)          ///< 8-bit signed delta encoding.
#define kCOMPRESSED_PHASE_ENCODING_10B_DELTA             (4)          ///< 10-bit signed delta encoding.
#define kCOMPRESSED_PHASE_ENCODING_12B_FAIL_1            (5)          ///< 8-bit unsigned index + 12-bit signed "fail 1st" encoding.  
#define kCOMPRESSED_PHASE_ENCODING_8B_FAIL_2             (6)          ///< 8-bit unsigned index + 8-bit unsigned "fail 2nd" encoding.  
#define kCOMPRESSED_PHASE_ENCODING_24B_FAIL_2            (7)          ///< 8-bit unsigned index + 24-bit unsigned "fail 2nd" encoding.  
#define kCOMPRESSED_PHASE_ENCODING_8B_NULL               (8)          ///< 8-bit unsigned index + 8-bit unsigned null count  encoding.  
/** @} */

/** 
 * Reports the bit depth of the index field associated with the encoding (if applicable). 
 * 
 * Index fields are always unsigned. Delta encodings do not have an index field.
 *
 * @public              @memberof kCompressedPhaseEncoding
 * @param   encoding    Encoding style.  
 * @return              Index field bit depth. 
 */
kFsFx(k32u) kCompressedPhaseEncoding_IndexBitDepth(kCompressedPhaseEncoding encoding); 

/** 
 * Reports the maximum index associated with the encoding (if applicable). 
 * 
 * This method reports the minimum numeric value of the 'index' field. Delta encodings do 
 * not have an index field.
 * 
 * @public              @memberof kCompressedPhaseEncoding
 * @param   encoding    Encoding style.  
 * @return              Maximum index. 
 */
kFsFx(k32s) kCompressedPhaseEncoding_IndexMax(kCompressedPhaseEncoding encoding); 

/** 
 * Reports the bit depth of the value field associated with the encoding (if applicable). 
 * 
 * The method reports the bit depth of detla, value, or count fields, depending on the encoding
 * style. The bit depth includes the sign bit, if applicable. 
 *
 * @public              @memberof kCompressedPhaseEncoding
 * @param   encoding    Encoding style.  
 * @return              Value field bit depth. 
 */
kFsFx(k32u) kCompressedPhaseEncoding_ValueBitDepth(kCompressedPhaseEncoding encoding); 

/** 
 * Reports whether the value field associated with the encoding is signed. 
 * 
 * @public              @memberof kCompressedPhaseEncoding
 * @param   encoding    Encoding style.  
 * @return              kTRUE if signed; kFALSE if unsigned. 
 */
kFsFx(k32s) kCompressedPhaseEncoding_ValueSigned(kCompressedPhaseEncoding encoding); 

/** 
 * Reports the minimum value associated with the encoding. 
 * 
 * This method reports the minimum numeric value of the 'value' field (delta, value, or count)
 * associated with the encoding.
 * 
 * @public              @memberof kCompressedPhaseEncoding
 * @param   encoding    Encoding style.  
 * @return              Minimum value. 
 */
kFsFx(k32s) kCompressedPhaseEncoding_ValueMin(kCompressedPhaseEncoding encoding); 

/** 
 * Reports the maximum value associated with the encoding. 
 * 
 * This method reports the maximum numeric value of the 'value' field (delta, value, or count)
 * associated with the encoding.
 * 
 * @public              @memberof kCompressedPhaseEncoding
 * @param   encoding    Encoding style.  
 * @return              Maximum value. 
 */
kFsFx(k32s) kCompressedPhaseEncoding_ValueMax(kCompressedPhaseEncoding encoding); 

/** 
 * Looks up the delta encoding identifier for the specified bit depth.
 * 
 * @public              @memberof kCompressedPhaseEncoding
 * @param   encoding    Receives encoding style.  
 * @param   bitDepth    Desired bit depth.
 * @return              Operation status (kERROR_NOT_FOUND if not supported).
 */
kFsFx(kStatus) kCompressedPhaseEncoding_FindDeltaEncoding(kCompressedPhaseEncoding* encoding, k32u bitDepth); 

/**
* @struct  kField
* @ingroup kFireSync
* @brief   Collection of static methods and macros for bit field manipulation.
*/

/** @relates  kField @{ */

/** 
 * Creates a bit mask from a 32-bit range. 
 * 
 * WARNING: this macro can generate masks only as wide as the number of bits in the 
 * integral type of the MSB operand, minus one.  E.g., if the MSB operand is of type 
 * k32u, this macro can generate a mask of up 31 bits (i.e., 0xFFFFFFFE or 0x7FFFFFFF). 
 * This limitation is due to undefined behavior in the C language for bits shifts 
 * greater than or equal to the width of an integral type. 
 *
 * @param   MSB   Leftmost bit index in range (highest).  
 * @param   LSB   Rightmost bit index in range (lowest). 
 * @return  Bit mask. 
 */
#define kField_ToMask_(MSB, LSB)      \
  (((((MSB) - (MSB) + 1) << ((MSB) - (LSB) + 1)) - 1) << (LSB))

/** 
 * Creates a bitfield by applying the specified mask and shift values.
 *
 * @param   MASK        Mask value (applied after shift). 
 * @param   SHIFT       Left shift value (applied before mask). 
 * @param   FLD_VAL     Field value.  
 * @return              Composed bitfield. 
 */
#define kField_Create_(MASK, SHIFT, FLD_VAL)                        \
    (((FLD_VAL) << SHIFT) & (MASK))

/** 
 * Creates a bitfield by token-pasting mask/shift macros using a specific naming scheme.
 *
 * @param   FIELD_NAME  Macro prefix, prepended to _MASK and _SHIFT. 
 * @param   FLD_VAL     Field value.  
 * @return              Composed bitfield. 
 */
#define kField_CreateNamed_(FIELD_NAME, FLD_VAL)                        \
    kField_Create_(FIELD_NAME##_MASK, FIELD_NAME##_SHIFT, FLD_VAL)

/** 
 * Creates a bitfield within the specified bit range. 
 * 
 * (MSB - LSB) must be <em>strictly</em> less than the bit width of the type of MSB. See remarks
 * in kField_ToMask_ documentation. 
 *
 * @param   MSB         Leftmost bit index in range (highest).  
 * @param   LSB         Rightmost bit index in range (lowest). 
 * @param   FLD_VAL     Field value.  
 * @return              Composed bitfield. 
 */
#define kField_CreateRange_(MSB, LSB, FLD_VAL)                \
    kField_Create_(kField_ToMask_(MSB, LSB), LSB, FLD_VAL)

/** 
 * Creates a bitfield by token-pasting left/right macros using a specific naming scheme.
 *
 * (MSB - LSB) must be <em>strictly</em> less than the bit width of the type of MSB. See remarks
 * in kField_ToMask_ documentation. 
 *
 * @param   FIELD_NAME  Macro prefix, prepended to _MSB and _LSB.
 * @param   FLD_VAL     Field value.  
 * @return              Composed bitfield. 
 */
#define kField_CreateNamedRange_(FIELD_NAME, FLD_VAL)                \
    kField_CreateRange_(FIELD_NAME##_MSB, FIELD_NAME##_LSB, FLD_VAL)

/** 
 * Extracts a bitfield by applying the specified mask and shift values.
 *
 * @param   REG_VAL     Register value.
 * @param   MASK        Mask value (applied before shift). 
 * @param   SHIFT       Shift value (applied after mask). 
 * @return              Value extracted from bitfield. 
 */
#define kField_Extract_(REG_VAL, MASK, SHIFT)               \
    (((REG_VAL) & (MASK)) >> (SHIFT))

/** 
 * Extracts a bitfield by token-pasting mask/shift macros using a specific naming scheme.
 *
 * @param   REG_VAL     Register value.
 * @param   FIELD_NAME  Macro prefix, prepended to _MASK and _SHIFT. 
 * @return              Value extracted from bitfield. 
 */
#define kField_ExtractNamed_(REG_VAL, FIELD_NAME)               \
    (((REG_VAL) & (FIELD_NAME##_MASK)) >> (FIELD_NAME##_SHIFT))

/** 
 * Extracts a bitfield from the specified bit range. 
 *
 * (MSB - LSB) must be <em>strictly</em> less than the bit width of the type of MSB. See remarks
 * in kField_ToMask_ documentation. 
 *
 * @param   REG_VAL     Register value.
 * @param   MSB         Leftmost bit index in range (highest).  
 * @param   LSB         Rightmost bit index in range (lowest). 
 * @return              Value extracted from bitfield. 
 */
#define kField_ExtractRange_(REG_VAL, MSB, LSB)               \
    kField_Extract_(REG_VAL, kField_ToMask_(MSB, LSB), LSB)

/** 
 * Extracts a bitfield by token-pasting left/right macros using a specific naming scheme.
 *
 * (MSB - LSB) must be <em>strictly</em> less than the bit width of the type of MSB. See remarks
 * in kField_ToMask_ documentation. 
 *
 * @param   REG_VAL     Register value.
 * @param   FIELD_NAME  Macro prefix, prepended to _MSB and _LSB.
 * @return              Value extracted from bitfield. 
 */
#define kField_ExtractNamedRange_(REG_VAL, FIELD_NAME)               \
    kField_ExtractRange_(REG_VAL, FIELD_NAME##_MSB, FIELD_NAME##_LSB)

/** 
 * Inserts a bitfield into an existing register value.
 *
 * @param   REG_REF     Pointer to register value.
 * @param   MASK        Bitfield mask value. 
 * @param   SHIFT       Bitfield shift value. 
 * @param   FLD_VAL     Field value.  
 */
#define kField_Insert_(REG_REF, MASK, SHIFT, FLD_VAL)       \
    (*(REG_REF) = (*(REG_REF) & ~(MASK)) | kField_Create_(MASK, SHIFT, FLD_VAL))

/** 
 * Inserts a bitfield into an existing register value by token-pasting mask/shift macros using a specific naming scheme.
 *
 * @param   REG_REF     Pointer to register value.
 * @param   FIELD_NAME  Macro prefix, prepended to _MASK and _SHIFT. 
 * @param   FLD_VAL     Field value.  
 */
#define kField_InsertNamed_(REG_REF, FIELD_NAME, FLD_VAL)       \
    kField_Insert_(REG_REF, FIELD_NAME##_MASK, FIELD_NAME##_SHIFT, FLD_VAL)

/** 
 * Inserts a bitfield into an existing register value.
 *
 * (MSB - LSB) must be <em>strictly</em> less than the bit width of the type of MSB. See remarks
 * in kField_ToMask_ documentation. 
 *
 * @param   REG_REF     Pointer to register value.
 * @param   MSB         Leftmost bit index in range (highest).  
 * @param   LSB         Rightmost bit index in range (lowest). 
 * @param   FLD_VAL     Field value.  
 */
#define kField_InsertRange_(REG_REF, MSB, LSB, FLD_VAL)       \
    kField_Insert_(REG_REF, kField_ToMask_(MSB, LSB), LSB, FLD_VAL)

/** 
 * Inserts a bitfield into an existing register value by token-pasting left/right macros using a specific naming scheme.
 *
 * (MSB - LSB) must be <em>strictly</em> less than the bit width of the type of MSB. See remarks
 * in kField_ToMask_ documentation. 
 *
 * @param   REG_REF     Pointer to register value.
 * @param   FIELD_NAME  Macro prefix, prepended to _MSB and _LSB.
 * @param   FLD_VAL     Field value.  
 */
#define kField_InsertNamedRange_(REG_REF, FIELD_NAME, FLD_VAL)       \
    kField_InsertRange_(REG_REF, FIELD_NAME##_MSB, FIELD_NAME##_LSB, FLD_VAL)

/** 
 * Takes the difference between register fields at the same location within two register values.
 *
 * @param   REG_VAL_A   Register value A.
 * @param   REG_VAL_B   Register value B.
 * @param   MASK        Mask value (applied before shift). 
 * @param   SHIFT       Shift value (applied after mask). 
 * @return              Value extracted from bitfield. 
 */
#define kField_Diff_(REG_VAL_A, REG_VAL_B, MASK, SHIFT)                                        \
        ((kField_Extract_(REG_VAL_A, MASK, SHIFT) - kField_Extract_(REG_VAL_B, MASK, SHIFT))   \
            & ((MASK) >> (SHIFT)))

/** 
 * Takes the difference between register fields at the same location within two register values by 
 * token-pasting mask/shift macros using a specific naming scheme.
 *
 * @param   REG_VAL_A   Register value A.
 * @param   REG_VAL_B   Register value B.
 * @param   FIELD_NAME  Macro prefix, prepended to _MASK and _SHIFT. 
 * @return              Value extracted from bitfield. 
 */
#define kField_DiffNamed_(REG_VAL_A, REG_VAL_B, FIELD_NAME)             \
        kField_Diff_(REG_VAL_A, REG_VAL_B, FIELD_NAME##_MASK, FIELD_NAME##_SHIFT)

/** 
 * Takes the difference between register fields at the same location within two register values.
 *
 * (MSB - LSB) must be <em>strictly</em> less than the bit width of the type of MSB. See remarks
 * in kField_ToMask_ documentation. 
 *
 * @param   REG_VAL_A   Register value A.
 * @param   REG_VAL_B   Register value B.
 * @param   MSB         Leftmost bit index in range (highest).  
 * @param   LSB         Rightmost bit index in range (lowest). 
 * @return              Value extracted from bitfield. 
 */
#define kField_DiffRange_(REG_VAL_A, REG_VAL_B, MSB, LSB)            \
    kField_Diff_(REG_VAL_A, REG_VAL_B, kField_ToMask_(MSB, LSB), LSB)

/** 
 * Takes the difference between register fields at the same location within two register values by 
 * token-pasting left/right macros using a specific naming scheme.
 *
 * (MSB - LSB) must be <em>strictly</em> less than the bit width of the type of MSB. See remarks
 * in kField_ToMask_ documentation. 
 *
 * @param   REG_VAL_A   Register value A.
 * @param   REG_VAL_B   Register value B.
 * @param   FIELD_NAME  Macro prefix, prepended to _MSB and _LSB.
 * @return              Value extracted from bitfield. 
 */
#define kField_DiffNamedRange_(REG_VAL_A, REG_VAL_B, FIELD_NAME)        \
    kField_DiffRange_(REG_VAL_A, REG_VAL_B, FIELD_NAME##_MSB, FIELD_NAME##_LSB)

/** @} */


/** 
 * Converts integer nano-scaled values to micro-scaled values (/1000). 
 *
 * @param   V       Nano-scaled value.
 * @return          Micro-scaled value.
 */
#define kNanoToMicro_(V)                ((V) / 1000)

/** 
 * Converts integer micro-scaled values to nano-scaled values (x1000). 
 *
 * @param   V       Micro-scaled value.
 * @return          Nano-scaled value.
 */
#define kMicroToNano_(V)                ((V) * 1000)


/** @relates kIoTest @{ */
#define kIO_TEST_IO_I2C_ID_EXT_AD7415           (0x49)
#define kIO_TEST_AD7415_SCALE_FACTOR            (250);
/** @} */

#include <kFireSync/kFsDef.x.h>

#endif
