#ifndef K_LD_MODEL_H
#define K_LD_MODEL_H

#include <kFireSync/kFsDef.h>
#include <kFireSync/kNodeDef.h>

typedef enum
{
    kLD_MODEL_TEMPERATURE_REFERENCE_NONE       = 0,
    kLD_MODEL_TEMPERATURE_REFERENCE_INTERNAL   = 1,
    kLD_MODEL_TEMPERATURE_REFERENCE_EXTERNAL   = 2,
    kLD_MODEL_TEMPERATURE_REFERENCE_CPU        = 3
} kLdModelTempReference;

/**
* kLdModel class is intended to facilitate behaviour configuration of the low level FireSync software with respect to programmable laser drivers, 
* with the high level objective of accurately and safely programming and then maintaining laser power and current limit over the full operating 
* temperature range. 
* 
* Currently, this objective breaks down to the following set of configuration properties: 
* - Should optical power and/or limit should be configured by the application
* - Operating optical power range of the driver
* - Operating temperature range of the driver
* - Temperature tolerance against a stable reference indicator (e.g. controller) to be used for validation of driver probe function
* - If required, temperature compensation coefficients for the power set point (intended for ACC diodes/drivers)
* - If required, temperature compensation coefficients for the power current limit (intended for ACC and APC diodes/drivers with programmable current limits)
*
* @class       kLdModel
* @extends     kObject
* @ingroup     kFireSync-Utils
* @brief       Class model for the kLdModel class.
*/
//typedef kObject kLdModel;            --forward-declared in kFsDef.x.h  

/**
* Generates the default file name based on the index of the laser driver. 
*
* @public               @memberof kLdModel
* @param   index        Driver index
* @param   fileName     Text buffer receiving object file name
* @param   capacity     Capacity of the output text buffer
* @return               Operation status.
*/
kFsFx(kStatus) kLdModel_FileName(kSize index, kChar* fileName, kSize capacity);

/**
* Constructs a kLdModel model. 
* 
* Failure to construct the object with an error code of kERROR_UNIMPLEMENTED, indicates that the 
* driver runtime behaviour has not been characterized. In this case, application level optical 
* power programming should not be allowed and no runtime adjustment is needed. 
*
* Note that it is very unlikely that the sensor with driver that has not fully been characterized
* will include a power calibration record _required_ for any runtime adjustment.
* 
* @public                   @memberof kLdModel
* @param   model            Destination for the constructed model handle.
* @param   lightModel       Target light model.
* @param   diodeModel       Target light diode model.
* @param   tolerateUnknown  Should this constructor tolerate an unknown light model?
* @param   alloc            Memory allocator (or kNULL for default).
* @return                   Operation status.
*/
kFsFx(kStatus) kLdModel_Construct(kLdModel* model, kLightModel lightModel, kLightDiodeModel diodeModel, kBool tolerateUnknown, kAlloc alloc);

/**
* Loads the model object from local storage at the specified path
*
* @public               @memberof kLdCal
* @param   model        The class object.
* @param   path         The file location to store the resulting output.
* @param   alloc        Memory allocator (or kNULL for default).
* @return               Operation status.
*/
kFsFx(kStatus) kLdModel_Load(kLdModel *model, const kChar *path, kAlloc alloc);

/**
* Loads the model object from a stream
*
* @public               @memberof kLdCal
* @param   model        The class object.
* @param   stream       Input stream object
* @param   alloc        Memory allocator (or kNULL for default).
* @return               Operation status.
*/
kFsFx(kStatus) kLdModel_LoadStream(kLdModel *model, kStream stream, kAlloc alloc);

/**
* Returns the light model enumerator corresponding to this object
*
* @public               @memberof kLdCal
* @param   model        The class object.
* @return               Light model enumerator
*/
kFsFx(kLightModel) kLdModel_LightModel(kLdModel model);

/**
* Returns the diode model enumerator corresponding to this object.
*
* @public               @memberof kLdModel
* @param   model        LdModel object.
* @return               Diode model enumerator. 
*/
kFsFx(kLightDiodeModel) kLdModel_DiodeModel(kLdModel model);

/**
* Specifies whether power set point is configurable for the queried laser driver model. True for all programmable laser driver modules
*
* @public               @memberof kLdCal
* @param   model        The class object.
* @return               Flag indicating whether power set point is configurable for this driver
*/
kFsFx(kBool) kLdModel_PowerEnabled(kLdModel model);

/**
* Specifies if power current limit is configurable for the queried laser driver model
*
* @public               @memberof kLdCal
* @param   model        The class object.
* @return               Flag indicating whether power current limit is configurable for this driver
*/
kFsFx(kBool) kLdModel_LimitEnabled(kLdModel model);

/**
* Specifies maximum optical power range (in mW), which is safely configurable for this laser driver. Any optical power 
* values configured by the application API must adhere to these limits to avoid failures to lase (lower limit) and damage
* to the diode (upper limit)
*
* @public               @memberof kLdCal
* @param   model        The class object.
* @param   outputMin    Pointer receiving lower bound of the optical power configuration range (mW). k64F_NULL is returned if no minimum is required.
* @param   outputMax    Pointer receiving upper bound of the optical power configuration range (mW). k64F_NULL is returned if no maximum is required.
* @return               Operation status
*/
kFsFx(kStatus) kLdModel_OutputRange(kLdModel model, k64f* outputMin, k64f* outputMax);

/**
* Specifies maximum operating temperature range of the driver, in degrees C. As outside of this range driver temperature compensation may
* result in failure to meet safety class, failure to lase or cause damage to the diode, the system must ensure that laser startup is prevented
* when driver temperature is outside of this range
*
* @public               @memberof kLdCal
* @param   model        The class object.
* @param   tempMin      Pointer receiving lower bound of the operating temperature range (degrees C). k64F_NULL is returned if no minimum is required.
* @param   tempMax      Pointer receiving upper bound of the operating temperature range (degrees C). k64F_NULL is returned if no maximum is required.
* @return               Operation status
*/
kFsFx(kStatus) kLdModel_TempRange(kLdModel model, k64f* tempMin, k64f* tempMax);

/**
* Specifies the temperature reference probe, which (if available) must be used to determine if the laser driver temperature probe
* is functioning correctly. 
* 
* Supported options are:
*  kLD_MODEL_TEMP_REFERENCE_NONE - no reference check required/possible
*  kLD_MODEL_TEMP_REFERENCE_INTERNAL - kHEALTH_ID_INTERNAL_TEMPERATURE
*  kLD_MODEL_TEMP_REFERENCE_EXTERNAL - kHEALTH_ID_EXTERNAL_TEMPERATURE
*  kLD_MODEL_TEMP_REFERENCE_CPU - kHEALTH_ID_MAX_CPU_TEMPERATURE
*
* @public               @memberof kLdCal
* @param   model        The class object.
* @return               Enumerator indicating temperature reference indicator
*/
kFsFx(kLdModelTempReference) kLdModel_TempReference(kLdModel model);

/**
* Specifies the maximum allowable deviation, in degrees C, between the driver temperature and the reference probe (queried via kLdModel_TempReference)
* Should the limit be exeeded, the light probe should be considered damaged. Light should be prevented from starting and the error reported to the
* user as a cause for an RMA
*
* @public               @memberof kLdCal
* @param   model        The class object.
* @return               Maximum allowable deviation from reference, in degrees C. 
*/
kFsFx(k64f) kLdModel_TempReferenceTolerance(kLdModel model);

/**
* Specifies whether power set point requires active temperature compensation. If true, the input to the set point must be
* periodically re-calculated (using kLdCalLookup) and updated to account for the power dependency w.r.t temperature.
*
* @public               @memberof kLdCal
* @param   model        The class object.
* @return               Flag specifying whether power set point requires active temperature compensation. 
*/
kFsFx(kBool) kLdModel_PowerCompensated(kLdModel model);

/**
* Returns the temperature compensation coefficient needed for accurate power set point compensation at runtime. 
* The value, should be passed onto kLdCalLookup_SetPowerCompensationInfo prior to using the lookup class
*
* @public               @memberof kLdCal
* @param   model        The class object.
* @param   coeff        Temperature compensation coefficient, measuring drift in optical power as a function of temperature (mW/Degree C)
* @return               Operation status
*/
kFsFx(kStatus) kLdModel_PowerCompensationCoeff(kLdModel model, k64f* coeff);

/**
* Specifies whether power current limit requires active temperature compensation. If true, the input to the current limit must be
* periodically re-calculated (using kLdCalLookup) and updated to account for the power dependency w.r.t temperature.
*
* @public               @memberof kLdCal
* @param   model        The class object.
* @return               Flag specifying whether power set point requires active temperature compensation.
*/
kFsFx(kBool) kLdModel_LimitCompensated(kLdModel model);

/**
* Returns the temperature compensation coefficient needed for accurate power current limit compensation at runtime.
* The value, should be passed onto kLdCalLookup_SetLimitCompensationInfo prior to using the lookup class
*
* @public               @memberof kLdCal
* @param   model        The class object.
* @param   coeff        Temperature compensation coefficient, measuring drift in optical power as a function of temperature (mW/Degree C)
* @return               Operation status
*/
kFsFx(kStatus) kLdModel_LimitCompensationCoeff(kLdModel model, k64f* coeff);

/**
* Reports the number of priming exposures that should be run at acquisition startup.
* 
* Priming exposures can be used at acquisition startup to prepare lights, providing more even illumination.
*
* @public           @memberof kLdCal
* @param   model    Light model object.
* @return           Number of priming exposures to run.
*/
kFsFx(k32u) kLdModel_PrimeCount(kLdModel model);

/**
* Reports the amount of time required for one light priming exposure.
* 
* Priming exposures can be used at acquisition startup to prepare lights, providing more even illumination.
*
* @public           @memberof kLdCal
* @param   model    Light model object.
* @return           Priming exposure time, in FS ns.
*/
kFsFx(k64u) kLdModel_PrimeExposure(kLdModel model);

#include <kFireSync/Utils/kLdModel.x.h>

#endif
