/** 
 * @file    kInfo.h
 * @brief   Declares the kInfo family of structures. 
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_INFO_H
#define K_FIRESYNC_INFO_H

#include <kFireSync/kFsDef.h>

/**
 * @struct  kViolation
 * @extends kValue
 * @ingroup kFireSync-Client-Utils
 * @brief   Represents types of constraint violations. 
 *
 * The following enumerators are defined:  
 * - #kVIOLATION_NONE
 * - #kVIOLATION_BOUNDS
 * - #kVIOLATION_GRANULARITY
 * - #kVIOLATION_UNIQUENESS
 * - #kVIOLATION_EXISTENCE
 * - #kVIOLATION_GROUP
 */
typedef k32s kViolation; 

/** @relates kViolation @{ */
#define kVIOLATION_NONE                 (0x00)     ///< No violations.
#define kVIOLATION_BOUNDS               (0x01)     ///< Value exceeds current min/max limits.
#define kVIOLATION_GRANULARITY          (0x02)     ///< Value is not evenly divisible by the specified granularity value. 
#define kVIOLATION_UNIQUENESS           (0x04)     ///< Value is equal to another value, where uniqueness is required.
#define kVIOLATION_EXISTENCE            (0x08)     ///< Value refers to an entity that does not exist. 
#define kVIOLATION_GROUP                (0x10)     ///< Peers in a group have different values, when they are constrained to be equal. 
#define kVIOLATION_INVALID              (0x20)     ///< Value is not acceptable.
/** @} */

/**
 * @struct  kAdjust
 * @extends kValue
 * @ingroup kFireSync-Client-Utils
 * @brief   Type of adjustment to perform to bring a setting into conformance with its constraints.
 */
typedef k32s kAdjust; 

/** @relates kAdjust @{ */
#define kADJUST_NONE                (0x00)                          ///< No adjustment.
#define kADJUST_RELATIVE_FLOOR      (0x01)                          ///< Adjust to relative constraints, rouding down.
#define kADJUST_RELATIVE_CEIL       (0x02)                          ///< Adjust to relative constraints, rouding up.
#define kADJUST_ABSOLUTE_FLOOR      (0x04)                          ///< Adjust to absolute constraints, rouding down.
#define kADJUST_ABSOLUTE_CEIL       (0x08)                          ///< Adjust to absolute constraints, rouding up.
#define kADJUST_AUTO                kADJUST_RELATIVE_FLOOR          ///< Automatic adjustment.
/** @} */

#include <kFireSync/Utils/kInfo.x.h>

/**
 * @struct  kInfo
 * @extends kValue
 * @ingroup kFireSync-Client-Utils
 * @brief   Represents constraint and validity information for a general setting. 
 */
typedef struct kInfo
{
    kBool isReadOnly;               ///< Value is read-only (cannot be modified). 
    kBool isGrouped;                ///< Value is constrained to be the same as its peers within a group.
    kViolation violations;          ///< Constraints violated by the current value. 
} kInfo;        

/** 
 * Initializes a kInfo structure to default values. 
 *
 * @public          @memberof kInfo
 * @param   info    Info structure. 
 * @return          Operation status.  
 */
kFsFx(kStatus) kInfo_Init(kInfo* info); 
    
/** 
 * Appends violation descriptions to the given list. 
 *
 * @public          @memberof kInfo
 * @param   info    Info structure. 
 * @param   list    Error list (kArrayList<kText128>). 
 * @param   source  Label for object containing the error. 
 * @param   field   Label for field containing the error. 
 * @return          Operation status.  
 */
kFsFx(kStatus) kInfo_AppendErrors(const kInfo* info, kArrayList list, const kChar* source, const kChar* field);

/**
 * @struct  kInfo32u
 * @extends kValue
 * @ingroup kFireSync-Client-Utils
 * @brief   Represents constraint and validity information for a 32-bit unsigned integer setting. 
 */
typedef struct kInfo32u                         
{
    k32u absoluteMin;               ///< Minimum valid value under any circumstances.
    k32u absoluteMax;               ///< Maximum valid value under any circumstances.
    k32u relativeMin;               ///< Minimum valid value, limited by current values of other (more dominant) settings.
    k32u relativeMax;               ///< Maximum valid value, limited by current values of other (more dominant) settings.
    k32u granularity;               ///< Value must be evenly divisible by this number. 
    kBool isReadOnly;               ///< Value is read-only (cannot be modified). 
    kBool isGrouped;                ///< Value is constrained to be the same as its peers within a group. (e.g. periods of states belonging to a camera might be constrained to be equal). 
    kViolation violations;          ///< Constraints violated by the current value. 
    k64f accurate;                  ///< Accurate representation of current value, typically used for display in meaningful units (optional). 
} kInfo32u;

/** 
 * Initializes a kInfo32u structure to default values. 
 *
 * @public              @memberof kInfo32u
 * @param   info        Info structure. 
 * @param   min         Minimum value. 
 * @param   max         Maximum value. 
 * @param   granularity Granularity constraint.
 * @return              Operation status.  
 */
kFsFx(kStatus) kInfo32u_Init(kInfo32u* info, k32u min, k32u max, k32u granularity); 

/** 
 * Validates an input value against min/max/granularity values.
 *
 * @public          @memberof kInfo32u
 * @param   info    Info structure. 
 * @param   value   Input value. 
 * @return          Operation status.  
 */
kFsFx(kStatus) kInfo32u_Validate(kInfo32u* info, k32u value); 

/** 
 * Adjusts the specified value based on min/max/granularity constaints.
 *
 * @public          @memberof kInfo32u
 * @param   info    Info structure. 
 * @param   value   Input value. 
 * @param   style   Adjustment style. 
 * @return          Adjusted value.
 */
kInlineFx(k32u) kInfo32u_Adjust(kInfo32u* info, k32u value, kAdjust style)
{
    return xkAdjustInfoValue(*info, value, style);
}

/** 
 * Appends violation descriptions to the given list. 
 *
 * @public          @memberof kInfo32u
 * @param   info    Info structure. 
 * @param   list    Error list (kArrayList<kText128>). 
 * @param   source  Label for object containing the error. 
 * @param   field   Label for field containing the error. 
 * @param   value   Current field value. 
 * @return          Operation status.  
 */
kFsFx(kStatus) kInfo32u_AppendErrors(const kInfo32u* info, kArrayList list, const kChar* source, const kChar* field, k32u value); 

/**
 * @struct  kInfo32s
 * @extends kValue
 * @ingroup kFireSync-Client-Utils
 * @brief   Represents constraint and validity information for a 32-bit signed integer setting. 
 */
typedef struct kInfo32s                         
{
    k32s absoluteMin;               ///< Minimum valid value under any circumstances.
    k32s absoluteMax;               ///< Maximum valid value under any circumstances.
    k32s relativeMin;               ///< Minimum valid value, limited by current values of other (more dominant) settings.
    k32s relativeMax;               ///< Maximum valid value, limited by current values of other (more dominant) settings.
    k32s granularity;               ///< Value must be evenly divisible by this number. 
    kBool isReadOnly;               ///< Value is read-only (cannot be modified). 
    kBool isGrouped;                ///< Value is constrained to be the same as its peers within a group. (e.g. periods of states belonging to a camera might be constrained to be equal). 
    kViolation violations;          ///< Constraints violated by the current value. 
    k64f accurate;                  ///< Accurate representation of current value, typically used for display in meaningful units (optional).  
} kInfo32s;

/** 
 * Initializes a kInfo32s structure to default values. 
 *
 * @public              @memberof kInfo32s
 * @param   info        Info structure. 
 * @param   min         Minimum value. 
 * @param   max         Maximum value. 
 * @param   granularity Granularity constraint.
 * @return              Operation status.  
 */
kFsFx(kStatus) kInfo32s_Init(kInfo32s* info, k32s min, k32s max, k32s granularity);            

/** 
 * Validates an input value against min/max/granularity values.
 *
 * @public          @memberof kInfo32s
 * @param   info    Info structure. 
 * @param   value   Input value. 
 * @return          Operation status.  
 */
kFsFx(kStatus) kInfo32s_Validate(kInfo32s* info, k32s value); 

/** 
 * Adjusts the specified value based on min/max/granularity constaints.
 *
 * @public          @memberof kInfo32s
 * @param   info    Info structure. 
 * @param   value   Input value. 
 * @param   style   Adjustment style. 
 * @return          Adjusted value.
 */
kInlineFx(k32s) kInfo32s_Adjust(kInfo32s* info, k32s value, kAdjust style)
{
    return xkAdjustInfoValue(*info, value, style);
}

/** 
 * Appends violation descriptions to the given list. 
 *
 * @public          @memberof kInfo32u
 * @param   info    Info structure. 
 * @param   list    Error list (kArrayList<kText128>). 
 * @param   source  Label for object containing the error. 
 * @param   field   Label for field containing the error. 
 * @param   value   Current field value. 
 * @return          Operation status.  
 */
kFsFx(kStatus) kInfo32s_AppendErrors(const kInfo32s* info, kArrayList list, const kChar* source, const kChar* field, k32s value); 

/**
 * @struct  kInfo64u
 * @extends kValue
 * @ingroup kFireSync-Client-Utils
 * @brief   Represents constraint and validity information for a 64-bit unsigned integer setting. 
 */
typedef struct kInfo64u
{
    k64u absoluteMin;               ///< Minimum valid value under any circumstances.
    k64u absoluteMax;               ///< Maximum valid value under any circumstances.
    k64u relativeMin;               ///< Minimum valid value, limited by current values of other (more dominant) settings.
    k64u relativeMax;               ///< Maximum valid value, limited by current values of other (more dominant) settings.
    k64u granularity;               ///< Value must be evenly divisible by this number. 
    kBool isReadOnly;               ///< Value is read-only (cannot be modified). 
    kBool isGrouped;                ///< Value is constrained to be the same as its peers within a group. (e.g. periods of states belonging to a camera might be constrained to be equal). 
    kViolation violations;          ///< Constraints violated by the current value. 
    k64f accurate;                  ///< Accurate representation of current value, typically used for display in meaningful units (optional). 
} kInfo64u;
                      
/** 
 * Initializes a kInfo64u structure to default values. 
 *
 * @public              @memberof kInfo64u
 * @param   info        Info structure. 
 * @param   min         Minimum value. 
 * @param   max         Maximum value. 
 * @param   granularity Granularity constraint.
 * @return              Operation status.  
 */
kFsFx(kStatus) kInfo64u_Init(kInfo64u* info, k64u min, k64u max, k64u granularity); 

/** 
 * Validates an input value against min/max/granularity values.
 *
 * @public          @memberof kInfo64u
 * @param   info    Info structure. 
 * @param   value   Input value. 
 * @return          Operation status.  
 */
kFsFx(kStatus) kInfo64u_Validate(kInfo64u* info, k64u value); 

/** 
 * Adjusts the specified value based on min/max/granularity constaints.
 *
 * @public          @memberof kInfo64u
 * @param   info    Info structure. 
 * @param   value   Input value. 
 * @param   style   Adjustment style. 
 * @return          Adjusted value.
 */
kInlineFx(k64u) kInfo64u_Adjust(kInfo64u* info, k64u value, kAdjust style)
{
    return xkAdjustInfoValue(*info, value, style);
}

/** 
 * Appends violation descriptions to the given list. 
 *
 * @public          @memberof kInfo32u
 * @param   info    Info structure. 
 * @param   list    Error list (kArrayList<kText128>). 
 * @param   source  Label for object containing the error. 
 * @param   field   Label for field containing the error. 
 * @param   value   Current field value. 
 * @return          Operation status.  
 */
kFsFx(kStatus) kInfo64u_AppendErrors(const kInfo64u* info, kArrayList list, const kChar* source, const kChar* field, k64u value); 

/**
 * @struct  kInfo64s
 * @extends kValue
 * @ingroup kFireSync-Client-Utils
 * @brief   Represents constraint and validity information for a 64-bit signed integer setting. 
 */
typedef struct kInfo64s
{
    k64s absoluteMin;               ///< Minimum valid value under any circumstances.
    k64s absoluteMax;               ///< Maximum valid value under any circumstances.
    k64s relativeMin;               ///< Minimum valid value, limited by current values of other (more dominant) settings.
    k64s relativeMax;               ///< Maximum valid value, limited by current values of other (more dominant) settings.
    k64s granularity;               ///< Value must be evenly divisible by this number. 
    kBool isReadOnly;               ///< Value is read-only (cannot be modified). 
    kBool isGrouped;                ///< Value is constrained to be the same as its peers within a group. (e.g. periods of states belonging to a camera might be constrained to be equal). 
    kViolation violations;          ///< Constraints violated by the current value. 
    k64f accurate;                  ///< Accurate representation of current value, typically used for display in meaningful units (optional).  
} kInfo64s;

/** 
 * Initializes a kInfo64s structure to default values. 
 *
 * @public              @memberof kInfo64s
 * @param   info        Info structure. 
 * @param   min         Minimum value. 
 * @param   max         Maximum value. 
 * @param   granularity Granularity constraint.
 * @return              Operation status.  
 */
kFsFx(kStatus) kInfo64s_Init(kInfo64s* info, k64s min, k64s max, k64s granularity); 

/** 
 * Validates an input value against min/max/granularity values.
 *
 * @public          @memberof kInfo64s
 * @param   info    Info structure. 
 * @param   value   Input value. 
 * @return          Operation status.  
 */
kFsFx(kStatus) kInfo64s_Validate(kInfo64s* info, k64s value); 

/** 
 * Adjusts the specified value based on min/max/granularity constaints.
 *
 * @public          @memberof kInfo64s
 * @param   info    Info structure. 
 * @param   value   Input value. 
 * @param   style   Adjustment style. 
 * @return          Adjusted value.
 */
kInlineFx(k64s) kInfo64s_Adjust(kInfo64s* info, k64s value, kAdjust style)
{
    return xkAdjustInfoValue(*info, value, style);
}

/** 
 * Appends violation descriptions to the given list. 
 *
 * @public          @memberof kInfo32u
 * @param   info    Info structure. 
 * @param   list    Error list (kArrayList<kText128>). 
 * @param   source  Label for object containing the error. 
 * @param   field   Label for field containing the error. 
 * @param   value   Current field value. 
 * @return          Operation status.  
 */
kFsFx(kStatus) kInfo64s_AppendErrors(const kInfo64s* info, kArrayList list, const kChar* source, const kChar* field, k64s value); 
             
/**
 * @struct  kInfoSize
 * @extends kValue
 * @ingroup kFireSync-Client-Utils
 * @brief   Represents constraint and validity information for a kSize setting. 
 */
typedef struct kInfoSize                         
{
    kSize absoluteMin;              ///< Minimum valid value under any circumstances.
    kSize absoluteMax;              ///< Maximum valid value under any circumstances.
    kSize relativeMin;              ///< Minimum valid value, limited by current values of other (more dominant) settings.
    kSize relativeMax;              ///< Maximum valid value, limited by current values of other (more dominant) settings.
    kSize granularity;              ///< Value must be evenly divisible by this number. 
    kBool isReadOnly;               ///< Value is read-only (cannot be modified). 
    kBool isGrouped;                ///< Value is constrained to be the same as its peers within a group. (e.g. periods of states belonging to a camera might be constrained to be equal). 
    kViolation violations;          ///< Constraints violated by the current value. 
    k64f accurate;                  ///< Accurate representation of current value, typically used for display in meaningful units (optional). 
} kInfoSize;

/** 
 * Initializes a kInfoSize structure to default values. 
 *
 * @public              @memberof kInfoSize
 * @param   info        Info structure. 
 * @param   min         Minimum value. 
 * @param   max         Maximum value. 
 * @param   granularity Granularity constraint.
 * @return              Operation status.  
 */
kFsFx(kStatus) kInfoSize_Init(kInfoSize* info, kSize min, kSize max, kSize granularity); 

/** 
 * Validates an input value against min/max/granularity values.
 *
 * @public          @memberof kInfoSize
 * @param   info    Info structure. 
 * @param   value   Input value. 
 * @return          Operation status.  
 */
kFsFx(kStatus) kInfoSize_Validate(kInfoSize* info, kSize value); 

/** 
 * Adjusts the specified value based on min/max/granularity constaints.
 *
 * @public          @memberof kInfoSize
 * @param   info    Info structure. 
 * @param   value   Input value. 
 * @param   style   Adjustment style. 
 * @return          Adjusted value.
 */
kInlineFx(kSize) kInfoSize_Adjust(kInfoSize* info, kSize value, kAdjust style)
{
    return xkAdjustInfoValue(*info, value, style);
}

/** 
 * Appends violation descriptions to the given list. 
 *
 * @public          @memberof kInfo32u
 * @param   info    Info structure. 
 * @param   list    Error list (kArrayList<kText128>). 
 * @param   source  Label for object containing the error. 
 * @param   field   Label for field containing the error. 
 * @param   value   Current field value. 
 * @return          Operation status.  
 */
kFsFx(kStatus) kInfoSize_AppendErrors(const kInfoSize* info, kArrayList list, const kChar* source, const kChar* field, kSize value); 

/**
 * @struct  kInfo64f
 * @extends kValue
 * @ingroup kFireSync-Client-Utils
 * @brief   Represents constraint and validity information for a double setting. 
 */
typedef struct kInfo64f
{
    k64f absoluteMin;               ///< Minimum valid value under any circumstances.
    k64f absoluteMax;               ///< Maximum valid value under any circumstances.
    k64f relativeMin;               ///< Minimum valid value, limited by current values of other (more dominant) settings.
    k64f relativeMax;               ///< Maximum valid value, limited by current values of other (more dominant) settings.
    k64f granularity;               ///< Value step size (provided only if available; otherwise, k64F_NULL). 
    k64f previous;                  ///< Previous valid value (provided only if available; otherwise, k64F_NULL). 
    k64f next;                      ///< Next valid value (provided only if available; otherwise, k64F_NULL). 
    kBool isReadOnly;               ///< Value is read-only (cannot be modified). 
    kBool isGrouped;                ///< Value is constrained to be the same as its peers within a group. (e.g. periods of states belonging to a camera might be constrained to be equal). 
    kViolation violations;          ///< Constraints violated by the current value. 
    k64f accurate;                  ///< Accurate representation of current value, typically used for display in meaningful units (optional). 
} kInfo64f;

/** 
 * Initializes a kInfo64f structure to default values. 
 *
 * @public              @memberof kInfo64f
 * @param   info        Info structure. 
 * @param   min         Minimum value. 
 * @param   max         Maximum value. 
 * @return              Operation status.  
 */
kFsFx(kStatus) kInfo64f_Init(kInfo64f* info, k64f min, k64f max); 

/** 
 * Validates an input value against min/max values.
 *
 * @public          @memberof kInfo64f
 * @param   info    Info structure. 
 * @param   value   Input value. 
 * @return          Operation status.  
 */
kFsFx(kStatus) kInfo64f_Validate(kInfo64f* info, k64f value); 

/** 
 * Appends violation descriptions to the given list. 
 *
 * @public          @memberof kInfo32u
 * @param   info    Info structure. 
 * @param   list    Error list (kArrayList<kText128>). 
 * @param   source  Label for object containing the error. 
 * @param   field   Label for field containing the error. 
 * @param   value   Current field value. 
 * @return          Operation status.  
 */
kFsFx(kStatus) kInfo64f_AppendErrors(const kInfo64f* info, kArrayList list, const kChar* source, const kChar* field, k64f value); 

/**
 * @struct  kInfoBool
 * @extends kValue
 * @ingroup kFireSync-Client-Utils
 * @brief   Represents constraint and validity information for a boolean setting. 
 */
typedef struct kInfoBool
{
    kBool absoluteMin;              ///< Minimum valid value under any circumstances.
    kBool absoluteMax;              ///< Maximum valid value under any circumstances.
    kBool relativeMin;              ///< Minimum valid value, limited by current values of other (more dominant) settings.
    kBool relativeMax;              ///< Maximum valid value, limited by current values of other (more dominant) settings.
    kBool isReadOnly;               ///< Value is read-only (cannot be modified). 
    kBool isGrouped;                ///< Value is constrained to be the same as its peers within a group.
    kViolation violations;          ///< Constraints violated by the current value. 
} kInfoBool;

/** 
 * Initializes a kInfoBool structure to default values. 
 *
 * @public          @memberof kInfoBool
 * @param   info    Info structure. 
 * @param   min     Minimum value. 
 * @param   max     Maximum value. 
 * @return          Operation status.  
 */
kFsFx(kStatus) kInfoBool_Init(kInfoBool* info, kBool min, kBool max); 

/** 
 * Validates an input value against min/max values.
 *
 * @public          @memberof kInfoBool
 * @param   info    Info structure. 
 * @param   value   Input value. 
 * @return          Operation status.  
 */
kFsFx(kStatus) kInfoBool_Validate(kInfoBool* info, kBool value); 

/** 
 * Appends violation descriptions to the given list. 
 *
 * @public          @memberof kInfo32u
 * @param   info    Info structure. 
 * @param   list    Error list (kArrayList<kText128>). 
 * @param   source  Label for object containing the error. 
 * @param   field   Label for field containing the error. 
 * @param   value   Current field value. 
 * @return          Operation status.  
 */
kFsFx(kStatus) kInfoBool_AppendErrors(const kInfoBool* info, kArrayList list, const kChar* source, const kChar* field, kBool value); 

/**
 * @struct  kInfoBits
 * @extends kValue
 * @ingroup kFireSync-Client-Utils
 * @brief   Represents constraint and validity information for a bit flag setting.  
 */
typedef struct kInfoBits                        
{
    k32u absoluteOptions;           ///< Set of valid options under any circumstances. 
    k32u relativeOptions;           ///< Set of valid options, limited by current values of other (more dominant) settings.
    kBool isReadOnly;               ///< Value is read-only (cannot be modified). 
    kBool isGrouped;                ///< Value is constrained to be the same as its peers within a group. 
    kViolation violations;          ///< Constraints violated by the current value. 
} kInfoBits;

/** 
 * Initializes a kInfoBits structure to default values. 
 *
 * @public          @memberof kInfoBits
 * @param   info    Info structure. 
 * @param   options Option bits. 
 * @return          Operation status.  
 */
kFsFx(kStatus) kInfoBits_Init(kInfoBits* info, k32u options); 

/** 
 * Checks whether the specified option is valid.
 * 
 * This function assumes that the option argument must contain exactly one bit. 
 *
 * @public          @memberof kInfoBits
 * @param   info    Info structure. 
 * @param   option  Option to validate. 
 * @return          Operation status.  
 */
kFsFx(kStatus) kInfoBits_ValidateOne(kInfoBits* info, k32u option); 

/** 
 * Checks whether the specified options are valid, setting appropriate violation flags. 
 * 
 * This function assumes that the options argument can contain multiple bits.  
 *
 * @public          @memberof kInfoBits
 * @param   info    Info structure. 
 * @param   options Option to validate. 
 * @return          Operation status.  
 */
kFsFx(kStatus) kInfoBits_ValidateAtLeastOne(kInfoBits* info, k32u options); 

/** 
 * Appends violation descriptions to the given list. 
 *
 * @public          @memberof kInfo32u
 * @param   info    Info structure. 
 * @param   list    Error list (kArrayList<kText128>). 
 * @param   source  Label for object containing the error. 
 * @param   field   Label for field containing the error. 
 * @param   value   Current field value. 
 * @return          Operation status.  
 */
kFsFx(kStatus) kInfoBits_AppendErrors(const kInfoBits* info, kArrayList list, const kChar* source, const kChar* field, k32u value); 

#include <kFireSync/Utils/kInfo.x.h>

#endif
