/** 
 * @file    kSvNode.h
 * @brief   Declares the kSvNode class and related types. 
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYC_SERVER_NODE_H
#define K_FIRESYC_SERVER_NODE_H

#include <kFireSync/Server/kSvDef.h>
#include <kFireSync/Server/kSvNode.x.h>

/**
 * @class       kSvNode
 * @extends     kObject
 * @ingroup     kFireSync-Server
 * @brief       Provides FireSync node network services and mediates access to node hardware.
 */
//typedef kObject kSvNode;          --forward-declared in kFsDef.x.h

/** 
 * Reports the device id of this node. 
 *
 * This method is thread-safe.
 * 
 * @public              @memberof kSvNode
 * @param   node        Node object. 
 * @return              Device id. 
 */
kFsFx(k32u) kSvNode_Id(kSvNode node);

/** 
 * Reports whether the node supports local file storage.
 *
 * This method is thread-safe.
 * 
 * @public              @memberof kSvNode
 * @param   node        Node object. 
 * @return              kTRUE if storage is supported.
 */
kFsFx(kBool) kSvNode_HasStorage(kSvNode node); 

/** 
 * Reports the root storage path, if the node supports local storage.
 *
 * To get the path to the user storage directory, combine the output of this function with kNODE_DIR_USER.  
 * 
 * This method is thread-safe.
 * 
 * @public              @memberof kSvNode
 * @param   node        Node object. 
 * @return              Storage root path.
 */
kFsFx(const kChar*) kSvNode_StoragePath(kSvNode node); 

/** 
 * Reports the number of network interfaces.
 * 
 * @internal            @memberof kSvNode
 * @param   node        Node object. 
 * @return              Number of network interfaces.
 */
kFsFx(kSize) kSvNode_NetInterfaceCount(kSvNode node);

/** 
 * Reports the current IP address configuration for this device, which can be dynamically 
 * updated due to DHCP activity. 
 * 
 * Note, this method can fail if the network adapter is currently being reconfigured. 
 *
 * This method is thread-safe.
 * 
 * @public              @memberof kSvNode
 * @param   node        Node object. 
 * @param   index       Interface index. 
 * @param   ipConfig    Current IP configuration.
 * @return              Operation status.
 */
kFsFx(kStatus) kSvNode_DynamicIpConfig(kSvNode node, kSize index, kIpConfig* ipConfig);

/** 
 * Sets network interface configuration and resets. 
 *
 * This method is thread-safe.
 * 
 * @public              @memberof kSvNode
 * @param   node        Node object. 
 * @param   index       Interface index. 
 * @param   ipConfig    Network interface configuration.
 * @return              Operation status.
 */
kFsFx(kStatus) kSvNode_SetAddressAndReset(kSvNode node, kSize index, const kIpConfig* ipConfig); 

/** 
 * Sets application information that will be made available via the kNode_Info function. 
 * 
 * This method method can only be called while all node services are stopped. 
 * 
 * @public              @memberof kSvNode
 * @param   node        Node object. 
 * @param   name        Application name.  
 * @param   version     Application version.
 * @return              Operation status.
 */
kFsFx(kStatus) kSvNode_SetAppInfo(kSvNode node, const kChar* name, kVersion version);

/** 
 * Sets reference to application object. 
 * 
 * Enables the application to set a reference to itself that can later be retreived 
 * from the server node object.
 * 
 * @public              @memberof kSvNode
 * @param   node        Node object. 
 * @param   app         Application object.
 * @return              Operation status.
 */
kFsFx(kStatus) kSvNode_SetApp(kSvNode node, kPointer app);

/** 
 * Gets a reference to the application object. 
 * 
 * @public              @memberof kSvNode
 * @param   node        Node object. 
 * @return              Application object (or kNULL if not set).
 */
kFsFx(kPointer) kSvNode_App(kSvNode node);

/** 
 * Registers a custom callback handler for incoming action requests. 
 *
 * Most action objects make use of the same kSystem object that is accessed via the kSvNode_Client 
 * function. But if this kSystem object is also used by the application, then a mutual exlusion 
 * mechasism should be implemented to prevent simultaneous access. By registering an action 
 * handler, the application takes responsibility for performing actions, and can implement custom 
 * thread synchronization/exclusion logic as needed. 
 *
 * To process the action, the callback should call kSvNode_PerformAction, passing the 'args' received by 
 * the callback as the kSvNode_PerformAction 'args' parameter. 
 * 
 * If an action handler is not installed by the application, a default action handler will be used. 
 * The default handler will execute actions without performing any special locking/unlocking steps. 
 * 
 * @public              @memberof kSvNode
 * @param   node        Node object. 
 * @param   function    Callback function. 
 * @param   receiver    Callback receiver. 
 * @return              Operation status. 
 * @see                 kSvNode_PerformAction
 */
kFsFx(kStatus) kSvNode_SetActionHandler(kSvNode node, kCallbackFx function, kPointer receiver);

/** 
 * Performs an action within an action handler callback. 
 * 
 * @public              @memberof kSvNode
 * @param   node        Node object. 
 * @param   args        Args parameter received by action handler callback.
 * @return              Operation status. 
 * @see                 kSvNode_SetActionHandler
 */
kFsFx(kStatus) kSvNode_PerformAction(kSvNode node, kPointer args);

/** 
 * Specifies that auto-assigned, ephemeral ports should be used for control, health, data, and web services. 
 * 
 * This option is useful when running multiple nodes on the same same host. 
 * 
 * @public              @memberof kSvNode
 * @param   node        Node object. 
 * @see                 kSvNode_SetActionHandler
 */
kFsFx(kStatus) kSvNode_UseEphemeralPorts(kSvNode node);

/** 
 * Registers a custom callback handler for incoming HTTP requests. 
 * 
 * Refer to the kHttpServer_SetHandler documentation for a description of callback behaviour. 
 * 
 * This method is thread-safe.
 * 
 * @public              @memberof kSvNode
 * @param   node        Node object. 
 * @param   function    Callback function. 
 * @param   receiver    Callback receiver. 
 * @return              Operation status. 
 * @see                 kHttpServer_SetHandler
 */
kFsFx(kStatus) kSvNode_SetWebHandler(kSvNode node, kCallbackFx function, kPointer receiver);

/** 
 * Starts node services (e.g., discovery/control servers). 
 * 
 * Note that in order for this node to be represented in the local kSystem to which it belongs, it 
 * is necessary to include the kSV_NODE_SERVICE_SYSTEM flag. Some node features, such as actions, may 
 * be unavailable until this step is performed. As a result of calling this method with the 
 * kSV_NODE_SERVICE_SYSTEM flag, the system object will be automatically refreshed, which may result 
 * the creation/removal of node handles. 
 *
 * @public              @memberof kSvNode
 * @param   node        Node object. 
 * @param   services    Service types. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kSvNode_StartServices(kSvNode node, kSvNodeService services);

/** 
 * Stops node services (e.g., discovery/control servers). 
 *
 * Node services are stopped automatically when the node instance is destroyed. However, this function 
 * can be used to explicity stop services before destroying the node. 
 * 
 * @public              @memberof kSvNode
 * @param   node        Node object. 
 * @param   services    Service types. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kSvNode_StopServices(kSvNode node, kSvNodeService services);

/** 
 * Reports whether a watchdog timer is available on this platform.
 *
 * @public              @memberof kSvNode
 * @param   node        Node object. 
 * @return              kTRUE if watchdog is available.  
 */
kFsFx(kBool) kSvNode_HasWatchdog(kSvNode node);  

/** 
 * Enables or disables the watchdog timer.
 * 
 * Watchdog timers are disabled by default. When the watchdog timer is enabled, the provided callback 
 * function will be invoked periodically, according to the checkPeriod argument. Refer to the kSvWatchdogFx 
 * documentation for information on how to implement this callback. 
 * 
 * The timeoutPeriod argument determines the countdown period of the underlying watchdog timer. 
 * If the countdown reaches zero without positive affirmation of system liveliness (determined by the  
 * callback), the node will automatically reset. The watchdog countdown is restarted each time 
 * that the callback affirms liveliness. Naturally, timeoutPeriod should be set to something higher 
 * than checkPeriod, with a margin of safety to account for thread scheduling variability. Setting 
 * timeoutPeriod to twice the duration of checkPeriod would be a reasonable starting point.
 * 
 * @public                  @memberof kSvNode
 * @param   node            Node object. 
 * @param   enable          kTRUE to enable the watchdog. 
 * @param   timeoutPeriod   Coundown period of the underlying watchdog (microseconds).  
 * @param   checkPeriod     Period at which the system liveliness callback will be invoked (microseconds). 
 * @param   callback        System liveliness callback. 
 * @param   receiver        Callback context pointer.
 * @return                  Operation status. 
 * @see                     kSvNode_HasWatchdog, kSvWatchdogFx
 */
kFsFx(kStatus) kSvNode_EnableWatchdog(kSvNode node, kBool enable, k64u timeoutPeriod, k64u checkPeriod, kSvWatchdogFx callback, kPointer receiver); 

/** 
 * Gets the kSystem object that hosts this server.  
 * 
 * @public          @memberof kSvNode
 * @param   node    Node object. 
 * @return          System object. 
 */
kFsFx(kSystem) kSvNode_System(kSvNode node); 

/** 
 * Gets a memory allocator that is suitable for quickly allocating numerous, short-lived objects. 
 * 
 * The message allocator is intended to allocate the kind of short-lived objects that are frequently 
 * generated during acquisition and data processing activities. It should not be used to allocate objects 
 * that will exist for a longer duration (e.g., service implementation objects). 
 * 
 * The message allocator can typically fulfill small memory requests faster than the normal application 
 * allocator (kAlloc_App). The tradeoff is that these allocations may be cached for a longer duration
 * before being returned to the underlying heap. 
 * 
 * The allocator object returned by this function is guaranteed to be valid for the lifetime 
 * of the kSvNode object. Ownership cannot be extended by increasing the allocator's reference count.  
 * 
 * @public              @memberof kSvNode
 * @param   node        Node object. 
 * @return              Message allocator object. 
 */
kFsFx(kAlloc) kSvNode_MessageAlloc(kSvNode node); 

/** 
 * Blocks until configured method of synchronization is established. 
 * 
 * This function can optionally be called during startup activitives to wait for synchronization. 
 * If the node is configured for master/network synchronization, this function will block until 
 * master synchronization is established or a timeout occurs. 
 *
 * @public              @memberof kSvNode
 * @param   node        Node object. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kSvNode_WaitForSync(kSvNode node);

/** 
 * This function can be called to initiate application restart/shutdown, causing  
 * a pending call to the kSvNode_WaitForShutdown function to return. 
 *
 * Application restarts enable information from error-checking that normally occurs
 * during shutdown to be preserved so that it can be accessed after restart. It can 
 * also be used (with some risk) as a faster form of reset when debugging. Application 
 * restarts may not be supported on every platform. If this feature is not supported, 
 * this method has no effect. 
 * 
 * This method is thread safe. 
 * 
 * @see                 kSvNode_WaitForShutdown
 * @public              @memberof kSvNode
 * @param   node        Node object. 
 * @param   restartApp  Specifies whether the appication should restart (use kFALSE to reboot the device).
 * @return              Operation status. 
 */
kFsFx(kStatus) kSvNode_NotifyShutdown(kSvNode node, kBool restartApp); 

/** 
 * Blocks until this node is ready for shutdown. 
 * 
 * This function is typically called after kSvNode_StartServices to wait for node activities to complete.
 * Node activies normally run indefinitely, and complete only when a reset command is given to the node via 
 * the kNode_Reset function.
 * 
 * This function can be used sucessfully only once per node object. After the first time that this function 
 * returns kOK, any subsequent calls to this function will produce undefined behaviour.
 * 
 * Use of this function is technically optional. Alternatively, it is possible to explicitly terminate node 
 * services by destroying this kSvNode instance. 
 *
 * @public              @memberof kSvNode
 * @param   node        Node object. 
 * @param   timeout     Timeout value (or kINFINITE to wait indefinitely). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kSvNode_WaitForShutdown(kSvNode node, k64u timeout);

/** 
 * Writes to the server node log.
 * 
 * This method is thread-safe.
 *
 * @public              @memberof kSvNode
 * @param   node        Node object. 
 * @param   message     Print format string. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kSvNode_Log(kSvNode node, const kChar* message);

/** 
 * Writes to the server node log.
 * 
 * This method is thread-safe.
 *
 * @public              @memberof kSvNode
 * @param   node        Node object. 
 * @param   options     Log options.
 * @param   source      Message source to print (set to kNULL if printing not desired).
 * @param   message     Print format string. 
 * @return              Operation status. 
 */
kInlineFx(kStatus) kSvNode_Log(kSvNode node, kLogOption options, const kChar* source, const kChar* message)
{
    return xkSvNode_Log(node, options, source, message);
}

/** 
 * Variable-argument version of kSvNode_Logf. 
 * 
 * Writes to the server node log.
 *
 * @public              @memberof kSvNode
 * @param   node        Node object. 
 * @param   format      Print format string. 
 * @param   argList     Variable argument list.
 * @return              Operation status. 
 */
kFsFx(kStatus) kSvNode_Logvf(kSvNode node, const kChar* format, kVarArgList argList);

/** 
 * Variable-argument version of kSvNode_Logf. 
 * 
 * Writes to the server node log.
 *
 * @public              @memberof kSvNode
 * @param   node        Node object. 
 * @param   options     Log options.
 * @param   source      Message source to print (set to kNULL if printing not desired).
 * @param   format      Print format string. 
 * @param   argList     Variable argument list.
 * @return              Operation status. 
 */
kInlineFx(kStatus) kSvNode_Logvf(kSvNode node, kLogOption options, const kChar* source, const kChar* format, kVarArgList argList)
{
    return xkSvNode_Logvf(node, options, source, format, argList);
}

/** 
 * Writes to the server node log.
 * 
 * This method is thread-safe.
 *
 * @public              @memberof kSvNode
 * @param   node        Node object. 
 * @param   format      Print format string. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kSvNode_Logf(kSvNode node, const kChar* format, ...);

/** 
 * Writes to the server node log.
 * 
 * This method is thread-safe.
 *
 * @public              @memberof kSvNode
 * @param   node        Node object. 
 * @param   options     Log options.
 * @param   source      Message source to print (set to kNULL if printing not desired).
 * @param   format      Print format string. 
 * @return              Operation status. 
 */
kInlineFx(kStatus) kSvNode_Logf(kSvNode node, kLogOption options, const kChar* source, const kChar* format, ...)
{
    return xkSvNode_Logf(node, options, source, format);
}

/** 
 * Reports the discovery port that will be used to broadcast information about this node.

 * @public              @memberof kSvNode
 * @param   node        Node object. 
 * @return              Port id. 
 */
kFsFx(k32u) kSvNode_NextDiscoveryPort(kSvNode node);

#endif

