/** 
 * @file    kPxSerialPipe.x.h
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_PX_SERIAL_PIPE_X_H
#define K_FIRESYNC_PX_SERIAL_PIPE_X_H

#include <kFireSync/Pipe/kPxPipe.h>
#include <kApi/Threads/kMsgQueue.h>

#define kPX_SERIAL_PIPE_QUIT_QUERY_PERIOD             (100000)          ///< Polling interval for quit (stop) status (us). 

/*
 * kPxSerialPipeItem
 */

/**
 * @internal
 * @struct  kPxSerialPipeItem
 * @extends kValue
 * @ingroup kFireSync-Pipe
 * @brief   Represents a message item that has been received and enqueued by a kPxSerialPipe object. 
 */
typedef struct kPxSerialPipeItem
{
    kMsgInfo message;           ///< Message object. 
    kSize size;                 ///< Calculated size of message object. 
} kPxSerialPipeItem;

kDeclareValueEx(kFs, kPxSerialPipeItem, kValue)

/*
 * kPxSerialPipe
 */

typedef struct kPxSerialPipeClass
{
    kPxPipeClass base;
    
    kAlloc messageAlloc;                //Message allocator. 
    
    kArrayList blocks;                  //List of processing blocks -- kArrayList<kPxBlock>. 
    kArrayList routes;                  //List of routes -- kArrayList<kRouteEntry>. 
    
    kLock messageLock;                  //Provides exclusive access to internal and external message queues. 
    kSemaphore messageSem;              //Semaphore representing outstanding messages to be processed.
    kQueue externalMessages;            //Unprocessed asynchronous messages from outside of the pipe-- kQueue<kPxSerialPipeItem>. 
    kQueue internalMessages;            //Unprocessed asynchronous messages from inside of the pipe -- kQueue<kPxSerialPipeItem>. 
     
    kArrayList activeMessages;          //Unprocessed block messages in current synchronous execution sequence -- kArrayList <kMsgInfo>. 
    kSize activeMessagesOffset;         //Offset at which to insert messages into activeMessages list.

    kSize maxQueueSize;                 //Maximum message bytes that can be enqueued. 
    kAtomicPointer queueSize;           //Count of message bytes enqueued but not yet submitted for processing. 
    kSize maxQueueCount;                //Maximum count of items that can be enqueued. 
    kAtomicPointer queueCount;          //Count of messages currently enqueued but not yet submitted for processing. 

    kThread thread;                     //Data processing thread.
    kAtomic32s shouldQuit;              //Thread quit flag. 
    kAtomic32s executionStatus;         //Pipe execution status.

    kHealthProbe messageDropProbe;      //Health probe for input message drops. 
    kHealthProbe bytesEnqueuedProbe;    //Health probe for bytes of input data pending processing.
    kHealthProbe itemsEnqueuedProbe;    //Health probe for messages of input data pending processing.
    kHealthProbe executionStatusProbe;  //Health probe for execution status.
    kHealthProbe totalMessageDropProbe; //Health probe for total message drops since start of the health log. 
    kHealthProbe executionErrorTotalProbe; //Health probe for total execution errors since start of the health log. 
} kPxSerialPipeClass; 

kDeclareClassEx(kFs, kPxSerialPipe, kPxPipe)
        
kFsFx(kStatus) kPxSerialPipe_Init(kPxSerialPipe pipe, kType type, k32u nodeId, kPxEnviron pipeEnviron, kAlloc alloc); 

kFsFx(kStatus) kPxSerialPipe_VRelease(kPxSerialPipe pipe);

kFsFx(kStatus) kPxSerialPipe_VSetMaxQueueSize(kPxSerialPipe pipe, kSize size); 
kFsFx(kStatus) kPxSerialPipe_VSetMaxQueueCount(kPxSerialPipe pipe, kSize count); 
kFsFx(kStatus) kPxSerialPipe_VClear(kPxSerialPipe pipe);
kFsFx(kStatus) kPxSerialPipe_VAddBlocks(kPxSerialPipe pipe, const kPxBlock* blocks, kSize count);
kFsFx(kStatus) kPxSerialPipe_VAddRoutes(kPxSerialPipe pipe, const kRouteEntry* routes, kSize count); 
kFsFx(kStatus) kPxSerialPipe_VStart(kPxSerialPipe pipe);
kFsFx(kStatus) kPxSerialPipe_VEngage(kPxSerialPipe pipe);
kFsFx(kStatus) kPxSerialPipe_VStop(kPxSerialPipe pipe);
kFsFx(kStatus) kPxSerialPipe_VPause(kPxSerialPipe pipe);
kFsFx(kStatus) kPxSerialPipe_VResume(kPxSerialPipe pipe);
kFsFx(kStatus) kPxSerialPipe_VReplay(kPxSerialPipe pipe, kMsgInfo message);
kFsFx(kStatus) kPxSerialPipe_VUpdateStats(kPxSerialPipe pipe);

kFsFx(kStatus) kPxSerialPipe_ClearStats(kPxSerialPipe pipe);
kFsFx(kStatus) kPxSerialPipe_ClearMessages(kPxSerialPipe pipe); 

kFsFx(kStatus) kPxSerialPipe_OnSend(kPxSerialPipe pipe, kPxPort port, kMsgSet msg);
kFsFx(kStatus) kPxSerialPipe_OnSendSync(kPxSerialPipe pipe, kPxPort port, kMsgSet msg);
kFsFx(kStatus) kPxSerialPipe_OnSendAsync(kPxSerialPipe pipe, kPxPort port, kMsgSet msg);

kFsFx(kStatus) kPxSerialPipe_ClearConnections(kPxSerialPipe pipe); 
kFsFx(kStatus) kPxSerialPipe_EnumerateConnections(kPxSerialPipe pipe); 
kFsFx(kStatus) kPxSerialPipe_FindPort(kPxSerialPipe pipe, k32u blockId, k32u portId, kPxPort* port); 

kFsFx(kStatus) kPxSerialPipe_StartProcessing(kPxSerialPipe pipe);
kFsFx(kStatus) kPxSerialPipe_StopProcessing(kPxSerialPipe pipe); 
kFsFx(kStatus) kPxSerialPipe_ProcessThread(kPxSerialPipe pipe); 
kFsFx(void) kPxSerialPipe_ProcessMessage(kPxSerialPipe pipe, kMsgInfo msgInfo); 

kFsFx(kStatus) kPxSerialPipe_EnqueueMessage(kPxSerialPipe pipe, kQueue queue, kMsgInfo info, kBool canDrop);

kInlineFx(kStatus) kPxSerialPipe_EnqueueExternalMessage(kPxSerialPipe pipe, kMsgInfo info, kBool canDrop)
{
    kObj(kPxSerialPipe, pipe); 
    
    return kPxSerialPipe_EnqueueMessage(pipe, obj->externalMessages, info, canDrop); 
}

kInlineFx(kStatus) kPxSerialPipe_EnqueueInternalMessage(kPxSerialPipe pipe, kMsgInfo info, kBool canDrop)
{
    kObj(kPxSerialPipe, pipe); 
    
    return kPxSerialPipe_EnqueueMessage(pipe, obj->internalMessages, info, canDrop); 
}

kFsFx(kStatus) kPxSerialPipe_DequeueMessage(kPxSerialPipe pipe, kMsgInfo* info, k64u timeout);
kFsFx(kStatus) kPxSerialPipe_DequeueAllMessages(kPxSerialPipe pipe);

kFsFx(kStatus) kPxSerialPipe_StartBlocks(kPxSerialPipe pipe); 
kFsFx(kStatus) kPxSerialPipe_EngageBlocks(kPxSerialPipe pipe); 
kFsFx(kStatus) kPxSerialPipe_StopBlocks(kPxSerialPipe pipe); 
kFsFx(kStatus) kPxSerialPipe_PauseBlocks(kPxSerialPipe pipe); 
kFsFx(kStatus) kPxSerialPipe_ResumeBlocks(kPxSerialPipe pipe); 

kFsFx(kStatus) kPxSerialPipe_NotifyDrop(kPxSerialPipe pipe); 

//consider kApi support at some point
kInlineFx(kSSize) kPxSerialPipe_AtomicSSizeGet(kAtomicPointer* variable)
{
    return (kSSize) kAtomicPointer_Get(variable); 
}

//consider kApi support at some point
kInlineFx(void) kPxSerialPipe_AtomicSSizeAdd(kAtomicPointer* variable, kSSize amount)
{
    kPointer oldValue, newValue; 

    do
    {
        oldValue = kAtomicPointer_Get(variable); 
        newValue = (kPointer)((kSSize)oldValue + amount); 
    }
    while (!kAtomicPointer_CompareExchange(variable, oldValue, newValue)); 
}

kInlineFx(void) kPxSerialPipe_NotifyItemEnqueued(kPxSerialPipe pipe, kSize size)
{
    kObj(kPxSerialPipe, pipe); 
    
    kPxSerialPipe_AtomicSSizeAdd(&obj->queueSize, (kSSize)size); 
    kPxSerialPipe_AtomicSSizeAdd(&obj->queueCount, 1); 
}

kInlineFx(void) kPxSerialPipe_NotifyItemDequeued(kPxSerialPipe pipe, kSize size)
{
    kObj(kPxSerialPipe, pipe); 
    
    kPxSerialPipe_AtomicSSizeAdd(&obj->queueSize, -1 * (kSSize)size); 
    kPxSerialPipe_AtomicSSizeAdd(&obj->queueCount, -1); 
}

kInlineFx(kSSize) kPxSerialPipe_QueueSize(kPxSerialPipe pipe)
{
    kObj(kPxSerialPipe, pipe); 

    return kPxSerialPipe_AtomicSSizeGet(&obj->queueSize); 
}

kInlineFx(kSSize) kPxSerialPipe_QueueCount(kPxSerialPipe pipe)
{
    kObj(kPxSerialPipe, pipe); 

    return kPxSerialPipe_AtomicSSizeGet(&obj->queueCount); 
}

kInlineFx(kBool) kPxSerialPipe_CanAcceptMessage(kPxSerialPipe pipe, kSize size)
{
    kObj(kPxSerialPipe, pipe); 

    return ((obj->maxQueueCount == kSIZE_NULL) || ((kPxSerialPipe_QueueCount(pipe) + 1) <= (kSSize)obj->maxQueueCount)) && 
           ((obj->maxQueueSize  == kSIZE_NULL) || ((kPxSerialPipe_QueueSize(pipe) + (kSSize)size) <= (kSSize)obj->maxQueueSize));
}

#endif
