/** 
 * @file    kPxPort.h
 * @brief   Declares the kPxPort type. 
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_PX_PORT_H
#define K_FIRESYNC_PX_PORT_H

#include <kFireSync/kNodeDef.h>
#include <kApi/Data/kQueue.h>
#include <kFireSync/Pipe/kPxPort.x.h>

/**
 * @class   kPxPort
 * @extends kObject
 * @ingroup kFireSync-Pipe
 * @brief   Represents a kPxBlock port.
 */
//typedef kObject kPxPort;        --forward-declared in kFsDef.x.h

/** 
 * Gets the unique numerical identifier for this port. 
 *
 * This method is thread-safe.
 * 
 * @public              @memberof kPxPort
 * @param   port        Port object.  
 * @return              Returns the unique identifier for this port. 
 */
kInlineFx(k32u) kPxPort_Id(kPxPort port)
{
    kObj(kPxPort, port); 
 
    return obj->id; 
}

/** 
 * Gets a descriptive name for the port. 
 *
 * This method is thread-safe.
 * 
 * @public              @memberof kPxPort
 * @param   port        Port object.  
 * @return              Port name; 
 */
kInlineFx(const kChar*) kPxPort_Name(kPxPort port)
{
    kObj(kPxPort, port); 
 
    return obj->name; 
}

/** 
 * Gets the directionality of the port (input vs. output).
 *
 * This method is thread-safe.
 * 
 * @public              @memberof kPxPort
 * @param   port        Port object.  
 * @return              Returns the directionality of the port. 
 */
kInlineFx(kPortDirection) kPxPort_Direction(kPxPort port)
{
    kObj(kPxPort, port); 
 
    return obj->direction; 
}

/** 
 * Enables an input port to accept foreign data. 
 * 
 * Messages can contain memory that is inaccessible to the host CPU (e.g., Cuda device memory). By default, these 
 * messages will be automatically converted to host-accessible memory before being passed to block input ports. 
 * 
 * This method can be used to override the default behavior, allowing messages that contain foreign data elements 
 * to be accepted. In this case, the receiver is responsible for ensuring that foreign data is not directly accessed 
 * by the CPU. 
 * 
 * This method can only be called from within a block's VOnSetup override.
 * 
 * @public              @memberof kPxPort
 * @param   port        Port object.  
 * @param   enable      kTRUE to enable foreign data to be accepted.  
 * @return              Operation status. 
 */
kInlineFx(kStatus) kPxPort_AcceptForeignData(kPxPort port, kBool enable)
{
    kObj(kPxPort, port); 

    obj->canAcceptForeignData = enable; 

    return kOK;
}

/** 
 * Reports whether foreign data can be accepted by an input port. 
 *
 * This method is thread-safe.
 * 
 * @public              @memberof kPxPort
 * @param   port        Port object.  
 * @return              kTRUE if foreign data can be accepted. 
 */
kInlineFx(kBool) kPxPort_CanAcceptForeignData(kPxPort port)
{
    kObj(kPxPort, port); 

    return obj->canAcceptForeignData; 
}

/** 
 * Enables an output port to emit messages asynchronously. 
 * 
 * By default, the EmitAsynchronousData property is set to kFALSE, meaning that messages can only be 
 * emitted synchronously. In this state, attempting to send a message outside the scope of a block's OnReceive 
 * handler will result in undefined behaviour. When the EmitAsynchronousData property is set to kTRUE, messages 
 * are delivered to the pipe asynchronously, enabling messages to be sent either inside or outside of a block's 
 * OnReceive handler. Refer to the kPxBlock_Send API documentation for a more complete discussion of the rules
 * surrounding asynchronous messaging.
 * 
 * Blocks that emit messages synchronously should leave this setting at its default (disabled), as synchronous 
 * processing is generally more efficient than asynchronous processing. 
 * 
 * This method can only be called from within a block's VOnSetup override.
 * 
 * @public              @memberof kPxPort
 * @param   port        Port object.  
 * @param   enable      kTRUE to enable messages to be emitted asynchronously.  
 * @return              Operation status. 
 * @see                 kPxBlock_Send, kPxBlock_OnEngage
 */
kInlineFx(kStatus) kPxPort_EnableAsynchronousData(kPxPort port, kBool enable)
{
    kObj(kPxPort, port); 

    obj->asynchronousDataEnabled = enable; 

    return kOK;
}
/** 
 * Reports whether an output port is configured to emit messages asynchronously. 
 *
 * This method is thread-safe.
 * 
 * @public              @memberof kPxPort
 * @param   port        Port object.  
 * @return              kTRUE if messages will be emitted asynchronously.  
 */
kInlineFx(kBool) kPxPort_AsynchronousDataEnabled(kPxPort port)
{
    kObj(kPxPort, port); 

    return obj->asynchronousDataEnabled; 
}

/** 
 * Enables an output port to drop messages. 
 * 
 * By default, the DropEnabled property is set to kFALSE, meaning that messages cannot be dropped. 
 * If this property is set to true, the pipe can decide (at its discretion) to drop messages 
 * that are sent on this port. The most common use for this feature is to prevent the unbounded 
 * accumulation of asynchronously-generated messages within pipe message processing buffers. 
 * 
 * This method can only be called from within a block's VOnSetup override.
 * 
 * @public              @memberof kPxPort
 * @param   port        Port object.  
 * @param   enable      kTRUE to enable messages to be dropped by the pipe.  
 * @return              Operation status. 
 * @see                 kPxBlock_Send, kPxBlock_OnEngage
 */
kInlineFx(kStatus) kPxPort_EnableDrop(kPxPort port, kBool enable)
{
    kObj(kPxPort, port); 

    obj->dropEnabled = enable; 

    return kOK;
}

/** 
 * Reports whether an output port is allowed to drop messages.
 *
 * This method is thread-safe.
 * 
 * @public              @memberof kPxPort
 * @param   port        Port object.  
 * @return              kTRUE if messages can be dropped.
 */
kInlineFx(kBool) kPxPort_DropEnabled(kPxPort port)
{
    kObj(kPxPort, port); 

    return obj->dropEnabled; 
}

/** 
 * Gets the block to which the port belongs. 
 *
 * This method is thread-safe.
 * 
 * @public              @memberof kPxPort
 * @param   port        Port object.  
 * @return              Block object. 
 */
kInlineFx(kPxBlock) kPxPort_Block(kPxPort port)
{
    kObj(kPxPort, port); 

    return obj->block; 
}

/** 
 * For an output port, reports whether the port is connected to any input ports. 
 * 
 * This method can only be called during or after a block's VOnStart method and 
 * before or during its VOnStop method. Within this timeframe, calls to this method
 * are thread-safe.
 * 
 * @public              @memberof kPxPort
 * @param   port        Port object.  
 * @return              kTRUE if the port is routed; otherwise kFALSE. 
 */
kInlineFx(kBool) kPxPort_IsRouted(kPxPort port)
{
    kObj(kPxPort, port); 

    return (kArrayList_Count(obj->connections) > 0); 
}

/**
* Count of enqueued input messages available on this input port. 
*
* This method can only be called from within a block's VOnReceive method.
* 
* @public              @memberof kPxPort
* @param   port        Port object.
* @return              Message count.
*/
kInlineFx(kSize) kPxPort_MessageCount(kPxPort port)
{
    kObj(kPxPort, port);

    return kQueue_Count(obj->messages); 
}

/**
* Accesses a message enqueued on this input port. 
*
* This method can only be called from within a block's VOnReceive method.
* 
* @public              @memberof kPxPort
* @param   port        Port object.
* @param   index       Enqueued message index (0 for next message). 
* @return              Message object.
*/
kFsFx(kMsgSet) kPxPort_MessageAt(kPxPort port, kSize index);

/**
* Accesses source information for a message enqueued on this input port.
*
* This method can only be called from within a block's VOnReceive method.
* 
* @public              @memberof kPxPort
* @param   port        Port object.
* @param   index       Enqueued message index (0 for next message).
* @return              Message source information.
*/
kFsFx(kMsgSource) kPxPort_MessageSourceAt(kPxPort port, kSize index);

/**
* Dequeues the next input message received on this input port.
*
* This method can only be called from within a block's VOnReceive method.
* 
* @public              @memberof kPxPort
* @param   port        Port object.
* @param   message     Receives message (ownership is transferred). 
* @param   source      Optionally receives message source information (can be null). 
* @return              Operation status. 
*/
kFsFx(kStatus) kPxPort_RemoveMessage(kPxPort port, kMsgSet* message, kMsgSource* source);

#endif
