/** 
 * @file    kHealthSummary.h
 * @brief   Declares the kHealthSummary class and related types. 
 *
 * @internal
 * Copyright (C) 2019-2022 by LMI Technologies Inc. All rights reserved.
 */
#ifndef K_FIRESYNC_HEALTH_SUMMARY_H
#define K_FIRESYNC_HEALTH_SUMMARY_H

#include <kFireSync/kNodeDef.h>
#include <kApi/Data/kMap.h>
#include <kFireSync/Health/kHealthSummary.x.h>

/**
 * @class   kHealthSummary
 * @extends kObject
 * @ingroup kFireSync-Health
 * @brief   Represents a summary of health log data.
 * 
 * Health log summaries consist of a set of stat items, where each stat corresponds to one 
 * health statistic in a health log. Each stat consists of metadata (name, id, instance) 
 * and an array of health log entries (k64s values) that were collected over time. 
 * 
 * All stat arrays contain the same number of log entries. Entries that were missing/inaccessible 
 * in the original log data are represented by k64S_NULL.
 * 
 * Stat items are ordered alphabetically according to stat name. 
 *
 * The following example (based on the implementation of kHealthSummary_ToString) illustrates how to 
 * format the content of a health summary as a CSV string. 
 * 
@code {.c}
kStatus FormatHealthSummary(kHealthSummary summary, kString str)
{
    for (kSize i = 0; i < kHealthSummary_StatCount(summary); ++i)
    {
        kHealthSummaryStat stat = kHealthSummary_StatAt(summary, i); 

        kCheck(kString_Addf(str, "%s,", kHealthSummary_StatName(summary, stat))); 

        for (kSize j = 0; j < kHealthSummary_EntryCount(summary); ++j)
        {
            k64s entry = kHealthSummary_EntryAt(summary, stat, j); 

            if (entry != k64S_NULL)
            {
                kCheck(kString_Addf(str, "%lld,", entry));  
            }
            else
            {
                kCheck(kString_Addf(str, ","));  
            }
        }
        kCheck(kString_Addf(str, "\n"));  
    }

    return kOK;
}
@endcode
 * 
 * kHealthSummary supports the kObject_Clone and kObject_Size methods.
 *
 * kHealthSummary supports the kdat6 serialization protocol. 
 */

//typedef kObject kHealthSummary;            --forward-declared in kFsDef.x.h  

/** 
 * @class   kHealthSummaryStat
 * @extends kObject
 * @ingroup kFireSync-Health
 * @brief   Represents a health statistic within a health log summary.
*/
//typedef kObject kHealthSummaryStat;            --forward-declared in kFsDef.x.h  

/**
 * Constructs a kHealthSummary object.
 *
 * @public                  @memberof kHealthSummary
 * @param   summary         Destination for the constructed object handle.
 * @param   allocator       Memory allocator (or kNULL for default).
 * @return                  Operation status.
 */
kFsFx(kStatus) kHealthSummary_Construct(kHealthSummary* summary, kAlloc allocator);

/**
 * Opens the health summary for writing.
 *
 * Each call to BeginCompose should be matched by a call to EndCompose when writing is
 * is complete. 
 * 
 * This method is used in the implementation of health services to compose a health 
 * summary object from log data.
 * 
 * @public              @memberof kHealthSummary
 * @param   summary     Health summary.
 * @param   entryCount  Count of log entries per stat.
 * @return              Operation status.
 */
kFsFx(kStatus) kHealthSummary_BeginCompose(kHealthSummary summary, kSize entryCount); 

/**
 * Adds a stat to the log. 
 *
 * This method is used in the implementation of health services to compose a health 
 * summary object from log data. BeginCompose must be called prior to calling AddStat
 * and EndCompose must be called when all stats have been written. 
 * 
 * The stat reference returned by this method can be used to access the entry data
 * associated with the stat. The entry array-lists will be initialized with the 
 * correct data type and capacity, but will be empty. It is the caller's responsibility to 
 * populate the arrays with data before calling kHealthSummary_EndCompose.
 * 
 * @public              @memberof kHealthSummary
 * @param   summary     Health summary.
 * @param   name        Stat name.
 * @param   id          Stat id. 
 * @param   instance    Stat instance.
 * @param   stat        Optionally receives reference to new stat. 
 * @return              Operation status.
 */
kFsFx(kStatus) kHealthSummary_AddStat(kHealthSummary summary, const kChar* name, kHealthId id, k32u instance, kHealthSummaryStat* stat); 

/**
 * Closes the health summary for writing.
 *
 * This method is used in the implementation of health services to compose a health 
 * summary object from log data.
 * 
 * @public              @memberof kHealthSummary
 * @param   summary     Health summary.
 * @return              Operation status.
 */
kFsFx(kStatus) kHealthSummary_EndCompose(kHealthSummary summary); 

/**
 * Reports the number of stats represented in the summary.
 *
 * Health log summaries consist of a set of stat items, where each stat corresponds to one 
 * health statistic in a health log. 
 * 
 * @public              @memberof kHealthSummary
 * @param   summary     Health summary.
 * @return              Counts of stats in the summary.
 */
kInlineFx(kSize) kHealthSummary_StatCount(kHealthSummary summary)
{
    kObj(kHealthSummary, summary);

    return kArrayList_Count(obj->stats);
}

/**
 * Reports the number of log entries per stat in the summary.
 *
 * All stat arrays contain the same number of log entries. Entries that were missing/inaccessible 
 * in the original log data are represented by k64S_NULL.
 * 
 * @public              @memberof kHealthSummary
 * @param   summary     Health summary.
 * @return              Counts of log entries per stat.
 */
kInlineFx(kSize) kHealthSummary_EntryCount(kHealthSummary summary)
{
    kObj(kHealthSummary, summary);

    return obj->entryCount;
}

/**
 * Gets a reference to the stat item at the specified index.
 *
 * Health log summaries consist of a set of stat items, where each stat corresponds to one 
 * health statistic in a health log. 
 * 
 * @public              @memberof kHealthSummary
 * @param   summary     Health summary.
 * @param   index       Stat index.
 * @param   stat        Receives stat reference.
 * @return              Operation status.
 */
kInlineFx(kStatus) kHealthSummary_StatItem(kHealthSummary summary, kSize index, kHealthSummaryStat* stat)
{
    kObj(kHealthSummary, summary);

    return kArrayList_ItemT(obj->stats, index, stat);
}

/**
 * Returns a reference to the stat item at the specified index.
 *
 * Health log summaries consist of a set of stat items, where each stat corresponds to one 
 * health statistic in a health log. 
 * 
 * @public              @memberof kHealthSummary
 * @param   summary     Health summary.
 * @param   index       Stat index.
 * @return              Health stat reference.
 */
kInlineFx(kHealthSummaryStat) kHealthSummary_StatAt(kHealthSummary summary, kSize index)
{
    kObj(kHealthSummary, summary);

    return kArrayList_AsT(obj->stats, index, kHealthSummaryStat);
}

/**
 * Looks up a stat item by its unique id-instance pair. 
 *
 * Health log summaries consist of a set of stat items, where each stat corresponds to one 
 * health statistic in a health log.
 * 
 * This method executes in amortized O(1) time. The first invocation may perform extra work 
 * that enables subsequent invocations to execute more efficiently. 
 * 
 * @public              @memberof kHealthSummary
 * @param   summary     Health summary.
 * @param   id          Stat id. 
 * @param   instance    Stat instance.
 * @param   stat        Receives stat reference. 
 * @return              Operation status (kERROR_NOT_FOUND if missing).
 */
kFsFx(kStatus) kHealthSummary_FindStat(kHealthSummary summary, kHealthId id, k32u instance, kHealthSummaryStat* stat);

/**
 * Prints the contents of the summary to a readable string. 
 *
 * This method is intended to support informal diagnostic activities. The actual format of 
 * the output string may change over time to suit present needs. Accordingly, if an exact/stable 
 * format is required, this method should be avoided.
 * 
 * @public              @memberof kHealthSummary
 * @param   summary     Health summary.
 * @param   str         String to receive summary content. 
 * @return              Operation status.
 */
kFsFx(kStatus) kHealthSummary_ToString(kHealthSummary summary, kString str); 

/**
 * Reports the name of the specified health stat. 
 * 
 * @public              @memberof kHealthSummary
 * @param   summary     Health summary.
 * @param   stat        Health stat.
 * @return              Stat name.
 */
kInlineFx(const kChar*) kHealthSummary_StatName(kHealthSummary summary, kHealthSummaryStat stat)
{
    return kHealthSummaryStat_Name(stat);
}

/**
 * Reports the ID associated with the specified health stat. 
 * 
 * @public              @memberof kHealthSummary
 * @param   summary     Health summary.
 * @param   stat        Health stat.
 * @return              Stat ID.
 */
kInlineFx(kHealthId) kHealthSummary_StatId(kHealthSummary summary, kHealthSummaryStat stat)
{
    return kHealthSummaryStat_Id(stat);
}

/**
 * Reports the instance value associated with the specified health stat. 
 * 
 * @public              @memberof kHealthSummary
 * @param   summary     Health summary.
 * @param   stat        Health stat.
 * @return              Stat instance value.
 */
kInlineFx(k32u) kHealthSummary_StatInstance(kHealthSummary summary, kHealthSummaryStat stat)
{
    return kHealthSummaryStat_Instance(stat);
}

/**
 * Provides a reference to the entry array associated with the specified stat.
 * 
 * All stat arrays contain the same number of log entries. Entries that were missing/inaccessible 
 * in the original log data are represented by k64S_NULL.
 *
 * @public              @memberof kHealthSummary
 * @param   summary     Health summary.
 * @param   stat        Health stat.
 * @return              Entry array; contains one k64s value per log entry.
 */
kInlineFx(kArrayList) kHealthSummary_EntryData(kHealthSummary summary, kHealthSummaryStat stat)
{
    return kHealthSummaryStat_Data(stat);
}

/**
 * Gets the log entry associated with the specified stat at the specified log entry index.
 * 
 * All stat arrays contain the same number of log entries. Entries that were missing/inaccessible 
 * in the original log data are represented by k64S_NULL.
 *
 * @public              @memberof kHealthSummary
 * @param   summary     Health summary.
 * @param   stat        Health stat.
 * @param   index       Entry index.
 * @param   value       Receives entry value.
 * @return              Operation status.
 */
kInlineFx(kStatus) kHealthSummary_EntryItem(kHealthSummary summary, kHealthSummaryStat stat, kSize index, k64s* value)
{
    kArrayList data = kHealthSummaryStat_Data(stat); 

    return kArrayList_ItemT(data, index, &value);
}

/**
 * Returns the log entry associated with the specified stat at the specified log entry index.
 * 
 * All stat arrays contain the same number of log entries. Entries that were missing/inaccessible 
 * in the original log data are represented by k64S_NULL.
 *
 * @public              @memberof kHealthSummary
 * @param   summary     Health summary.
 * @param   stat        Health stat.
 * @param   index       Entry index.
 * @return              Entry value.
 */
kInlineFx(k64s) kHealthSummary_EntryAt(kHealthSummary summary, kHealthSummaryStat stat, kSize index)
{
    kArrayList data = kHealthSummaryStat_Data(stat); 

    return kArrayList_AsT(data, index, k64s);
}

#endif
