/** 
 * @file    kFireSync/Health/kHealthProbe.h
 * @brief   Declares the kHealthProbe class and related types. 
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_HEALTH_PROBE_H
#define K_FIRESYNC_HEALTH_PROBE_H

#include <kFireSync/kNodeDef.h>
#include <kFireSync/Health/kHealthProbe.x.h>

/**
 * @class   kHealthProbe
 * @ingroup kFireSync-Health
 * @brief   Represents a health probe.
 * 
 * Health probes are intended to allow applications to report values that can be collected and presented 
 * as part of periodic health snapshots. Construction-time options can be used to specify the types of 
 * health snapshots in which a probe will be included: 
 * * kHEALTH_OPTION_STREAM: Live-streamed health updates. 
 * * kHEALTH_OPTION_PERSIST: Logged health updates. 
 *
 * Construction-time options only specify desired behavior; actual behavior is limited by the capabilities 
 * of the environment in which the application is running. E.g., if the environment does not support persistence, 
 * then the kHEALTH_OPTION_PERSIST flag will have no effect; probe construction will succeed as though this option
 * had not been specified.
 * 
 * For persistent probes, if the probe value was previously logged to non-volatile storage, then
 * the probe's initial value can be automatically loaded from the most recently-logged value at construction 
 * time by adding the kHEALTH_OPTION_RESTORE option. Alternatively, the most recently-logged value can 
 * be loaded to the probe's baseline property with the kHEALTH_OPTION_BASELINE option.
 * 
 * Health probes are identified within health snapshots via two distinct mechanisms. First, every 
 * health probe must have a unique string identifier (name). By convention, probe names consist of multiple 
 * forward-slash-separated components. E.g., "Camera/0/ExposureCount" might be used to represent the number 
 * of exposures on the first camera (camera 0) within a FireSync node. Second, health probes are also typically 
 * expected to have unique numeric identifiers. E.g., the probe with name "Camera/0/ExposureCount" is represented 
 * by the unique pair (id=kHEALTH_ID_CAMERA_EXPOSURE_COUNT, instance=0). Unique application-defined identifiers 
 * must have numeric values greater than kHEALTH_ID_APPLICATION_BASE; lower-numbered IDs are reserved for use 
 * by the FireSync platform. 
 * 
 * While it is necessary to provide a unique name for every probe, it is only strictly required to define unique numeric 
 * identifiers for persistent probes (kHEALTH_OPTION_PERSIST). For streamed probes, kHEALTH_ID_NULL can be used to 
 * create anonymous probes. The health statistics associated with anonymous probes will appear in streamed health 
 * snapshots, but may not be recognizable by software tools that mine data via numeric IDs. Most often, anonymous 
 * probes are used informally, e.g., to avoid allocating a temporary unique ID. 
 *
 * Internally, health probes maintain two fields that contribute to the current value reported by the probe: 
 * 'value' and 'baseline'. The value and baseline fields can be specified independently (e.g., via the SetValue and 
 * SetBaseline methods). The quantity reported by the probe's public Value method is the sum of these two internal fields. 
 * E.g., the sequence SetBaseline(5), SetValue(10) produces a reported Value() of 15. 
 * 
 * Use of baseline-related API features is strictly optional. The purpose of the baseline feature is to enable 
 * health indicators to express the current value of the probe in relation to a reference (baseline) value. 
 * If the kHEALTH_OPTION_BASELINE option is specified at construction time, then the most recent value from 
 * persistent storage will be used to initialize the probe baseline. This approach is most commonly used with persistent 
 * counters, relieving the client of the need to consider the value loaded from persistent storage when setting the 
 * current value of the probe. The previously-logged amount will automatically be added to the latest specified amount 
 * whenever the probe value is reported.
 * 
 * Health probes are associated with a health provider, which is the engine that periodically 
 * collects and summarizes health data. A default health provider may be installed by the application 
 * environment (e.g., physical, stand-alone FireSync node environments always include a default health 
 * provider). If not, or if a more specific collection provider is desired (e.g., in an application  
 * that instantiates multiple software-only nodes, where each node has its own health provider), use a 
 * kHealthProbe constructor that accepts a health provider reference as an input argument. 
 */
//typedef kObject kHealthProbe;            --forward-declared in kFsDef.x.h  

/** 
 * Constructs a kHealthProbe object with default provider and options.
 * 
 * Health probes created with this constructor will be configured for steaming only (kHEALTH_OPTION_STREAM). 
 * Refer to the kHealthProbe class summary for more information on construction options.
 *
 * @public                  @memberof kHealthProbe
 * @param   probe           Destination for the constructed object handle. 
 * @param   name            Descriptive label (e.g. "Hw/Event0/QueueCount").
 * @param   id              Health stat identifier (or kHEALTH_ID_NULL). 
 * @param   instance        Distinguishes multiple health stats with same id.
 * @param   initialValue    Initial probe value.
 * @param   allocator       Memory allocator (or kNULL for default). 
 * @return                  Operation status. 
 */
kInlineFx(kStatus) kHealthProbe_Construct(kHealthProbe* probe, const kChar* name, k32u id, k32u instance, k64s initialValue, kAlloc allocator)
{
    return xkHealthProbe_ConstructImpl(probe, name, id, instance, initialValue, kNULL, kHEALTH_OPTION_STREAM, allocator);
}

/** 
 * Constructs a kHealthProbe object with the specified provider and options.
 * 
 * Refer to the kHealthProbe class summary for more information on construction options.
 *
 * @public                  @memberof kHealthProbe
 * @param   probe           Destination for the constructed object handle. 
 * @param   name            Descriptive label (e.g. "Hw/Event0/QueueCount").
 * @param   id              Health stat identifier (or kHEALTH_ID_NULL). 
 * @param   instance        Distinguishes multiple health stats with same id.
 * @param   defaultValue    Default probe value (if the probe is persistent and the kHEALTH_OPTION_RESTORE is provided, this default will be ignored). 
 * @param   provider        Health provider (or kNULL for default). 
 * @param   options         Health probe options. 
 * @param   allocator       Memory allocator (or kNULL for default). 
 * @return                  Operation status. 
 */
kInlineFx(kStatus) kHealthProbe_Construct(kHealthProbe* probe, const kChar* name, k32u id, k32u instance, k64s defaultValue, kHealth provider, kHealthOption options, kAlloc allocator)
{
    return xkHealthProbe_ConstructImpl(probe, name, id, instance, defaultValue, provider, options, allocator);
}

/** 
 * Sets the current value of a health probe.
 * 
 * Note that the quantity reported by the Value accessor may differ from the value set via this 
 * method if a baseline value has been established. Refer to the kHealthProbe class summary for 
 * more information.
 *
 * @public              @memberof kHealthProbe
 * @param   probe       Probe object.  
 * @param   value       Health probe value.  
 * @return              Operation status. 
 */
kInlineFx(kStatus) kHealthProbe_SetValue(kHealthProbe probe, k64s value)
{
    return xkHealthProbe_VTable(probe)->VSetValue(probe, value);
}

/** 
 * Adds an amount to the current value of a health probe.
 *
 * @public              @memberof kHealthProbe
 * @param   probe       Probe object.  
 * @param   value       Amount to be added to health probe value.
 * @return              Operation status. 
 */
kInlineFx(kStatus) kHealthProbe_AddValue(kHealthProbe probe, k64s value)
{
    return xkHealthProbe_VTable(probe)->VAddValue(probe, value);
}

/** 
 * Sets the baseline value. 
 * 
 * Health probes can be configured with a baseline amount such that the Value accessor 
 * will return the sum of the amounts specified with SetBaseline and SetValue. E.g., the sequence
 * SetBaseline(5), SetValue(10) produces a reported Value of 15. 
 * 
 * The purpose of the baseline feature is to enable health indicator clients to specify the current 
 * value in relation to a reference (baseline) value. This approach is most commonly used with persistent 
 * counters. 
 * 
 * Refer to the kHealthProbe class summary for more information.
 *
 * @public              @memberof kHealthProbe
 * @param   probe       Probe object.  
 * @param   baseline    Baseline value.  
 * @return              Operation status. 
 */
kInlineFx(kStatus) kHealthProbe_SetBaseline(kHealthProbe probe, k64s baseline)
{
    return xkHealthProbe_VTable(probe)->VSetBaseline(probe, baseline);
}

/** 
 * Transfers the current probe value to the baseline. 
 * 
 * This method is equivalent to calling SetBaseline(Value()) followed by SetValue(0). 
 * 
 * Refer to the kHealthProbe class summary for more information on probe baselines.
 * 
 * @public              @memberof kHealthProbe
 * @param   probe       Probe object.  
 * @return              Operation status. 
 */
kInlineFx(kStatus) kHealthProbe_UpdateBaseline(kHealthProbe probe)
{
    return xkHealthProbe_VTable(probe)->VUpdateBaseline(probe);
}

/** 
 * Reports the current value of the health probe. 
 * 
 * The current value is the sum of the amounts previously specified via SetBaseline and 
 * SetValue. Refer to the kHealthProbe class summary for more information.
 *
 * @public              @memberof kHealthProbe
 * @param   probe       Probe object.  
 * @return              Value.
 */
kInlineFx(k64s) kHealthProbe_Value(kHealthProbe probe)
{
    return xkHealthProbe_VTable(probe)->VValue(probe);
}

/** 
 * Reports the name of the health probe. 
 *
 * @public              @memberof kHealthProbe
 * @param   probe       Probe object.
 * @return              Name of the health probe. 
 */
kInlineFx(const kChar*) kHealthProbe_Name(kHealthProbe probe)
{
    kObj(kHealthProbe, probe); 

    return obj->name;
}

/** 
 * Reports the ID of the health probe. 
 *
 * @public              @memberof kHealthProbe
 * @param   probe       Probe object.
 * @return              ID of the health probe. 
 */
kInlineFx(k32u) kHealthProbe_Id(kHealthProbe probe)
{
    kObj(kHealthProbe, probe); 

    return obj->id;
}

/** 
 * Reports the instance of the health probe. 
 *
 * @public              @memberof kHealthProbe
 * @param   probe       Probe object.
 * @return              Instance of the health probe. 
 */
kInlineFx(k32u) kHealthProbe_Instance(kHealthProbe probe)
{
    kObj(kHealthProbe, probe); 

    return obj->instance;
}

/** 
 * Reports the options associated with the indicator. 
 *
 * @public              @memberof kHealthProbe
 * @param   probe       Probe object.
 * @return              Options.
 */
kInlineFx(kHealthOption) kHealthProbe_Options(kHealthProbe probe)
{
    kObj(kHealthProbe, probe); 

    return obj->options;
}

#endif


