/** 
 * @file    kHealthLog.h
 * @brief   Declares the kHealthLog class and related types. 
 *
 * @internal
 * Copyright (C) 2019-2022 by LMI Technologies Inc. All rights reserved.
 */
#ifndef K_FIRESYNC_HEALTH_LOG_H
#define K_FIRESYNC_HEALTH_LOG_H

#include <kFireSync/kNodeDef.h>

/** 
 * Callback signature to determine whether the log should be pruned to reclaim space.
 * 
 * @param   receiver    Context pointer. 
 * @return              kTRUE if the log should be pruned to reclaim space (nearing capacity).
 */
typedef kBool (kCall* kHealthLogPruneFx)(kPointer receiver);

#include <kFireSync/Health/kHealthLog.x.h>

/**
 * @class   kHealthLog
 * @extends kObject
 * @ingroup kFireSync-Health
 * @brief   Implements persistent health logging.
 */
//typedef kObject kHealthLog;            --forward-declared in kFsDef.x.h  

/**
 * Constructs a kHealthLog object.
 * 
 * Log content is parsed during construction to establish previous stat values.
 *
 * @public                  @memberof kHealthLog
 * @param   log             Destination for the constructed object handle.
 * @param   directory       Log directory; contents will be managed by the log object.
 * @param   fileLimit       Maximum number of files to be retained in the log (or k32U_MAX for unlimited).
 * @param   sizeLimit       Maximum size of the log, in bytes (or k32U_MAX for unlimited).
 * @param   blockSize       Hint for ideal size for a single log file (or k32U_NULL for default). 
 * @param   allocator       Memory allocator (or kNULL for default).
 * @return                  Operation status.
 */
kFsFx(kStatus) kHealthLog_Construct(kHealthLog* log, const kChar* directory, k32u fileLimit, k32u sizeLimit, k32u blockSize, kAlloc allocator);

/**
* Adds or updates metadata for the specified stat.
* 
* This method must be called to notify the log about the existence of each health stat. It is only 
* necessary to call this method once for each id-instance pair. However, this method can be called 
* again at any time to update the health stat name.
* 
* This method is intended to be fast; no file I/O is performed. 
* 
* This method is thread-safe.
* 
* @public                   @memberof kHealthLog
* @param   log              Health log object.
* @param   id               Health stat id. 
* @param   instance         Health stat instance. 
* @param   name             Health stat name. 
* @param   shouldRestore    Should the previous value be reloaded as current value at construction time? 
* @param   value            Initial value. 
* @return                   Operation status.
*/
kFsFx(kStatus) kHealthLog_RegisterStat(kHealthLog log, kHealthId id, k32u instance, const kChar* name, kBool shouldRestore, k64s value);

/**
* Notes that a stat has gone offline. 
* 
* This method can optionally be called to note that a statistic is not currenty available. For non-restoring stats, this 
* will cause any future logged values to be reported as nulls until the stat is registered again. 
* 
* This method is intended to be fast; no file I/O is performed. 
* 
* This method is thread-safe.
* 
* @public                   @memberof kHealthLog
* @param   log              Health log object.
* @param   id               Health stat id. 
* @param   instance         Health stat instance. 
* @return                   Operation status.
*/
kFsFx(kStatus) kHealthLog_UnregisterStat(kHealthLog log, kHealthId id, k32u instance);

/**
* Gets the latest logged value of the specifid health stat.
*
* This method is intended to be fast; no file I/O is performed. 
* 
* This method is thread-safe.
* 
* @public               @memberof kHealthLog
* @param   log          Health log object.
* @param   id           Health indicator identifier.
* @param   instance     Health indicator instance.
* @param   value        Receives latest logged value of the specified statistic.
* @return               Operation status.
*/
kFsFx(kStatus) kHealthLog_FindStatValue(kHealthLog log, kHealthId id, k32u instance, k64s* value);

/**
* Marks the log to be cleared the next time that it is reloaded. 
*
* After the log is destroyed and reconstructed, the log will be empty. 
* 
* This method is thread-safe.
* 
* @public           @memberof kHealthLog
* @param   log      Health log object.
* @return           Operation status.
*/
kFsFx(kStatus) kHealthLog_Clear(kHealthLog log);

/**
* Adds a new snapshot to the health log.
* 
* This method is thread-safe.
* 
* @public               @memberof kHealthLog
* @param   log          Health log object.
* @param   statList     List of health statistics to be saved (kArrayList<kHealthStat>).
* @return               Operation status.
*/
kFsFx(kStatus) kHealthLog_CreateSnapshot(kHealthLog log, kArrayList statList);

/**
* Builds a log summary by scanning all snapshot files.
* 
* This method is thread-safe.
* 
* @public               @memberof kHealthLog
* @param   log          Health log object.
* @param   summary      Receives summary of log history.
* @return               Operation status.
*/
kFsFx(kStatus) kHealthLog_Summarize(kHealthLog log, kHealthSummary summary);

#endif
