/** 
 * @file    kProfileMsg.h
 * @brief   Declares the kProfileMsg type. 
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_PROFILE_MSG_H
#define K_FIRESYNC_PROFILE_MSG_H

#include <kFireSync/Data/kMsgSet.h>
#include <kFireSync/Data/kProfileMsg.x.h>

/**
 * @class   kProfileMsg
 * @extends kMsgSet
 * @ingroup kFireSync-Data
 * @brief   Represents a data message (FPGA LUT). 
 */
//typedef kMsgSet kProfileMsg;            --forward-declared in kFsDef.x.h

/** 
 * Constructs a kProfileMsg object.
 *
 * @public                  @memberof kProfileMsg
 * @param   msg             Destination for the constructed object handle.
 * @param   pointType       Range point type.
 * @param   pointCount      Number of points in the message.
 * @param   profileCount    Number of profiles in the message.
 * @param   hasSlices       Has slice info.
 * @param   allocator       Memory allocator (or kNULL for default).
 * @return                  Operation status. 
 */
kFsFx(kStatus) kProfileMsg_Construct(kProfileMsg* msg, kType pointType, kSize pointCount, kSize profileCount, kBool hasSlices, kAlloc allocator); 

/** 
 * Prepare attaching the message content to external buffer(s).
 *
 * @public                  @memberof kProfileMsg
 * @param   msg             ProfileMsg object. 
 * @param   pointType       Type for range point.
 * @param   pointCount      Number of points in the message.
 * @param   profileCount    Number of profiles in the message.
 * @return                  Operation status. 
 */
kFsFx(kStatus) kProfileMsg_BeginAttach(kProfileMsg msg, kType pointType, kSize pointCount, kSize profileCount);

/** 
 * Attaches one of the profiles to external buffer(s). 
 *
 * @public                  @memberof kProfileMsg
 * @param   msg             ProfileMsg object. 
 * @param   itemIndex       Item index.
 * @param   stamp           Stamp pointer array.
 * @param   pointData       Data pointer array.
 * @param   intensityData   Intensity pointer array.
 * @param   sliceData       Slice pointer array.
 * @return                  Operation status. 
 */
kFsFx(kStatus) kProfileMsg_Attach(kProfileMsg msg, kSize itemIndex, kStamp* stamp, void* pointData, void* intensityData, void* sliceData);

/** 
 * Copies the source message.
 *
 * @public              @memberof kProfileMsg
 * @param   msg         ProfileMsg object.
 * @param   source      Source msg to be copied.
 * @return              Operation status. 
 */
kFsFx(kStatus) kProfileMsg_Assign(kProfileMsg msg, kProfileMsg source);

/** 
 * Returns a pointer to the stamp specified by item index.
 *
 * @public              @memberof kProfileMsg
 * @param   msg         ProfileMsg object.  
 * @param   itemIndex   Item index.
 * @return              Returns a pointer to the stamp specified by item index.
 */
kInlineFx(kStamp*) kProfileMsg_StampAt(kProfileMsg msg, kSize itemIndex)
{
    return xkProfileMsg_RefAt(msg, itemIndex)->stamp; 
}

/** 
 * Returns a pointer to a specific point.
 *
 * @public              @memberof kProfileMsg
 * @param   msg         ProfileMsg object.  
 * @param   itemIndex   Item index.
 * @param   pointIndex  Profile point index.
 * @return              Returns a pointer to a specific point.
 */
kInlineFx(void*) kProfileMsg_PointAt(kProfileMsg msg, kSize itemIndex, kSize pointIndex)
{
    kObj(kProfileMsg, msg); 
    kProfileMsgRef* ref = xkProfileMsg_RefAt(msg, itemIndex);
    
    kAssert(pointIndex < obj->pointCount); 

    return kPointer_ItemOffset(ref->pointData, pointIndex, kType_Size(obj->pointType)); 
}

/** 
 * Returns a strongly-typed pointer to a specific point.
 *
 * A debug assertion will be raised if the size of the specified point data type is not equal to the 
 * size of the message point data type.
 * 
 * @relates                     kProfileMsg
 * @param   kProfileMsg_msg     ProfileMsg object.  
 * @param   kSize_itemIndex     Item index.
 * @param   kSize_pointIndex    Profile point index.
 * @param   T                   Point data type identifier (e.g., kPoint16u).
 * @return                      Returns a strongly-typed pointer to a specific point.
 */
#define kProfileMsg_PointAtT(kProfileMsg_msg, kSize_itemIndex, kSize_pointIndex, T) \
    kCast(T*, xkProfileMsg_PointAtT(kProfileMsg_msg, kSize_itemIndex, kSize_pointIndex, sizeof(T)))

/** 
 * Returns a pointer to a specific intensity value.
 *
 * @public              @memberof kProfileMsg
 * @param   msg         ProfileMsg object.  
 * @param   itemIndex   Item index.
 * @param   pointIndex  Intensity pixel index.
 * @return              Returns a pointer to a specific intensity value.
 */
kInlineFx(k8u*) kProfileMsg_IntensityAt(kProfileMsg msg, kSize itemIndex, kSize pointIndex)
{
    kObj(kProfileMsg, msg); 
    k8u* intensities = xkProfileMsg_RefAt(msg, itemIndex)->intensities;
    
    kAssert(pointIndex < obj->pointCount); 

    return &intensities[pointIndex]; 
}

/** 
 * Returns a pointer to a specific slice value.
 *
 * @public              @memberof kProfileMsg
 * @param   msg         ProfileMsg object.  
 * @param   itemIndex   Item index.
 * @param   pointIndex  Slice pixel index.
 * @return              Returns a pointer to a specific slice value.
 */
kInlineFx(k16u*) kProfileMsg_SliceAt(kProfileMsg msg, kSize itemIndex, kSize pointIndex)
{
    kObj(kProfileMsg, msg); 
    k16u* slices = xkProfileMsg_RefAt(msg, itemIndex)->slices;
    
    kAssert(pointIndex < obj->pointCount); 

    return &slices[pointIndex]; 
}

/** 
 * Returns the point type.
 *
 * @public              @memberof kProfile
 * @param   msg         ProfileMsg object.  
 * @return              Returns the point type.
 */
kInlineFx(kType) kProfileMsg_PointType(kProfileMsg msg)
{
    kObj(kProfileMsg, msg); 
    
    return obj->pointType; 
}

/** 
 * Returns the number of points in the message.
 *
 * @public              @memberof kProfile
 * @param   msg         ProfileMsg object.  
 * @return              Returns the number of points in the message.
 */
kInlineFx(kSize) kProfileMsg_PointCount(kProfileMsg msg)
{
    kObj(kProfileMsg, msg); 
    
    return obj->pointCount; 
}

/** 
 * Returns the number of profiles in the message.
 *
 * @public              @memberof kProfile
 * @param   msg         ProfileMsg object.  
 * @return              Returns the number of profiles in the message.
 */
kInlineFx(kSize) kProfileMsg_ProfileCount(kProfileMsg msg)
{
    kObj(kProfileMsg, msg); 
    
    return obj->profileCount; 
}

/** 
 * Returns kTRUE if message has slice info, otherwise kFALSE
 *
 * @public              @memberof kProfile
 * @param   msg         ProfileMsg object.  
 * @return              Returns if message has slice info.
 */
kInlineFx(kBool) kProfileMsg_HasSlices(kProfileMsg msg)
{
    kObj(kProfileMsg, msg); 

    return obj->hasSlices;
}

#endif
