/** 
 * @file    kCrc32.h
 * @brief   Declares the kCrc32 class. 
 *
 * @internal
 * Copyright (C) 2008-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_CRC32_H
#define K_FIRESYNC_CRC32_H

#include <kFireSync/kFsDef.h>

/** @relates kCrc32 @{ */
#define kCRC32_POLYNOMIAL_NORMAL    (0x04C11DB7)    ///Polyomial for standard CRC-32.
/** @} */

/**
 * @class       kCrc32
 * @extends     kObject
 * @ingroup     kFireSync-Data
 * @brief       Implements a CRC-32 generator. 
 */
//typedef kObject kCrc32;            --forward-declared in kFsDef.x.h 

/**
 * @struct      kCrc32Item
 * @extends     kValue
 * @ingroup     kFireSync-Data
 * @brief       Represents the state of an ongoing CRC-32 calculation.
 */
typedef struct kCrc32Item
{
    k32u remainder;         ///state of ongoing calculation
} kCrc32Item; 

/** 
 * Returns a kCrc32 object initialized with polynomial kCRC32_POLYNOMIAL_NORMAL. 
 *
 * @public      @memberof kCrc32
 * @return      CRC object.
 */
kFsFx(kCrc32) kCrc32_Normal();

/** 
 * Constructs a kCrc32 object.
 *
 * @public              @memberof kCrc32
 * @param   crc         Destination for the constructed object handle. 
 * @param   polynomial  Polynomial to use for CRC generation.
 * @param   allocator   Memory allocator (or kNULL for default). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kCrc32_Construct(kCrc32* crc, k32u polynomial, kAlloc allocator);

/** 
 * Begins a streaming CRC calculation.
 *
 * @public          @memberof kCrc32
 * @param   crc     CRC object.
 * @param   item    Represents state of streaming CRC calculation.
 * @return          Operation status. 
 */
kFsFx(kStatus) kCrc32_Begin(kCrc32 crc, kCrc32Item* item);

/** 
 * Updates a streaming CRC calculation.
 *
 * @public          @memberof kCrc32
 * @param   crc     CRC object.
 * @param   item    Represents state of streaming CRC calculation.
 * @param   data    Input data. 
 * @param   size    Size of input data. 
 * @return          Operation status. 
 */
kFsFx(kStatus) kCrc32_Update(kCrc32 crc, kCrc32Item* item, const void* data, kSize size);

/** 
 * Calculates current result of a streaming CRC calculation.
 *
 * @public          @memberof kCrc32
 * @param   crc     CRC object.
 * @param   item    Represents state of streaming CRC calculation.
 * @return          CRC result.
 */
kFsFx(k32u) kCrc32_Result(kCrc32 crc, const kCrc32Item* item);

/** 
 * Performs a CRC-32 calculation over the given data.
 * 
 * This method is equivalent to using the Begin/Update/Result methods.
 *
 * @public          @memberof kCrc32
 * @param   crc     CRC object.
 * @param   data    Input data. 
 * @param   size    Size of input data. 
 * @param   result  Receives the result of the calculation.
 * @return          Operation status. 
 */
kFsFx(kStatus) kCrc32_Calculate(kCrc32 crc, const void* data, kSize size, k32u* result); 

#include <kFireSync/Data/kCrc32.x.h>

#endif
