/** 
 * @file    kCompressedPhase.x.h
 *
 * @internal
 * Copyright (C) 2018-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_COMPRESSED_PHASE_X_H
#define K_FIRESYNC_COMPRESSED_PHASE_X_H

#define kCOMPRESSED_PHASE_WORD_BUFFER_CAPCITY           (64)      //Maximum number of 32-bit words read out from stream in one op; stack allocated, don't go overboard. 
#define kCOMPRESSED_PHASE_ITEM_BUFFER_CAPCITY           (256)     //Maximum number of items bit-packed and emitted as a single packet word.  

#define kCOMPRESSED_PHASE_PACKET_PACKET_SIZE            (4096)     //Default size of complete packet (intentionally consistent with PL). 
#define kCOMPRESSED_PHASE_PACKET_HEADER_SIZE            (16)       //Size of PL packet header.
#define kCOMPRESSED_PHASE_PACKET_CONTENT_CAPACITY       (kCOMPRESSED_PHASE_PACKET_PACKET_SIZE - kCOMPRESSED_PHASE_PACKET_HEADER_SIZE)      

#define kCOMPRESSED_PHASE_PACKETS_PER_BLOCK             (16)       //For internally allocated packets, number of packets per memory block.

typedef struct kCompressedPhaseStreamReadOutput
{
    kCompressedPhaseStream stream;   
    kArrayList data; 
} kCompressedPhaseStreamReadOutput;

kDeclareValueEx(kFs, kCompressedPhaseStreamReadItem, kValue)

//Used in stream read/write queues; delta queues only make use of one of the two fields
typedef struct kCompressedPhaseItem
{
    k32s v0; 
    k32s v1;
} kCompressedPhaseItem;

/*
 * kCompressedPhasePacket
 */

typedef struct kCompressedPhasePacket
{
    kPointer data;                              //Pointer to packet data.
    kSize size;                                 //Used bytes in this packet.
    kSize capacity;                             //Byte capacity of this packet.
} kCompressedPhasePacket;

kDeclareValueEx(kFs, kCompressedPhasePacket, kValue)

typedef kStatus(kCall* kCompressedPhaseStreamReadFx)(void* source, kSize sourceSize, void* destination);

typedef struct kCompressedPhaseStreamReadItem
{
    void* source;                              //Pointer to packet data.
    kSize sourceSize;                          //Packet length.
    void* destination;                         //Pointer to unpacked output.
    kCompressedPhaseStreamReadFx readFx;       //Packet parsing method.
} kCompressedPhaseStreamReadItem;

/*
 * kCompressedPhaseStream
 */

typedef struct kCompressedPhaseStreamClass
{   
    kObjectClass base; 

    kCompressedPhase parent;                    //Parent object.

    kCompressedPhaseStreamId id;                //Stream identifier.
    kCompressedPhaseEncoding encoding;          //Stream encoding style.

    kSize maxItemsPerPackedWordSet;             //How many entries are ideally read/written at once; based on encoding. 

    kArrayList packetList;                      //List of packets in stream -- kArrayList<kCompressedPhasePacket>.

    kSize packetIndex;                          //Current packet index (read mode).
    kByte* packetIt;                            //Currrent position within packet buffer (read or write modes).
    kByte* packetItEnd;                         //End position within packet buffer (read or write modes).

    kCompressedPhaseItem itemQueue[kCOMPRESSED_PHASE_ITEM_BUFFER_CAPCITY];   //Caches data for reading/writing (read or write modes).
    kSize itemQueueCount;                       //Count of cached items (read or write modes).
    kSize itemQueueEnd;                         //Total count of items cached on previous read (read mode).

    kSize itemCount;                            //Total items for this stream; typically provided with last packet for attached streams.
    kSize streamSize;                           //Total count of 'size' fields accross all packets.
} kCompressedPhaseStreamClass; 

kDeclareClassEx(kFs, kCompressedPhaseStream, kObject)

kFsFx(kStatus) xkCompressedPhaseStream_Construct(kCompressedPhaseStream* stream, kCompressedPhase parent, kCompressedPhaseStreamId id, kCompressedPhaseEncoding encoding, kAlloc allocator); 

kFsFx(kStatus) xkCompressedPhaseStream_Init(kCompressedPhaseStream stream, kType type, kCompressedPhase parent, kCompressedPhaseStreamId id, kCompressedPhaseEncoding encoding, kAlloc alloc); 
kFsFx(kStatus) xkCompressedPhaseStream_VRelease(kCompressedPhaseStream stream);

kFsFx(kSize) xkCompressedPhaseStream_VSize(kCompressedPhaseStream data);

kFsFx(kStatus) xkCompressedPhaseStream_Clear(kCompressedPhaseStream stream);

kFsFx(kStatus) xkCompressedPhaseStream_AttachStream(kCompressedPhaseStream stream, kCompressedPhaseStream source);

kFsFx(kStatus) xkCompressedPhaseStream_FinalizePacket(kCompressedPhaseStream stream);
kFsFx(kStatus) xkCompressedPhaseStream_AllocatePacket(kCompressedPhaseStream stream);
kFsFx(kStatus) xkCompressedPhaseStream_AdvancePacket(kCompressedPhaseStream stream);

kFsFx(kStatus) xkCompressedPhaseStream_WriteWord(kCompressedPhaseStream stream, const k32u* data, kSize partCount);

kFsFx(kStatus) xkCompressedPhaseStream_WriteDeltaBits(kCompressedPhaseStream stream, k32u bitCount);
kFsFx(kStatus) xkCompressedPhaseStream_WriteDelta(kCompressedPhaseStream stream, k32s delta);

kFsFx(kStatus) xkCompressedPhaseStream_Write12bFail1(kCompressedPhaseStream stream);
kFsFx(kStatus) xkCompressedPhaseStream_Write8bFail2(kCompressedPhaseStream stream);
kFsFx(kStatus) xkCompressedPhaseStream_Write24bFail2(kCompressedPhaseStream stream);
kFsFx(kStatus) xkCompressedPhaseStream_WriteFail(kCompressedPhaseStream stream, k32s index, k32s value);

kFsFx(kStatus) xkCompressedPhaseStream_Write8bNull(kCompressedPhaseStream stream);
kFsFx(kStatus) xkCompressedPhaseStream_WriteNull(kCompressedPhaseStream stream, k32s index, k32s count);

kFsFx(kStatus) xkCompressedPhaseStream_BeginRead(kCompressedPhaseStream stream);
kFsFx(kStatus) xkCompressedPhaseStream_ReadWords(kCompressedPhaseStream stream, k32u* data, kSize capacity, kSize* readCount);
kFsFx(kStatus) xkCompressedPhaseStream_FillDeltaReadCache(kCompressedPhaseStream stream, k32u bitsPerItem);

kInlineFx(kStatus) xkCompressedPhaseStream_ReadDelta(kCompressedPhaseStream stream, k32s* delta)
{
    kObj(kCompressedPhaseStream, stream); 
 
    if (obj->itemQueueCount == 0)
    {
        switch (obj->encoding)
        {
            case kCOMPRESSED_PHASE_ENCODING_4B_DELTA:       kCheck(xkCompressedPhaseStream_FillDeltaReadCache(stream, 4));       break;
            case kCOMPRESSED_PHASE_ENCODING_6B_DELTA:       kCheck(xkCompressedPhaseStream_FillDeltaReadCache(stream, 6));       break;  
            case kCOMPRESSED_PHASE_ENCODING_8B_DELTA:       kCheck(xkCompressedPhaseStream_FillDeltaReadCache(stream, 8));       break;
            case kCOMPRESSED_PHASE_ENCODING_10B_DELTA:      kCheck(xkCompressedPhaseStream_FillDeltaReadCache(stream, 10));      break;
            default:                                        return kERROR_FORMAT;
        }
    }

    *delta = obj->itemQueue[obj->itemQueueEnd - obj->itemQueueCount].v0;
    obj->itemQueueCount--;

    return kOK; 
}

kFsFx(kStatus) xkCompressedPhaseStream_Fill12bFail1ReadCache(kCompressedPhaseStream stream);
kFsFx(kStatus) xkCompressedPhaseStream_Fill8bFail2ReadCache(kCompressedPhaseStream stream);
kFsFx(kStatus) xkCompressedPhaseStream_Fill24bFail2ReadCache(kCompressedPhaseStream stream);

kInlineFx(kStatus) xkCompressedPhaseStream_ReadFail(kCompressedPhaseStream stream, k32s* index, k32s* value)
{
    kObj(kCompressedPhaseStream, stream); 
 
    if (obj->itemQueueCount == 0)
    {
        switch (obj->encoding)
        {
        case kCOMPRESSED_PHASE_ENCODING_12B_FAIL_1:       kCheck(xkCompressedPhaseStream_Fill12bFail1ReadCache(stream));       break;
        case kCOMPRESSED_PHASE_ENCODING_8B_FAIL_2:        kCheck(xkCompressedPhaseStream_Fill8bFail2ReadCache(stream));        break;
        case kCOMPRESSED_PHASE_ENCODING_24B_FAIL_2:       kCheck(xkCompressedPhaseStream_Fill24bFail2ReadCache(stream));       break;
        default:                                          return kERROR_FORMAT;
        }
    }
    
    *index = obj->itemQueue[obj->itemQueueEnd - obj->itemQueueCount].v0;
    *value = obj->itemQueue[obj->itemQueueEnd - obj->itemQueueCount].v1;
    
    obj->itemQueueCount--;

    return kOK;
}

kFsFx(kStatus) xkCompressedPhaseStream_FillNull8bReadCache(kCompressedPhaseStream stream);

kInlineFx(kStatus) xkCompressedPhaseStream_ReadNull(kCompressedPhaseStream stream, k32s* index, k32s* count)
{
    kObj(kCompressedPhaseStream, stream);

    if (obj->itemQueueCount == 0)
    {
        switch (obj->encoding)
        {
        case kCOMPRESSED_PHASE_ENCODING_8B_NULL:        kCheck(xkCompressedPhaseStream_FillNull8bReadCache(stream));       break;
        default:                                        return kERROR_FORMAT;
        }
    }

    *index = obj->itemQueue[obj->itemQueueEnd - obj->itemQueueCount].v0;
    *count = obj->itemQueue[obj->itemQueueEnd - obj->itemQueueCount].v1;

    obj->itemQueueCount--;

    return kOK;
}

kFsFx(kStatus) xkCompressedPhaseStream_EndWrite(kCompressedPhaseStream stream);

kFsFx(kSize) xkCompressedPhaseStream_PredictedStreamSize(kCompressedPhaseStream stream, kSize itemCount);
kFsFx(kStatus) xkCompressedPhaseStream_SetItemCount(kCompressedPhaseStream stream, kSize itemCount);

kFsFx(kStatus) xkCompressedPhaseStream_IncrementStreamSize(kCompressedPhaseStream stream, kSize size);

kInlineFx(kStatus) xkCompressedPhaseStream_Id(kCompressedPhaseStream stream)
{
    kObj(kCompressedPhaseStream, stream); 

    return obj->id; 
}

kInlineFx(kStatus) xkCompressedPhaseStream_Encoding(kCompressedPhaseStream stream)
{
    kObj(kCompressedPhaseStream, stream); 

    return obj->encoding; 
}

kInlineFx(kSize) xkCompressedPhaseStream_ItemCount(kCompressedPhaseStream stream)
{
    kObj(kCompressedPhaseStream, stream); 

    return obj->itemCount; 
}

kInlineFx(kArrayList) xkCompressedPhaseStream_PacketList(kCompressedPhaseStream stream)
{
    kObj(kCompressedPhaseStream, stream); 

    return obj->packetList; 
}

kInlineFx(kSize) xkCompressedPhaseStream_PacketCount(kCompressedPhaseStream stream)
{
    kObj(kCompressedPhaseStream, stream); 

    return kArrayList_Count(obj->packetList); 
}

kInlineFx(kCompressedPhasePacket*) xkCompressedPhaseStream_PacketAt(kCompressedPhaseStream stream, kSize index)
{
    kObj(kCompressedPhaseStream, stream); 

    return kArrayList_AtT(obj->packetList, index, kCompressedPhasePacket); 
}

/*
 * kCompressedPhase
 */

typedef struct kCompressedPhaseClass
{
    kObjectClass base; 

    kSize subframeIndex;                        //Subframe index. 
    kBool isFinalSubframe;                      //Does this object represent the final subframe in sequence?

    kSize subframeLength;                       //Uncompressed length of subframe, in pixels.
    kSize length[2];                            //Uncompressed dimension lengths, in pixels.
    k32u phasePrediction;                       //Phase prediction value.
   
    kArrayList packetBlocks;                    //List of memory blocks used to allocate packets; last block is currenty serving free packets -- kArrayList<kArray2<kByte>>.    
    kSize packetContentCapacity;                //Maximum content area in each packet (bytes). 
    kSize freePacketIndex;                      //Row index of next free packet (refers to last block in packetBlocks).
    
    kArrayList streams;                         //Stream contexts -- kArrayList<kCompressedPhaseStream>.

} kCompressedPhaseClass; 

kDeclareClassEx(kFs, kCompressedPhase, kObject)
        
kFsFx(kStatus) xkCompressedPhase_Init(kCompressedPhase data, kType type, kAlloc alloc); 
kFsFx(kStatus) xkCompressedPhase_VRelease(kCompressedPhase data); 

kFsFx(kStatus) xkCompressedPhase_VClone(kCompressedPhase data, kCompressedPhase other, kAlloc valueAlloc, kObject context); 
kFsFx(kSize) xkCompressedPhase_VSize(kCompressedPhase data);

kFsFx(kStatus) kCompressedPhase_IntensityBitDepths(kCompressedPhase data, k32u* deltaBitDepth, k32u* fail1BitDepth, k32u* fail2BitDepth, k32u* nullBitDepth);
kFsFx(kStatus) kCompressedPhase_PhaseBitDepths(kCompressedPhase data, k32u* deltaBitDepth, k32u* fail1BitDepth, k32u* fail2BitDepth, k32u* nullBitDepth);

kFsFx(kStatus) xkCompressedPhase_WriteDat6V1(kCompressedPhase data, kSerializer serializer);
kFsFx(kStatus) xkCompressedPhase_ReadDat6V1(kCompressedPhase data, kSerializer serializer);

kFsFx(kStatus) xkCompressedPhase_ClearPackets(kCompressedPhase data);

kFsFx(kStatus) xkCompressedPhase_AllocatePacket(kCompressedPhase data, kPointer* packetBuffer, kSize* packetCapacity);

kFsFx(kStatus) xkCompressedPhase_ExportStreamToCsv(kCompressedPhase data, kCompressedPhaseStream stream, const kChar* fileName); 

kFsFx(kStatus) xkCompressedPhase_ReserveBuffers(kSize subframeCapacity, kArrayList delta, kArrayList fail1, kArrayList fail2, kArrayList null, kObject* context, kAlloc allocator);

//TODO: clean up in FSS-1266
kFsFx(kStatus) kCompressedPhase_ReadPhaseStreamsEx(kCompressedPhase data, kArrayList delta, kArrayList fail1, kArrayList fail2, kArrayList null, kObject* context, int optHint);
kFsFx(kStatus) kCompressedPhase_ReadIntensityStreamsEx(kCompressedPhase data, kArrayList delta, kArrayList fail1, kArrayList fail2, kArrayList null, kObject* context, int optHint);

/*
 * kCompressedPhaseReader
 */

typedef struct kCompressedPhaseReaderClass
{
    kObjectClass base; 

    kCompressedPhaseStream stream;              //Read-mode stream; is attached to a stream within the kCompressedPhase dataset.
    
} kCompressedPhaseReaderClass; 

kDeclareClassEx(kFs, kCompressedPhaseReader, kObject)
        
kFsFx(kStatus) xkCompressedPhaseReader_Init(kCompressedPhaseReader reader, kType type, kAlloc alloc); 
kFsFx(kStatus) xkCompressedPhaseReader_VRelease(kCompressedPhaseReader reader); 

#endif
