/** 
 * @file    kTempControl.h
 * @brief   Declares the kTempControl class. 
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_LIGHT_TEMP_H
#define K_FIRESYNC_LIGHT_TEMP_H

#include <kFireSync/kNodeDef.h>

/**
 * @class   kTempControl
 * @extends kObject
 * @ingroup kFireSync-Client
 * @brief   Represents configuration for (PID) temperature regulation. 
 */
//typedef kObject kTempControl;        --forward-declared in kFsDef.x.h

/** 
 * Resets temperature regulation settings to defaults.
 *
 * @public          @memberof kTempControl
 * @param control   TempControl object.  
 * @return          Operation status. 
 */
kFsFx(kStatus) kTempControl_Clear(kTempControl control);

/** 
 * Enables or disables the temperature regulation. 
 * 
 * @public              @memberof kTempControl
 * @param   control     TempControl object.  
 * @param   enable      Specifies whether to enable or disable the temperature regulation.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kTempControl_Enable(kTempControl control, kBool enable);

/** 
 * Reports whether the temperature regulation is currently enabled. 
 *
 * @public              @memberof kTempControl
 * @param   control     TempControl object.  
 * @return              Returns whether the temperature regulation is currently enabled. 
 */
kFsFx(kBool) kTempControl_IsEnabled(kTempControl control);

/** 
 * Reports constraint and validity information for the Enabled setting. 
 *
 * @public              @memberof kTempControl
 * @param   control     TempControl object. 
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kTempControl_EnabledInfo(kTempControl control, kInfoBool* info);

/** 
 * Sets the setpoint. 
 * 
 * @public              @memberof kTempControl
 * @param   control     TempControl object.  
 * @param   setpoint    Setpoint value.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kTempControl_SetSetpoint(kTempControl control, k64f setpoint);

/** 
 * Reports the current setpoint setting. 
 *
 * @public              @memberof kTempControl
 * @param   control     TempControl object.  
 * @return              Returns the setpoint value. 
 */
kFsFx(k64f) kTempControl_Setpoint(kTempControl control);

/** 
 * Reports constraint and validity information for the Setpoint setting. 
 *
 * @public              @memberof kTempControl
 * @param   control     TempControl object. 
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kTempControl_SetpointInfo(kTempControl control, kInfo64f* info);

/** 
 * Sets the dt. 
 * 
 * @public              @memberof kTempControl
 * @param   control     TempControl object.  
 * @param   dt          Dt value.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kTempControl_SetDt(kTempControl control, k64f dt);

/** 
 * Reports the current dt setting. 
 *
 * @public              @memberof kTempControl
 * @param   control     TempControl object.  
 * @return              Returns the dt value. 
 */
kFsFx(k64f) kTempControl_Dt(kTempControl control);

/** 
 * Reports constraint and validity information for the Dt setting. 
 *
 * @public              @memberof kTempControl
 * @param   control     TempControl object. 
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kTempControl_DtInfo(kTempControl control, kInfo64f* info);

/** 
 * Sets the kp. 
 * 
 * @public              @memberof kTempControl
 * @param   control     TempControl object.  
 * @param   kp          Kp value.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kTempControl_SetKp(kTempControl control, k64f kp);

/** 
 * Reports the current kp setting. 
 *
 * @public              @memberof kTempControl
 * @param   control     TempControl object.  
 * @return              Returns the kp value. 
 */
kFsFx(k64f) kTempControl_Kp(kTempControl control);

/** 
 * Reports constraint and validity information for the Kp setting. 
 *
 * @public              @memberof kTempControl
 * @param   control     TempControl object. 
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kTempControl_KpInfo(kTempControl control, kInfo64f* info);

/** 
 * Sets the ki. 
 * 
 * @public              @memberof kTempControl
 * @param   control     TempControl object.  
 * @param   ki          Ki value.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kTempControl_SetKi(kTempControl control, k64f ki);

/** 
 * Reports the current ki setting. 
 *
 * @public              @memberof kTempControl
 * @param   control     TempControl object.  
 * @return              Returns the ki value. 
 */
kFsFx(k64f) kTempControl_Ki(kTempControl control);

/** 
 * Reports constraint and validity information for the Ki setting. 
 *
 * @public              @memberof kTempControl
 * @param   control     TempControl object. 
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kTempControl_KiInfo(kTempControl control, kInfo64f* info);

/** 
 * Sets the kd. 
 * 
 * @public              @memberof kTempControl
 * @param   control     TempControl object.  
 * @param   kd          Kd value.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kTempControl_SetKd(kTempControl control, k64f kd);

/** 
 * Reports the current kd setting. 
 *
 * @public              @memberof kTempControl
 * @param   control     TempControl object.  
 * @return              Returns the kd value. 
 */
kFsFx(k64f) kTempControl_Kd(kTempControl control);

/** 
 * Reports constraint and validity information for the Kd setting. 
 *
 * @public              @memberof kTempControl
 * @param   control     TempControl object. 
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kTempControl_KdInfo(kTempControl control, kInfo64f* info);

/** 
 * Sets the ks. 
 * 
 * @public              @memberof kTempControl
 * @param   control     TempControl object.  
 * @param   ks          Ks value.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kTempControl_SetKs(kTempControl control, k64f ks);

/** 
 * Reports the current ks setting. 
 *
 * @public              @memberof kTempControl
 * @param   control     TempControl object.  
 * @return              Returns the ks value. 
 */
kFsFx(k64f) kTempControl_Ks(kTempControl control);

/** 
 * Reports constraint and validity information for the Ks setting. 
 *
 * @public              @memberof kTempControl
 * @param   control     TempControl object. 
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kTempControl_KsInfo(kTempControl control, kInfo64f* info);

/** 
 * Sets the min. 
 * 
 * @public              @memberof kTempControl
 * @param   control     TempControl object.  
 * @param   min         Min value.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kTempControl_SetMin(kTempControl control, k64f min);

/** 
 * Reports the current min setting. 
 *
 * @public              @memberof kTempControl
 * @param   control     TempControl object.  
 * @return              Returns the min value. 
 */
kFsFx(k64f) kTempControl_Min(kTempControl control);

/** 
 * Reports constraint and validity information for the Min setting. 
 *
 * @public              @memberof kTempControl
 * @param   control     TempControl object. 
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kTempControl_MinInfo(kTempControl control, kInfo64f* info);

/** 
 * Sets the max. 
 * 
 * @public              @memberof kTempControl
 * @param   control     TempControl object.  
 * @param   max         Max value.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kTempControl_SetMax(kTempControl control, k64f max);

/** 
 * Reports the current max setting. 
 *
 * @public              @memberof kTempControl
 * @param   control     TempControl object.  
 * @return              Returns the max value. 
 */
kFsFx(k64f) kTempControl_Max(kTempControl control);

/** 
 * Reports constraint and validity information for the Max setting. 
 *
 * @public              @memberof kTempControl
 * @param   control     TempControl object. 
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kTempControl_MaxInfo(kTempControl control, kInfo64f* info);

/** 
 * Sets the integratorMax. 
 * 
 * @public                  @memberof kTempControl
 * @param   control         TempControl object.  
 * @param   integratorMax   IntegratorMax value.  
 * @return                  Operation status. 
 */
kFsFx(kStatus) kTempControl_SetIntegratorMax(kTempControl control, k64f integratorMax);

/** 
 * Reports the current integratorMax setting. 
 *
 * @public              @memberof kTempControl
 * @param   control     TempControl object.  
 * @return              Returns the integratorMax value. 
 */
kFsFx(k64f) kTempControl_IntegratorMax(kTempControl control);

/** 
 * Reports constraint and validity information for the IntegratorMax setting. 
 *
 * @public              @memberof kTempControl
 * @param   control     TempControl object. 
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kTempControl_IntegratorMaxInfo(kTempControl control, kInfo64f* info);

#include <kFireSync/Client/kTempControl.x.h>

#endif
