/** 
 * @file    kSystem.h
 * @brief   Declares the kSystem type. 
 *
 * @internal
 * Copyright (C) 2008-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_SYSTEM_H
#define K_FIRESYNC_SYSTEM_H

#include <kFireSync/kNodeDef.h>

/**
 * @class   kSystem
 * @extends kObject
 * @ingroup kFireSync-Client
 * @brief   Represents a collection of FireSync nodes (sensors).
 * 
 * The kSystem class supports node discovery, provides a variety of convenience functions to support multi-node 
 * operations, and supports receiving data and health messages from nodes.
 * 
 * When using the normal kSystem constructor (kSystem_Construct), an initial round of node discovery 
 * occurs before the constructor returns. After construction, any nodes that were detected can be accessed 
 * using the kSystem_NodeCount and kSystem_NodeAt methods. 
 * 
 * Changes to system topology (e.g., new sensor found, existing sensor goes offline) are detected asynchronously 
 * in a background thread. However, nodes are not automatically added to or removed from a system object. 
 * To learn whether topology changes have been detected, use the thread-safe kSystem_HasChanges method. To update 
 * the system object to reflect the latest changes, use the kSystem_Refresh method. 
 *
 * The kSystem_Connect, kSystem_Disconnect, kSystem_Verify, and kSystem_Apply methods can be used to perform 
 * common system-wide operations. 
 * 
 * The kSystem_Start and kSystem_Stop methods are typically used to start/stop data acquisition and processing 
 * activities across multiple nodes. The kSystem_SetDataHandler function can be used to register for asynchronous 
 * data delivery. Alternatively, the kSystem_ReceiveData function can be used to poll for new data messages. 
 * A variety of other methods are provided for more advanced start/stop operations. 
 * 
 * The kSystem_OpenHealth and kSystem_CloseHealth methods are typically used to open/close channels to receive 
 * periodic health messages from nodes. The kSystem_SetHealthHandler function can be used to register for asynchronous 
 * health delivery. Alternatively, the kSystem_ReceiveHealth function can be used to poll for new health messages. 
 * 
 * kSystem methods are not thread-safe, except where noted.
 * 
 * The following example illustrates how to connect, register for data, and start all nodes. 
 @code {.c}

kStatus MyApp_RunSensors(MyApp app)
{
    kSystem system = kNULL; 

    kTry
    {
        //construct a FireSync system object; discovers nodes in the network
        kTest(kSystem_Construct(&system, kNULL)); 

        //establish control connections to all sensors (best effort; does not fail if some cannot be connected)
        kTest(kSystem_Connect(system)); 

        //register for data messages
        kTest(kSystem_SetDataHandler(system, MyApp_OnData, app)); 

        //start all connected sensors
        kTest(kSystem_Start(system)); 

        //give the sensors a few seconds to run; any data messages will be received by the registered handler
        kTest(kThread_Sleep(5*1000*1000)); 

        //stop all connected sensors
        kTest(kSystem_Stop(system)); 
    }
    kFinally
    {
        kObject_Destroy(system); 

        kEndFinally();
    }

    return kOK; 
}

kStatus kCall MyApp_OnData(MyApp app, kSystem system, kMsgInfo msgInfo)
{
    //do something with the data
    //...

    kObject_Dispose(msgInfo); 

    return kOK; 
}

@endcode
 */
//typedef kObject kSystem;         //--forward-declared in kFsDef.x.h

/**
 * Reports the FireSync protocol version supported by this class. 
 * @return  Protocol version.
 */
kFsFx(kVersion) kSystem_ProtocolVersion(); 

/** 
 * Constructs a kSystem object with default discovery providers. 
 *
 * @public              @memberof kSystem
 * @param   system      Destination for the constructed object handle. 
 * @param   allocator   Memory allocator (or kNULL for default). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kSystem_Construct(kSystem* system, kAlloc allocator); 

/** 
 * Constructs a kSystem object without automatically starting discovery clients. 
 *
 * This constructor can be used to construct a system object without starting discovery, so that the system 
 * object can be configured before starting discovery clients.
 *
 * @public              @memberof kSystem
 * @param   system      Destination for the constructed object handle. 
 * @param   allocator   Memory allocator (or kNULL for default). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kSystem_ConstructEx(kSystem* system, kAlloc allocator); 

/** 
 * Constructs a kSystem object and adds a server node that is bound to the hardware platform singleton node. 
 *
 * This constructor can be used by node-server applications to simplify the processs of constructing 
 * and configuring a system object that is suited to hosting a server object that is bound to the underlying 
 * hardware platform singleton node. Specifically, this method ensures that the discovery port used by kSystem 
 * to discover other sensors matches the discovery port that the server node has been configured to use. 
 * 
 * This method is provided for the sake of convenience; if customization is required, the implementation of this 
 * method can be copied into user applications and tailored to suit the circumstances. 
 * 
 * This methods starts discovery clients and refreshes the system object before returning.
 * 
 * @public              @memberof kSystem
 * @param   system      Destination for the constructed system object handle (must be destroyed by caller).
 * @param   serverNode  Destination for the constructed server object handle (owned by system; not destroyed by caller).
 * @param   allocator   Memory allocator (or kNULL for default). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kSystem_ConstructServerSystem(kSystem* system, kSvNode* serverNode, kAlloc allocator);

/** 
 * Locates the specified discovery provider. 
 * 
 * @public                  @memberof kSystem
 * @param   system          kSystem object.  
 * @param   providerType    Desired type of provider.
 * @param   provider        Receives provider object, if found.
 * @return                  Operation status. 
 */
kFsFx(kStatus) kSystem_FindDiscoveryProvider(kSystem system, kType providerType, kDiscoveryProvider* provider); 

/** 
 * Adds a discovery provider to the system. 
 * 
 * Use this function in conjunction with the kSystem_ConstructEx function to add a custom 
 * discovery provider to the system before starting discovery. 
 * 
 * @public              @memberof kSystem
 * @param   system      kSystem object.  
 * @param   provider    Object that implements the kDiscoveryProvider interface (transfers ownership).
 * @return              Operation status. 
 */
kFsFx(kStatus) kSystem_AddDiscovery(kSystem system, kDiscoveryProvider provider); 

/** 
 * Removes a discovery provider from the system. 
 * 
 * Use this function in conjunction with the kSystem_ConstructEx function to remove a 
 * discovery provider from the system before starting discovery. 
 * 
 * @public                  @memberof kSystem
 * @param   system          kSystem object.  
 * @param   providerType    Type of provider to be removed.
 * @return                  Operation status. 
 */
kFsFx(kStatus) kSystem_RemoveDiscovery(kSystem system, kType providerType); 

/** 
 * Removes all discovery providers and adapters from the system. 
 * 
 * Use this function in conjunction with the kSystem_ConstructEx function to remove all default 
 * discovery providers. This approach can be used to provide a clean slate, after which any 
 * desired discovery providers can be added.
 * 
 * @public              @memberof kSystem
 * @param   system      kSystem object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kSystem_ClearDiscovery(kSystem system); 

/** 
 * Specifies a network adapter to be used for discovery. 
 * 
 * After composing a custom FireSync system using kSystem_ConstructEx and then adding any 
 * desired discovery providers, use this function to specify one or more network adapters
 * that should be used for discovery. 
 * 
 * By default, if no adapters are specified, any adapter may be used.
 *
 * @public                  @memberof kSystem
 * @param   system          kSystem object.
 * @param   interfaceName   Interface name.
 * @return                  Operation status. 
 */
 kFsFx(kStatus) kSystem_AddDiscoveryAdapter(kSystem system, const kChar* interfaceName);

/** 
 * Starts discovery client. 
 * 
 * After composing a custom FireSync system using the kSystem_ConstructEx and kSystem_AddDiscovery 
 * functions, use this function to start discovery services. 
 * 
 * @public              @memberof kSystem
 * @param   system      kSystem object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kSystem_StartDiscovery(kSystem system); 

/** 
 * Stops discovery client. 
 * 
 * This function can be used when debugging node firmware to prevent discovery timeouts from 
 * causing node disconnection.
 * 
 * @public              @memberof kSystem
 * @param   system      kSystem object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kSystem_StopDiscovery(kSystem system); 

/** 
 * Registers a callback function that can be used to filter discovered nodes.
 * 
 * If a filter callback will be used, it should be registered prior starting the discovery with
 * kStartDiscovery. The callback must not be changed while discovery is running.
 *
 * The callback function will be asynchronously invoked in a background thread. It can be called 
 * at any time, until the callback is unregistered. Various internal locks may be held by the thread 
 * that invokes the callback; accordingly, the implementation of the callback should avoid acquiring 
 * any locks that might result in deadlock.
 * 
 * @public              @memberof kSystem
 * @param   system      kSystem object.  
 * @param   filter      Callback function.  
 * @param   receiver    Receiver of the callback.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kSystem_SetNodeFilter(kSystem system, kNodeFilterFx filter, kPointer receiver);

/** 
 * Adds a server node to the system. 
 * 
 * This function creates a FireSync server node. Server nodes are created within the same operating system 
 * process as the caller. After adding a server node to the system, low-level server configuration can be 
 * performed by calling kSvNode methods. Note, the kSystem instance owns the server node; the handle returned 
 * by this method should not be destroyed by the caller. 
 * 
 * A server node is represented in the context of kSystem by a kNode object. After calling kSystem_AddServerNode 
 * to create the node, a kNode handle associated with the server can be accessed via kSystem and used to 
 * perform high-level configuration and control operations. 
 * 
 * The hardware node passed to this function is used as the backend for the server node. If the caller 
 * constructed the hardware node, it can be destroyed by the caller after this method returns. The server node
 * will manipulate the hardware node's reference count as needed to control its lifetime.  
 * 
 * Note: kSystem_Refresh is called automatically by this method. 
 * 
 * @public                  @memberof kSystem
 * @param   system          kSystem object.  
 * @param   hardwareNode    Hardware node that provides underlying platform services (see remarks on ownership).  
 * @param   serverNode      Receives server node handle, which can be used for additional configuration  (see remarks on ownership).  
 * @return                  Operation status. 
 * @see                     kHxNode, kSvNode, kSystem_RemoveServerNode
 */
kFsFx(kStatus) kSystem_AddServerNode(kSystem system, kHxNode hardwareNode, kSvNode* serverNode); 

/** 
 * Removes a server node from the system. 
 * 
 * Note: kSystem_Refresh is called automatically by this method. 
 * 
 * @public              @memberof kSystem
 * @param   system      kSystem object.  
 * @param   nodeId      Server node id.    
 * @return              Operation status. 
 * @see                 kSystem_AddServerNode, kSystem_RemoveAllServerNodes
 */
kFsFx(kStatus) kSystem_RemoveServerNode(kSystem system, k32u nodeId); 

/** 
 * Removes all server nodes from the system. 
 * 
 * Note: kSystem_Refresh is called automatically by this method. 
 * 
 * @public              @memberof kSystem
 * @param   system      kSystem object.  
 * @return              Operation status. 
 * @see                 kSystem_AddServerNode, kSystem_RemoveServerNode
 */
kFsFx(kStatus) kSystem_RemoveAllServerNodes(kSystem system);

/** 
 * Installs a health provider that can be used to generate health statistics for the local host. 
 *
 * This method can be used to enable kSystem to report rudimentary health statistics for the 
 * local host computer. This method should not normally be used within a sensor.
 * 
 * If a host health provider has been set, then health messages will be generated when the host 
 * health channel is opened, using either kSystem_OpenHostHealth or kSystem_OpenHealth. The health 
 * provider should be registered prior to opening health connections and cannot be changed while 
 * health connections are open.
 * 
 * @public              @memberof kSystem
 * @param   system      System object.  
 * @param   provider    Health provider (or kNULL to unregister). 
 * @param   id          Node identifier to use for host health messages (e.g., kHEALTH_MSG_SOURCE_HOST) 
 * @return              Operation status. 
 */
kFsFx(kStatus) kSystem_SetHostHealthProvider(kSystem system, kHealth provider, k32u id);

/** 
 * Reports whether the system has changes that require a refresh. 
 * 
 * Nodes can undergo autonomous state changes that require client state to be refreshed 
 * (e.g. sensor goes offline). The kSystem_HasChanges function can be used to determine 
 * if such changes have occurred.  
 * 
 * The kSystem_HasChanges function does not perform communcation, and consequently, will not 
 * require the caller to block for a long duration. If changes are detected, the kSystem_Refresh 
 * function can be called to resolve the changes. 
 * 
 * This method is thread-safe.
 *
 * @public              @memberof kSystem
 * @param   system      kSystem object.  
 * @return              kTRUE if the system has changes.
 */
kFsFx(kBool) kSystem_HasChanges(kSystem system);

/** 
 * Reports system changes that have occurred. 
 * 
 * Nodes can undergo autonomous state changes that require client state to be refreshed 
 * (e.g. sensor goes offline). The kSystem_Changes function can be used to determine 
 * the types of such changes that may have occurrred.
 * 
 * The kSystem_Changes function does not perform communcation, and consequently, will not 
 * require the caller to block for a long duration. 
 *
 * If topology changes are detected, the kSystem_Refresh function should be called to resolve 
 * the changes. If configuration changes are detected, kSystem_Refresh can optionally be used to 
 * resolve the changes globally; alternatively, kNode_Refresh can be used to resolve the 
 * changes on a per-node basis.
 * 
 * This method is thread-safe.
 *
 * @public              @memberof kSystem
 * @param   system      kSystem object.  
 * @return              Bitset describing system changes.
 */
kFsFx(kSystemChange) kSystem_Changes(kSystem system);

/** 
 * Updates client state to reflect any changes that have occurred in the node network. 
 * 
 * Nodes can undergo autonomous state changes that require client state to be refreshed 
 * (e.g. sensor goes offline). The kSystem_Refresh function resynchronizes client state 
 * with the current state of the sensor network. 
 * 
 * Calling this function may destroy or modify local node objects. The kSystem_HasChanges 
 * function can be used prior to calling kSystem_Refresh, to determine whether a refresh is 
 * needed.
 * 
 * @public              @memberof kSystem
 * @param   system      kSystem object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kSystem_Refresh(kSystem system);

/** 
 * Updates client state to reflect any changes that have occurred in the node network. 
 * 
 * Nodes can undergo autonomous state changes that require client state to be refreshed 
 * (e.g. sensor goes offline). The kSystem_Refresh function resynchronizes client state 
 * with the current state of the sensor network. 
 * 
 * Calling this function may destroy or modify local node objects. The kSystem_HasChanges 
 * function can be used prior to calling kSystem_Refresh, to determine whether a refresh is 
 * needed.
 * 
 * @public              @memberof kSystem
 * @param   system      kSystem object.  
 * @param   forceAll    Force node refresh.  
 * @param   removed     Populated with nodes removed from the system. 
 * @param   added       Populated with nodes added to the system. 
 * @param   refreshed   Populated with nodes that were refreshed. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kSystem_RefreshEx(kSystem system, kBool forceAll, kArrayList removed, kArrayList added, kArrayList refreshed); 

/** 
 * Starts broadcasting rescue request messages.
 * 
 * Some nodes have a feature that enables them to listen for special "force rescue" messages during 
 * boot up. If that feature is present and enabled, and a "force rescue" message is received at boot 
 * time, the node will load its rescue program instead of its primary program. This is a development-time 
 * feature that can be used to revive devices when the primary program repeatedly crashes on boot. 
 *
 * Refer to the kNode_EnableRescueWait documentation for information on how to enable this feature 
 * on nodes that support it. 
 * 
 * @public              @memberof kSystem
 * @param   system      kSystem object.  
 * @return              Operation status. 
 * @see                 kSystem_StopRescue, kNode_EnableRescueWait
 */
kFsFx(kStatus) kSystem_StartRescue(kSystem system);

/** 
 * Stops broadcasting rescue request messages.
 * 
 * @public              @memberof kSystem
 * @param   system      kSystem object.  
 * @return              Operation status. 
 * @see                 kSystem_StartRescue
 */
kFsFx(kStatus) kSystem_StopRescue(kSystem system);

/** 
 * Attempts to connects to all discovered nodes. 
 * 
 * This function is equivalent to calling kNode_Connect for each node in the system. 
 * 
 * @public              @memberof kSystem
 * @param   system      kSystem object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kSystem_Connect(kSystem system); 

/** 
 * Disconnects from all nodes. 
 * 
 * This function is equivalent to calling kNode_Disconnect for each node in the system. 
 * 
 * @public              @memberof kSystem
 * @param   system      kSystem object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kSystem_Disconnect(kSystem system); 

/** 
 * Checks the validity of all system settings. 
 *
 * Use kSystem_ErrorCount/kSystem_ErrorAt to access the list of errors. If kSystem_ErrorCount reports
 * zero errors, then all validation checks passed. 
 * Use kSystem_WarningCount/kSystem_WarningAt to access the list of warnings. Warning may cause run-time
 * error under certain circumstance but not always.
 *
 * @public              @memberof kSystem
 * @param   system      System object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kSystem_Verify(kSystem system);

/** 
 * Returns the number of errors that were detected when the system was last verified. 
 *
 * @public              @memberof kSystem
 * @param   system      System object.  
 * @return              Returns the number of errors that were detected when the system was last verified. 
 */
kFsFx(kSize) kSystem_ErrorCount(kSystem system);

/** 
 * Gets the verification error string at the specified index. 
 *
 * @public              @memberof kSystem
 * @param   system      System object.  
 * @param   index       Index of the error.   
 * @param   error       Buffer to accept the null-terminated error string.   
 * @param   capacity    Size, in characters, of the string buffer.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kSystem_ErrorAt(kSystem system, kSize index, kChar* error, kSize capacity);

/** 
 * Returns the number of warnings that were detected when the system was last verified. 
 *
 * @public              @memberof kSystem
 * @param   system      System object.  
 * @return              Returns the number of warnings that were detected when the system was last verified. 
 */
kFsFx(kSize) kSystem_WarningCount(kSystem system);

/** 
 * Gets the verification warning string at the specified index. 
 *
 * @public              @memberof kSystem
 * @param   system      System object.  
 * @param   index       Index of the warning.   
 * @param   warning       Buffer to accept the null-terminated warning string.   
 * @param   capacity    Size, in characters, of the string buffer.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kSystem_WarningAt(kSystem system, kSize index, kChar* warning, kSize capacity);

/** 
 * Sends current settings to all modified nodes. 
 *
 * This function can be called only when nodes are in the Ready state.
 *
 * @public              @memberof kSystem
 * @param   system      kSystem object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kSystem_Apply(kSystem system);

/** 
 * Checks for errors and applies settings if no errors are detected. 
 *
 * This is a convenience method that combines kSystem_Verify and kSystem_Apply. Refer to the documentation for those
 * methods for more information.
 *
 * @public              @memberof kNode
 * @param   system      System object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kSystem_VerifyAndApply(kSystem system);

/** 
 * Sets the per-node network buffer size for data channels.
 * 
 * This setting determines the size of the socket buffer used for node data channels. The default 
 * value works well for most networks. The following helpful constants are available: 
 * * kSYSTEM_DATA_NET_BUFFER_SIZE_SMALL_SYSTEM
 * * kSYSTEM_DATA_NET_BUFFER_SIZE_MEDIUM_SYSTEM
 * * kSYSTEM_DATA_NET_BUFFER_SIZE_LARGE_SYSTEM
 * 
 * Existing data connections must be closed and re-opened in order for this setting to take effect. 
 * 
 * @public              @memberof kSystem
 * @param   system      kSystem object.  
 * @param   size        Per-node network buffer size (bytes). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kSystem_SetDataNetBufferSize(kSystem system, kSize size); 

/** 
 * Gets the per-node network buffer size for data channels.
 * 
 * @public              @memberof kSystem
 * @param   system      kSystem object.  
 * @return              Per-node network buffer size (bytes). 
 */
kFsFx(kSize) kSystem_DataNetBufferSize(kSystem system); 

/** 
 * Registers a callback function that can be used to receive data messages.
 * 
 * If a data callback will be used, it should be registered prior to opening any 
 * sensor data connections. The callback cannot be changed while sensors are sending data.
 *
 * The callback function will be asynchronously invoked in a background thread. It can be called 
 * at any time, until the callback is unregistered. Various internal locks may be held by the thread 
 * that invokes the callback; accordingly, the implementation of the callback should avoid acquiring 
 * any locks that might result in deadlock. In particular, this includes any locks that might be 
 * held by the application when opening or closing data connections. If in doubt, the safest course 
 * is to enqueue the received data for processing by another application thread, or use 
 * the kSystem_ReceiveData method instead of kSystem_SetDataHandler.
 * 
 * Message objects (kMsgInfo) received by the registered callback function should be disposed (kObject_Dispose)
 * when no longer needed. Data messages can be retained beyond the duration of the callback. 
 * 
 * Callbacks can be used as an alternative to kSystem_ReceiveData, but cannot be used 
 * at the same time (registering a callback will prevent kSystem_ReceiveData from working). 
 *
 * @public              @memberof kSystem
 * @param   system      System object.  
 * @param   function    Data handler function, or kNULL to unregister the callback.  
 * @param   receiver    User-defined context pointer, to be passed to the callback function.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kSystem_SetDataHandler(kSystem system, kCallbackFx function, kPointer receiver);

/** 
 * Waits until at least one data message has been received and then returns the message.
 *
 * This function can be used as an alternative to registering a data callback.
 *
 * Messages receieved via this method should be disposed (kObject_Dispose) when no longer needed. 
 *
 * This method is thread-safe.
 *
 * @public              @memberof kSystem
 * @param   system      System object.   
 * @param   msg         The message object that was received (see note).  
 * @param   timeout     Timeout duration, in microseconds.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kSystem_ReceiveData(kSystem system, kMsgInfo* msg, k64u timeout);

/**
 * Clears all data from the data queue.
 *
 * This method is thread-safe.
 *
 * @public             @memberof kSystem
 * @param   system     System object.
 * @return             Operation status.
 */
kFsFx(kStatus) kSystem_ClearData(kSystem system);

/**
 * Sets the maximum amount of data retained by the data queue.
 *
 * This function controls the maximum amount of data that can be locally enqueued for delivery through 
 * the kSystem_ReceiveData function.
 * 
 * Is it safe to call this function at any time, including during acquisition. Excess
 * messages will be dropped from the queue, if needed.
 *
 * This method is thread-safe. 
 *
 * @public             @memberof kSystem
 * @param   system     System object.
 * @param   size       Maximum total size of all data items in queue, in bytes.
 * @return             Operation status.
 */
kFsFx(kStatus) kSystem_SetDataQueueSize(kSystem system, kSize size);

/**
 * Reports the maximum total data size of all items in the data queue.
 *
 * This method is thread-safe.
 *
 * @public             @memberof kSystem
 * @param   system     System object.
 * @return             Maximum total recursive data size, in bytes.
 */
kFsFx(kSize) kSystem_DataQueueSize(kSystem system);

/**
 * Opens data connections to all connected nodes. 
 * 
 * This operation is performed automatically when using functions such as kSystem_Start. 
 * However, this function can be used in conjuction with functions such as kSystem_StartNodes for finer 
 * control.
 *
 * @public              @memberof kSystem
 * @param   system      System object.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kSystem_OpenData(kSystem system);

/**
 * Ends data collection from all connected nodes until the next time that nodes are started/resumed. 
 * 
 * It is not necessary to explicitly call this method when using high-level methods such as kSystem_Start/kSystem_Stop. 
 * This method is intended to be used in conjunction with methods such as kSystem_StartNodes/kSystem_StopNodes, 
 * for finer control.
 *
 * This method can optionally be called before kSystem_StopNodes to ensure that leftover
 * data messages from one session (e.g., between StartNodes and StopNodes) are not delivered during the next 
 * session. This method is useful when data connections are not closed between sessions.
 * 
 * When this method is used, the delivery of data messages is guaranteed to stop sometime between when this 
 * method is called and when kSystem_StopNodes completes. 
 * 
 * @public              @memberof kSystem
 * @param   system      System object.
 * @return              Operation status.
*/
kFsFx(kStatus) kSystem_EndDataSession(kSystem system);

/** 
 * Closes data connections to all connected nodes. 
 * 
 * @public              @memberof kSystem
 * @param   system      System object.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kSystem_CloseData(kSystem system); 

/** 
 * Registers a callback function that can be used to receive health messages.
 *
 * If a health callback will be used, it should be registered prior to opening any 
 * sensor health connections. The callback cannot be changed while sensors are sending health messages.
 *
 * The callback function will be asynchronously invoked in a background thread. It can be called 
 * at any time, until the callback is unregistered. Various internal locks may be held by the thread 
 * that invokes the callback; accordingly, the implementation of the callback should avoid acquiring 
 * any locks that might result in deadlock. In particular, this includes any locks that might be 
 * held by the application when opening or closing health connections. If in doubt, the safest course 
 * is to enqueue the received health messages for processing by another application thread, or use 
 * the kSystem_ReceiveHealth method instead of kSystem_SetHealthHandler.
 * 
 * Message objects (kHealthMsg) received by the registered callback function should be disposed (kObject_Dispose)
 * when no longer needed. Health messages can be retained beyond the duration of the callback. 
 * 
 * Callbacks can be used as an alternative to kSystem_ReceiveHealth, but cannot be used 
 * at the same time (registering a callback will prevent kSystem_ReceiveHealth from working). 

 * @public              @memberof kSystem
 * @param   system      System object.  
 * @param   function    Health handler function, or kNULL to unregister the callback.  
 * @param   receiver    User-defined context pointer, to be passed to the callback function.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kSystem_SetHealthHandler(kSystem system, kCallbackFx function, kPointer receiver);

/** 
 * Waits until at least one health message has been received and then returns the message.
 *
 * This function can be used as an alternative to registering a health callback.
 * 
 * Messages receieved via this method should be disposed (kObject_Dispose) when no longer needed. 
 *
 * This method is thread-safe.
 *
 * @public              @memberof kSystem
 * @param   system      System object.   
 * @param   info        The health message object that was received.  
 * @param   timeout     Timeout duration, in microseconds.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kSystem_ReceiveHealth(kSystem system, kHealthMsg* info, k64u timeout);

/**
 * Clears all health messages from the health queue.
 *
 * This method is thread-safe.
 *
 * @public                 @memberof kSystem
 * @param   system         System object.
 * @return                 Operation status.
 */
kFsFx(kStatus) kSystem_ClearHealth(kSystem system);

/** 
 * Opens health channels to all connected nodes. 
 *
 * @public              @memberof kSystem
 * @param   system      System object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kSystem_OpenHealth(kSystem system);

/** 
 * Closes all health channels. 
 *
 * @public              @memberof kSystem
 * @param   system      System object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kSystem_CloseHealth(kSystem system); 

/** 
 * Opens channel to local host health provider. 
 *
 * @public              @memberof kSystem
 * @param   system      System object.  
 * @return              Operation status. 
 * @see                 kSystem_SetHostHealthProvider, kSystem_CloseHealth
 */
kFsFx(kStatus) kSystem_OpenHostHealth(kSystem system); 

/** 
 * Closes channel to local host health provider.
 *
 * @public              @memberof kSystem
 * @param   system      System object.  
 * @return              Operation status. 
 * @see                 kSystem_OpenHostHealth
 */
kFsFx(kStatus) kSystem_CloseHostHealth(kSystem system); 

/** 
 * Opens data connections and starts all connected sensors. 
 * 
 * @public              @memberof kSystem
 * @param   system      System object.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kSystem_Start(kSystem system); 

/** 
 * Ends data sessions and stops all connected sensors. 
 * 
 * This function does not automatically close data connections to sensors. Data connections are retained 
 * in order to improve cycle times for repeated start/stop operations. The kSystem_CloseData method can 
 * be used to explicitly close data connections, if desired. 
 *  
 * @public              @memberof kSystem
 * @param   system      System object.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kSystem_Stop(kSystem system); 

/** 
 * Prepares all connected nodes to be started. 
 * 
 * This operation is performed automatically when using kSystem_Start. However, this function can be used in 
 * conjuction with functions such as kSystem_EngageNodes and kSystem_OpenData for finer control. 
 *
 * @public              @memberof kSystem
 * @param   system      System object.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kSystem_StartNodes(kSystem system); 

/** 
 * Stops all connected nodes. 
 * 
 * This operation is performed automatically when using kSystem_Stop. However, this function can be used in 
 * conjuction with functions such as kSystem_CloseData for finer control. 
 *
 * @public              @memberof kSystem
 * @param   system      System object.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kSystem_StopNodes(kSystem system); 

/** 
 * Engages all connected nodes (finalizes a start or resume operation). 
 * 
 * This operation is performed automatically when using kSystem_Start. However, this function 
 * can be used in conjuction with kSystem_StartNodes for finer control. 
 * 
 * The startTime and/or startEncoder arguments can be used to request that event managers using continuous control 
 * mode (kEVENT_CONTROL_CONTINUOUS) start at a specific time or location. If these arguments are omitted (null) 
 * event managers will still be synchronized, but may start at slightly different times or locations. 
 *
 * @public                  @memberof kSystem
 * @param   system          System object.   
 * @param   startTime       System start time (FS us), or k64U_NULL for auto-synchronized start.
 * @param   startEncoder    System start encoder (ticks), or k64S_NULL for auto-synchronized start. 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kSystem_EngageNodes(kSystem system, k64u startTime, k64s startEncoder); 

/** 
 * Reports the number of nodes in the system. 
 *
 * @public              @memberof kSystem
 * @param   system      System object.   
 * @return              Node count. 
 */
kFsFx(kSize) kSystem_NodeCount(kSystem system); 

/** 
 * Gets the node at the specified index. 
 *
 * @public              @memberof kSystem
 * @param   system      System object.   
 * @param   index       Node index. 
 * @return              Node object. 
 */
kFsFx(kNode) kSystem_NodeAt(kSystem system, kSize index); 

/** 
 * Finds the node with the specified id. 
 *
 * @public              @memberof kSystem
 * @param   system      System object.   
 * @param   id          Node id (serial number). 
 * @param   node        Optionally receives node object (if found). 
 * @return              Operation status (kERROR_NOT_FOUND if node not found). 
 */
kFsFx(kStatus) kSystem_FindNode(kSystem system, k32u id, kNode* node);

#include <kFireSync/Client/kSystem.x.h>

#endif
