/** 
 * @file    kSerialOut.h
 * @brief   Declares the kSerialOut type. 
 *
 * @internal
 * Copyright (C) 2008-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_SERIAL_OUT_H
#define K_FIRESYNC_SERIAL_OUT_H

#include <kFireSync/kNodeDef.h>

/**
 * @class   kSerialOut
 * @extends kObject
 * @ingroup kFireSync-Client
 * @brief   Represents serial configuration.
 */
//typedef kObject kSerialOut;        --forward-declared in kFsDef.x.h

/** 
 * Gets the id of the output. 
 *
 * @public              @memberof kSerialOut
 * @param   serial      Serial object.  
 * @return              Returns the unique id of this serial. 
 */
kFsFx(kSize) kSerialOut_Id(kSerialOut serial);

/** 
 * Resets serial output mode settings to defaults. 
 *
 * @public              @memberof kSerialOut
 * @param   serial      Serial object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kSerialOut_Clear(kSerialOut serial);

/** 
 * Enables or disables the serial. 
 *
 * Disabled serials are not checked during verification, and do not respond to 
 * Start/Pause/Resume/Stop commands. 
 *
 * @public              @memberof kSerialOut
 * @param   serial      Serial object.  
 * @param   enable      Specifies whether to enable or disable the serial.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kSerialOut_Enable(kSerialOut serial, kBool enable);

/** 
 * Reports whether the serial is currently enabled. 
 *
 * @public              @memberof kSerialOut
 * @param   serial      Serial object.  
 * @return              Returns whether the serial is currently enabled. 
 */
kFsFx(kBool) kSerialOut_IsEnabled(kSerialOut serial);

/** 
 * Sets the control mode of the serial.
 *
 * For control modes that suppport multiple sources (e.g. kSERIAL_OUT_CONTROL_TRIGGERED_BY_EVENT), 
 * use the kSerialOut_SetControlId function to specify a specific source.
 *
 * @public              @memberof kSerialOut
 * @param   serial      Serial object.  
 * @param   type        Serial control mode.
 * @return              Operation status. 
 */
kFsFx(kStatus) kSerialOut_SetControl(kSerialOut serial, kSerialOutControl type);

/** 
 * Reports the control mode of the serial. 
 *
 * @public              @memberof kSerialOut
 * @param   serial      Serial object.  
 * @return              Returns the control mode of the serial. 
 */
kFsFx(kSerialOutControl) kSerialOut_Control(kSerialOut serial);

/** 
 * Reports constraint and validity information for the Control setting. 
 *
 * @public              @memberof kSerialOut
 * @param   serial      Serial object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kSerialOut_ControlInfo(kSerialOut serial, kInfoBits* info);

/** 
 * Sets the control source id for a serial. 
 *
 * For control modes that suppport multiple sources (e.g. kSERIAL_OUT_CONTROL_TRIGGERED_BY_EVENT), 
 * this function cam be used to specify a specific source id.
 *
 * @public              @memberof kSerialOut
 * @param   serial      Serial object.  
 * @param   id          Specifies the control source id.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kSerialOut_SetControlId(kSerialOut serial, k32u id);

/** 
 * Gets the control source id for a serial.
 *
 * @public              @memberof kSerialOut
 * @param   serial      Serial object.  
 * @return              Control source id.
 */
kFsFx(k32u) kSerialOut_ControlId(kSerialOut serial);

/** 
 * Reports constraint and validity information for the ControlId setting. 
 *
 * @public              @memberof kSerialOut
 * @param   serial      Serial object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kSerialOut_ControlIdInfo(kSerialOut serial, kInfo* info);

/** 
 * Sets the mode for a serial. 
 *
 * @public              @memberof kSerialOut
 * @param   serial      Serial object.  
 * @param   mode        Specifies the mode.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kSerialOut_SetMode(kSerialOut serial, kSerialOutMode mode);

/** 
 * Gets the mode for a serial.
 *
 * @public              @memberof kSerialOut
 * @param   serial      Serial object.  
 * @return              Mode.
 */
kFsFx(kSerialOutMode) kSerialOut_Mode(kSerialOut serial);

/** 
 * Reports constraint and validity information for the mode setting. 
 *
 * @public              @memberof kSerialOut
 * @param   serial      Serial object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kSerialOut_ModeInfo(kSerialOut serial, kInfoBits* info);

/** 
 * Sets the bit rate for a serial. 
 *
 * @public              @memberof kSerialOut
 * @param   serial      Serial object.  
 * @param   rate        Specifies the bit rate.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kSerialOut_SetBitRate(kSerialOut serial, k32u rate);

/** 
 * Gets the bit rate for a serial.
 *
 * @public              @memberof kSerialOut
 * @param   serial      Serial object.  
 * @return              Bit rate.
 */
kFsFx(k32u) kSerialOut_BitRate(kSerialOut serial);

/** 
 * Reports constraint and validity information for the bit rate setting. 
 *
 * @public              @memberof kSerialOut
 * @param   serial      Serial object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kSerialOut_BitRateInfo(kSerialOut serial, kInfo32u* info);

/** 
 * Transmit data buffer.
 *
 * @public              @memberof kSerialOut
 * @param   serial      Serial object.  
 * @param   buffer      Pointer to data buffer.  
 * @param   size        Data buffer length.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kSerialOut_Write(kSerialOut serial, const void* buffer, kSize size);

/** 
 * Retrieve statistics.
 *
 * @public              @memberof kSerialOut
 * @param   serial      Serial object.  
 * @param   stats       Receives the stats structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kSerialOut_Stats(kSerialOut serial, kSerialOutStats* stats);

#include <kFireSync/Client/kSerialOut.x.h>

#endif
