/** 
 * @file    kProjector.h
 * @brief   Declares the kProjector type. 
 *
 * @internal
 * Copyright (C) 2012-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_PROJECTOR_H
#define K_FIRESYNC_PROJECTOR_H

#include <kFireSync/kNodeDef.h>

/**
 * @class   kProjector
 * @extends kObject
 * @ingroup kFireSync-Client
 * @brief   Represents projector configuration.
 */
//typedef kObject kProjector;        --forward-declared in kFsDef.x.h

/** 
 * Gets the model of the projector.
 *
 * The projector supported is determined by current PL programs. 
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.  
 * @return              Returns the model of the projector.
 */
kFsFx(kProjectorModel) kProjector_Model(kProjector projector);

/** 
 * Gets the width of the projector display.
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.  
 * @return              Returns the width of the projector dimension.
 */
kFsFx(kSize) kProjector_Width(kProjector projector);

/**
* Gets the height of the projector display.
*
* @public              @memberof kProjector
* @param   projector   Projector object.
* @return              Returns the height of the projector dimension.
*/
kFsFx(kSize) kProjector_Height(kProjector projector);

/** 
 * Gets the id of the projector module.
 *
 * The id of a projector module is the same as the index of the projector module in the 
 * node's projector list.
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.  
 * @return              Returns the id of the projector module.
 */
kFsFx(kSize) kProjector_Id(kProjector projector);

/** 
 * Resets projector mode settings to defaults.
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.  
 * @param   stateCount  Number of projector states to create.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kProjector_Clear(kProjector projector, kSize stateCount);

/** 
 * Enables or disables the projector. 
 *
 * Disabled projectors are not checked during verification, and do not respond to 
 * Start/Pause/Resume/Stop commands. 
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.  
 * @param   enable      Specifies whether to enable or disable the projector.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kProjector_Enable(kProjector projector, kBool enable);

/** 
 * Reports whether the projector is currently enabled. 
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.  
 * @return              Returns whether the projector is currently enabled. 
 */
kFsFx(kBool) kProjector_IsEnabled(kProjector projector);

/** 
 * Reports whether a projector device is connected and communicating. 
 *
 * Projectors are detected immediately after a node is reset; this value is not updated dynamically. 
 * 
 * Physical connection has no effect on configuration; a projector module without a connected 
 * projector can still be configured and verified. At runtime, a disconnected projector module will 
 * not produce images, but the disconnected status will not be treated as an error that 
 * prevents the system from running.
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.  
 * @return              Returns whether a physical projector is connected to the projector module.
 */
kFsFx(kBool) kProjector_IsConnected(kProjector projector);

/** 
 * Sets the control mode of the projector.
 *
 * For control modes that suppport multiple sources (e.g. kPROJECTOR_CONTROL_TRIGGERED_BY_EVENT), 
 * use the kProjector_SetControlId function to select a specific source.
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.  
 * @param   type        Projector control mode.
 * @return              Operation status. 
 */
kFsFx(kStatus) kProjector_SetControl(kProjector projector, kProjectorControl type);

/** 
 * Reports the control mode of the projector. 
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.  
 * @return              Returns the control mode of the projector. 
 */
kFsFx(kProjectorControl) kProjector_Control(kProjector projector);

/** 
 * Reports constraint and validity information for the Control setting. 
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kProjector_ControlInfo(kProjector projector, kInfoBits* info);

/** 
 * Sets the control source id for a projector. 
 *
 * For control modes that suppport multiple sources (e.g. kPROJECTOR_CONTROL_TRIGGERED_BY_EVENT), 
 * this function cam be used to specify a specific source id.
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.  
 * @param   id          Specifies the control source id.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kProjector_SetControlId(kProjector projector, k32u id);

/** 
 * Gets the control source id for a projector.
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.  
 * @return              Control source id.
 */
kFsFx(k32u) kProjector_ControlId(kProjector projector);

/** 
 * Reports constraint and validity information for the ControlId setting. 
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kProjector_ControlIdInfo(kProjector projector, kInfo* info);

/** 
 * Creates and adds a new state with default values.
 *
 * The new state is added to the end of the projector's state list. 
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.  
 * @param   state       Receives the new state object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kProjector_AddState(kProjector projector, kProjectorState* state);

/** 
 * Destroys the specified state and removes it from the projector's state list.
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.  
 * @param   state       State object to be deleted.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kProjector_DeleteState(kProjector projector, kProjectorState state);

/** 
 * Swaps the order of two states in the projector's state list.
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.  
 * @param   index0      Index of the first state object to be swapped.  
 * @param   index1      Index of the second state object to be swapped.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kProjector_SwapStates(kProjector projector, kSize index0, kSize index1);

/** 
 * Gets the number of states belonging to the projector. 
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.  
 * @return              Returns the number of states belonging to the projector. 
 */
kFsFx(kSize) kProjector_StateCount(kProjector projector);

/** 
 * Reports constraint and validity information for the StateCount setting. 
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kProjector_StateCountInfo(kProjector projector, kInfoSize* info);

/** 
 * Gets the state at the specified index.
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.  
 * @param   index       Index of the state.  
 * @return              Returns the state at the specified index.
 */
kFsFx(kProjectorState) kProjector_StateAt(kProjector projector, kSize index);

/** 
 * Creates and adds a new projector control extension of the specified type.
 *
 * The new extension is added to the end of the projector's extension list. 
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.  
 * @param   type        Type of extension to add. 
 * @param   extension   Receives the new extension object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kProjector_AddExtension(kProjector projector, kType type, kProjectorExt* extension);

/** 
 * Destroys the specified projector control extension and removes it from the projector's extension list.
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.  
 * @param   extension   Extension object to be deleted.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kProjector_DeleteExtension(kProjector projector, kProjectorExt extension);

/** 
 * Gets the number of control extensions that have been configured.
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.  
 * @return              Count of projector control extensions. 
 */
kFsFx(kSize) kProjector_ExtensionCount(kProjector projector);

/** 
 * Gets the control extension at the specified index.
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.  
 * @param   index       Index of the extension.  
 * @return              Extension at the specified index.
 */
kFsFx(kProjectorExt) kProjector_ExtensionAt(kProjector projector, kSize index);

/**
 * Enables or disables dynamic frame rate for the projector.
 *
 * Some projector devices support multiple frame rates (e.g., DLPC300). The dynamic
 * frame rate feature allows FireSync to automatically vary the frame rate 
 * based on the requested exposure time. Disabling dynamic frame rate ensures a 
 * uniform frame rate (typically 60Hz). 
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.
 * @param   enable      Enable dynamic frame rate?
 * @return              Operation status.
 */
kFsFx(kStatus) kProjector_EnableDynamicFrameRate(kProjector projector, kBool enable);

/**
 * Reports whether dynamic frame rate is currently enabled.
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.
 * @return              Reports whether dynamic frame rate is enabled.
 */
kFsFx(kBool) kProjector_IsDynamicFrameRateEnabled(kProjector projector);

/**
 * Reports constraint and validity information for the DynamicFrameRate setting.
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.
 * @param   info        Receives the info structure.
 * @return              Operation status.
 */
kFsFx(kStatus) kProjector_DynamicFrameRateInfo(kProjector projector, kInfoBool* info);

/** 
 * Sets the number of frames to project in a sequence.
 *
 * In 1D_8PP mode, the frame count determines the number of distinct output patterns to 
 * generate from the individual rows of the uploaded pattern array.  This setting has 
 * no effect in 2D_1BPP mode. 
 * 
 * @public              @memberof kProjector
 * @param   projector   Projector object.  
 * @param   frameCount  Number of frames in a sequence.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kProjector_SetFrameCount(kProjector projector, kSize frameCount);

/** 
 * Gets the number of frames in a sequence.
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.  
 * @return              The number of frames in a sequence.
 */
kFsFx(kSize) kProjector_FrameCount(kProjector projector);

/** 
 * Reports constraint and validity information for the FrameCount setting.
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kProjector_FrameCountInfo(kProjector projector, kInfoSize* info);

/** 
 * Sets the projector pattern bit depth.
 *
 * This setting only affects the output bit depth of the projector display. The bit depth 
 * of pattern data provided as input to the kProjector_LoadPatterns method should always 
 * be either 1 bit or 8 bits.
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.  
 * @param   bitDepth    Pattern bit depth.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kProjector_SetBitDepth(kProjector projector, kSize bitDepth);

/** 
 * Gets the projected pattern bit depth.
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.  
 * @return              The pattern bit depth.
 */
kFsFx(kSize) kProjector_BitDepth(kProjector projector);

/** 
 * Reports constraint and validity information for the BitDepth setting.
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kProjector_BitDepthInfo(kProjector projector, kInfoSize* info);

/** 
 * Reports the time required by this projector to respond to an event.
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.  
 * @return              Minimum period, in FireSync nanoseconds. 
 */
kFsFx(k64u) kProjector_MinimumPeriod(kProjector projector);

/** 
 * Sets the expected format for input pattern data.
 * 
 * Refer to kProjector_LoadPatterns for a description of how pattern data 
 * is interpreted. 
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.  
 * @param   mode        Specifies the format of patterns.
 * @return              Operation status. 
 * @see                 kProjector_LoadPatterns. 
*/
kFsFx(kStatus) kProjector_SetPatternMode(kProjector projector, kProjectorPatternMode mode);

/** 
 * Reports the current input pattern format.
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.  
 * @return              Returns the pattern format.
 * @see                 kProjector_SetPatternMode.
 */
kFsFx(kProjectorPatternMode) kProjector_PatternMode(kProjector projector);

/** 
 * Reports constraint and validity information for the PatternMode setting. 
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kProjector_PatternModeInfo(kProjector projector, kInfoBits* info);

/** 
 * Sets the output pixel value corresponding to the specified pattern input value. 
 *
 * This setting is used in 2D_1BPP pattern mode, in which the input pattern consists
 * of one-bit pixels. Palette values determine the output display intensities corresponding 
 * to pixel value 0 and pixel value 1. 
 * 
 * @public              @memberof kProjector
 * @param   projector   Projector object.  
 * @param   index       Palette index (e.g., 0 or 1 for 1-BPP pattern). 
 * @param   pixel       Output pixel value.
 * @return              Operation status. 
 * @see                 kProjector_LoadPatterns, kProjector_Palette
 */
kFsFx(kStatus) kProjector_SetPalette(kProjector projector, kSize index, k32u pixel);

/** 
 * Gets the output pixel value corresponding to the specified pattern input value. 
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.  
 * @param   index       Palette index (e.g., 0 or 1 for 1-BPP pattern). 
 * @return              Output pixel value.
 */
kFsFx(k32u) kProjector_Palette(kProjector projector, kSize index);

/** 
 * Reports constraint and validity information for the Palette setting.
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.  
 * @param   index       Palette index (e.g., 0 or 1 for 1-BPP pattern). 
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kProjector_PaletteInfo(kProjector projector, kSize index, kInfo32u* info);

/**
 * Enables or disables light output for the projector.
 *
 * By default the light output is enabled. Light output can disabled for allowing normal 
 * trigger conditions to be used while no light is emitted from the device.
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.
 * @param   enable      Enable light output?
 * @return              Operation status.
 */
kFsFx(kStatus) kProjector_EnableLightOutput(kProjector projector, kBool enable);

/**
 * Reports whether light output is currently enabled.
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.
 * @return              Reports whether light output is enabled.
 * @see                 kProjector_EnableLightOutput
 */
kFsFx(kBool) kProjector_LightOutputEnabled(kProjector projector);

/**
 * Reports constraint and validity information for the LightOutput setting.
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.
 * @param   info        Receives the info structure.
 * @return              Operation status.
 */
kFsFx(kStatus) kProjector_LightOutputInfo(kProjector projector, kInfoBool* info);

/**
 * Enables or disables auto-shutdown for the projector.
 *
 * Overheat protection turns off LED when DMD temperature reaches a certain threshold.
 * Under some circumstances, the overheat protection can be turned off so projector
 * will not shutdown automatically.
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.
 * @param   enable      Enable light output?
 * @return              Operation status.
 */
kFsFx(kStatus) kProjector_EnableAutoShutdown(kProjector projector, kBool enable);

/**
 * Reports whether auto-shutdown is currently enabled.
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.
 * @return              Reports whether auto-shutdown is enabled.
 * @see                 kProjector_EnableLightOutput
 */
kFsFx(kBool) kProjector_AutoShutdownEnabled(kProjector projector);

/**
 * Reports constraint and validity information for the AutoShutdown setting.
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.
 * @param   info        Receives the info structure.
 * @return              Operation status.
 */
kFsFx(kStatus) kProjector_AutoShutdownInfo(kProjector projector, kInfoBool* info);

/** 
 * Loads the input pattern to be projected.
 *
 * The interpretation of the pattern data depends on the current pattern mode. In 
 * 2D_1BPP mode, pattern data should be provided as a kArray2<k8u>, with eight 1-bit pixels 
 * packed into each array element. Each output pixel in the 2D display is represented by 
 * a single bit in this 2D input array. 
 * 
 * In 1D_8BPP mode, pattern data should be provided as a kArray2<k8u>, with one 8-bit pixel 
 * in each array element. In this mode, each row of the input pattern is used to generate 
 * an entire 2D display frame by repeating the same input row across every output row.  
 * The resulting 2D display frame contains vertical stripes of varying widths. The FrameCount 
 * property determines how many individual input rows will be used to generate output frames 
 * (output frame N is defined by input pattern row N, modulo input pattern height).  
 *
 * If settings have been changed, kNode_Apply should be called prior to calling 
 * kProjector_LoadPatterns, to ensure that uncommitted settings are successfully applied before 
 * writing a pattern to the node. This ensures that the node has updated projector properites, 
 * which are required to correctly interpret the pattern.
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.  
 * @param   patterns    Pattern(s) to project.  
 * @return              Operation status. 
 * @see                 kProjector_SetPatternMode, kProjector_SetFrameCount
 */
kFsFx(kStatus) kProjector_LoadPatterns(kProjector projector, kArray2 patterns);

/** 
 * Loads the per row gains to input pattern.
 *
 * The gains should be provided as a kArray1<k64f>.
 * The number of gains has to be equal to the height of projector display. Not all the projector
 * models support loading row gains, it will return error if not supported.
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.  
 * @param   gains       Per row gains.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kProjector_LoadRowGains(kProjector projector, kArray1 gains);

/** 
 * Loads the per column gains to input pattern.
 *
 * The gains should be provided as a kArray1<k64f>.
 * The number of gains has to be equal to the width of projector display. Not all the projector
 * models support loading column gains, it will return error if not supported.
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.  
 * @param   gains       Per row gains.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kProjector_LoadColGains(kProjector projector, kArray1 gains);


/** 
 * Retrieve statistics.
 *
 * @public              @memberof kProjector
 * @param   projector   Projector object.  
 * @param   stats       Receives the stats structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kProjector_Stats(kProjector projector, kProjectorStats* stats);

#include <kFireSync/Client/kProjector.x.h>

#endif
