/** 
 * @file    kPipe.h
 * @brief   Declares the kPipe class. 
 *
 * @internal
 * Copyright (C) 2008-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_PIPE_H
#define K_FIRESYNC_PIPE_H

#include <kFireSync/kNodeDef.h>

/**
 * @class   kPipe
 * @extends kObject
 * @ingroup kFireSync-Client
 * @brief   Represents pipe configuration.
 */
//typedef kObject kPipe;        --forward-declared in kFsDef.x.h

/** 
 * Gets the number of block types currently available on this node. 
 *
 * The block types available on a particular node depend on the firmware application currently 
 * installed on the node.  
 *
 * @public              @memberof kPipe
 * @param   pipe        Pipe object.  
 * @return              Returns the number of block types currently available on this node. 
 */
kFsFx(kSize) kPipe_TypeCount(kPipe pipe); 

/** 
 * Gets the name of the block type at the specified index.
 *
 * @public              @memberof kPipe
 * @param   pipe        Pipe object.  
 * @param   index       Index of the block type.   
 * @return              Block type name. 
 */
kFsFx(const kChar*) kPipe_TypeNameAt(kPipe pipe, kSize index); 

/** 
 * Gets a description of the block type at the specified index.
 *
 * @public              @memberof kPipe
 * @param   pipe        Pipe object.  
 * @param   index       Index of the block type.   
 * @return              Block type description.
 */
kFsFx(const kChar*) kPipe_TypeDescriptionAt(kPipe pipe, kSize index); 

/** 
 * Resets pipe mode settings to defaults.
 *
 * @public              @memberof kPipe
 * @param   pipe        Pipe object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kPipe_Clear(kPipe pipe); 

/**
* Sets the pipe processing engine type.
*
* @public              @memberof kPipe
* @param   pipe        Pipe object.
* @param   type        Processing engine type.
* @return              Operation status.
*/
kFsFx(kStatus) kPipe_SetEngine(kPipe pipe, kPipeEngine type);

/**
* Gets the pipe processing engine type.
*
* @public              @memberof kPipe
* @param   pipe        Pipe object.
* @return              Processing engine type.
*/
kFsFx(kPipeEngine) kPipe_Engine(kPipe pipe);

/**
* Reports constraint and validity information for the Engine setting. 
*
* @public              @memberof kPipe
* @param   pipe        Pipe object.
* @param   info        Receives the info structure.  
* @return              Operation status.
*/
kFsFx(kStatus) kPipe_EngineInfo(kPipe pipe, kInfoBits* info);

/**
* Sets the desired number of threads used by the pipe engine. 
* 
* This setting is ignored by single-threaded pipe engines.
*
* @public              @memberof kPipe
* @param   pipe        Pipe object.
* @param   count       Desired number of threads (minimum 1). 
* @return              Operation status.
*/
kFsFx(kStatus) kPipe_SetThreadCount(kPipe pipe, kSize count);

/**
* Gets the desired number of threads used by the pipe engine. 
*
* @public              @memberof kPipe
* @param   pipe        Pipe object.
* @return              Desired number of threads. 
*/
kFsFx(kSize) kPipe_ThreadCount(kPipe pipe);

/**
* Reports constraint and validity information for the ThreadCount setting.
*
* @public              @memberof kPipe
* @param   pipe        Pipe object.
* @param   info        Receives the info structure.
* @return              Operation status. 
*/
kFsFx(kStatus) kPipe_ThreadCountInfo(kPipe pipe, kInfoSize* info);

/** 
 * Sets the thread priority class for any pipe threads.
 * 
 * Indiscriminate thread priority adjustments can result in deadlock/starvation under 
 * high processing load. Accordingly, thread priority adjustment should usually be avoided, 
 * except where indicated by the FireSync Software Team to support special cases. 
 *
 * Refer to kThread_SetPriority for more information on thread priorities. 
 * 
 * @public                  @memberof kPipe
 * @param   pipe            Pipe object.  
 * @param   priorityClass   Thread priority class.  
 * @return                  Operation status.
 * @see                     kThread_SetPriority
 */
kFsFx(kStatus) kPipe_SetThreadPriorityClass(kPipe pipe, kThreadPriorityClass priorityClass);

/** 
 * Gets the thread priority class for any pipe threads.
 *
 * @public              @memberof kPipe
 * @param   pipe        Pipe object.  
 * @return              Thread priority class.
 */
kFsFx(kThreadPriorityClass) kPipe_ThreadPriorityClass(kPipe pipe);

/**
* Reports constraint and validity information for the ThreadPriorityClass setting.
*
* @public              @memberof kPipe
* @param   pipe        Pipe object.
* @param   info        Receives the info structure.
* @return              Operation status. 
*/
kFsFx(kStatus) kPipe_ThreadPriorityClassInfo(kPipe pipe, kInfo32s* info);

/** 
 * Sets the thread priority offset for any pipe threads.
 *
 * @public                  @memberof kPipe
 * @param   pipe            Pipe object.  
 * @param   priorityOffset  Thread priority offset.
 * @return                  Operation status.
 * @see                     kThread_SetPriority
 */
kFsFx(k32s) kPipe_SetThreadPriorityOffset(kPipe pipe, k32s priorityOffset);

/** 
 * Gets the thread priority offset for any pipe threads.
 *
 * @public              @memberof kPipe
 * @param   pipe        Pipe object.  
 * @return              Thread priority offset.
 */
kFsFx(k32s) kPipe_ThreadPriorityOffset(kPipe pipe);

/**
* Reports constraint and validity information for the ThreadPriorityOffset setting.
*
* @public              @memberof kPipe
* @param   pipe        Pipe object.
* @param   info        Receives the info structure.
* @return              Operation status. 
*/
kFsFx(kStatus) kPipe_ThreadPriorityOffsetInfo(kPipe pipe, kInfo32s* info);

/** 
 * Sets the thread affinity for any pipe threads.
 * 
 * Thread affinity adjustments can result in suboptimal performance for some workloads.
 * Accordingly, thread affinity adjustment should usually be avoided, except where indicated 
 * by the FireSync Software Team to support special cases. 
 *
 * Refer to kThread_SetAffinity for more information on thread priorities.  
 *
 * @public              @memberof kPipe
 * @param   pipe        Pipe object.  
 * @param   affinity    Thread affinity.  
 * @return              Operation status.
 * @see                 kThread_SetAffinity
 */
kFsFx(kStatus) kPipe_SetThreadAffinity(kPipe pipe, kBitArray affinity);

/** 
 * Gets the thread affinity for any pipe threads.
 *
 * @public              @memberof kPipe
 * @param   pipe        Pipe object.  
 * @param   affinity    Thread affinity.  
 * @return              Operation status.
 */
kFsFx(kStatus) kPipe_ThreadAffinity(kPipe pipe, kBitArray affinity);

/**
* Reports constraint and validity information for the ThreadAffinity setting.
*
* @public              @memberof kPipe
* @param   pipe        Pipe object.
* @param   info        Receives the info structure.
* @return              Operation status. 
*/
kFsFx(kStatus) kPipe_ThreadAffinityInfo(kPipe pipe, kInfo* info);

/** 
 * Sets the maximum count for the pipe's message input queue. 
 *
 * @public              @memberof kPipe
 * @param   pipe        Pipe object.  
 * @param   count       Maximum queue item count.
 * @return              Operation status. 
 */
kFsFx(kStatus) kPipe_SetMaxQueueCount(kPipe pipe, kSize count); 

/** 
 * Gets the maximum count of the pipe's message input queue. 
 *
 * @public              @memberof kPipe
 * @param   pipe        Pipe object.  
 * @return              Maximum queue item count.
 */
kFsFx(kSize) kPipe_MaxQueueCount(kPipe pipe); 

/** 
 * Sets the maximum total data size for the pipe's message input queue. 
 *
 * @public              @memberof kPipe
 * @param   pipe        Pipe object.  
 * @param   size        Maximum total data size, in bytes
 * @return              Operation status. 
 */
kFsFx(kStatus) kPipe_SetMaxQueueSize(kPipe pipe, kSize size); 

/** 
 * Gets the maximum total data size of the pipe's message input queue. 
 *
 * @public              @memberof kPipe
 * @param   pipe        Pipe object.  
 * @return              Maximum total queue data size, in bytes.
 */
kFsFx(kSize) kPipe_MaxQueueSize(kPipe pipe); 

/** 
 * Creates and adds a new block to the pipe.
 *
 * The new block is added to the end of the pipe's block list. 
 * 
 * This function involves communication with the remote node. 
 *
 * @public              @memberof kPipe
 * @param   pipe        Pipe object.  
 * @param   typeName    Specifies the type of block to add.   
 * @param   name        Name of the new block.  
 * @param   block       Receives the new block object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kPipe_AddBlock(kPipe pipe, const kChar* typeName, const kChar* name, kBlock* block); 

/** 
 * Destroys the specified block and removes it from the pipe's block list.
 *
 * @public              @memberof kPipe
 * @param   pipe        Pipe object.  
 * @param   block       Block object to be deleted.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kPipe_DeleteBlock(kPipe pipe, kBlock block); 

/** 
 * Gets the total number of blocks belonging to the pipe (including system and user blocks). 
 *
 * @public              @memberof kPipe
 * @param   pipe        Pipe object.  
 * @return              Total number of blocks belonging to the pipe. 
 */
kFsFx(kSize) kPipe_BlockCount(kPipe pipe); 

/** 
 * Gets the block at the specified index.
 *
 * @public              @memberof kPipe
 * @param   pipe        Pipe object.  
 * @param   index       Index of the block.  
 * @return              Block at the specified index.
 */
kFsFx(kBlock) kPipe_BlockAt(kPipe pipe, kSize index); 

/** 
 * Gets the number of user-created blocks belonging to the pipe. 
 *
 * @public              @memberof kPipe
 * @param   pipe        Pipe object.  
 * @return              Number of user-created blocks. 
 */
kFsFx(kSize) kPipe_UserBlockCount(kPipe pipe); 

/** 
 * Gets the user-created block at the specified index.
 *
 * @public              @memberof kPipe
 * @param   pipe        Pipe object.  
 * @param   index       Index of the block.  
 * @return              User-created block at the specified index.
 */
kFsFx(kBlock) kPipe_UserBlockAt(kPipe pipe, kSize index); 

/** 
 * Swaps the order of two user blocks in the pipe's user block list.
 *
 * @public              @memberof kPipe
 * @param   pipe        Pipe object.  
 * @param   index0      Index of the first block to be swapped within user block list.   
 * @param   index1      Index of the second block to be swapped within user block list.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kPipe_SwapUserBlocks(kPipe pipe, kSize index0, kSize index1);

/** 
 * Adds a message route to the pipe.
 *
 * @public              @memberof kPipe
 * @param   pipe        Pipe object.  
 * @param   sourceBlock Source block id.   
 * @param   sourcePort  Source port id.   
 * @param   destBlock   Destination block id.   
 * @param   destPort    Destination port id.   
 * @param   route       Optionally receives the new route object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kPipe_AddRoute(kPipe pipe, k32u sourceBlock, k32u sourcePort, k32u destBlock, k32u destPort, kRoute* route);

/** 
 * Adds a message route to the pipe.
 *
 * @public              @memberof kPipe
 * @param   pipe        Pipe object.  
 * @param   source      Source end-point.   
 * @param   destination Destination end-point.   
 * @param   route       Optionally receives the new route object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kPipe_AddRoutePoints(kPipe pipe, const kRoutePoint* source, const kRoutePoint* destination, kRoute* route);

/** 
 * Adds a message route to the pipe by resolving end-point names. 
 *
 * @public              @memberof kPipe
 * @param   pipe        Pipe object.  
 * @param   sourceBlock Source block name.   
 * @param   sourcePort  Source port name.   
 * @param   destBlock   Destination block name. 
 * @param   destPort    Destination port name.
 * @param   route       Optionally receives the new route object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kPipe_AddRouteStr(kPipe pipe, const kChar* sourceBlock, const kChar* sourcePort, const kChar* destBlock, const kChar* destPort, kRoute* route);

/** 
 * Determines the routing end-point of a port from block/port names.
 *
 * @public              @memberof kPipe
 * @param   pipe        Pipe object.  
 * @param   blockName   Name of the block containing the port.  
 * @param   portName    Name of the port.  
 * @param   routePoint  Receives route end-point information.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kPipe_ResolveRoutePoint(kPipe pipe, const kChar* blockName, const kChar* portName, kRoutePoint* routePoint); 

/** 
 * Finds a block instance by name. 
 *
 * @public              @memberof kPipe
 * @param   pipe        Pipe object.  
 * @param   blockName   Name of the block to find.
 * @param   block       Receives block instance. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kPipe_ResolveBlock(kPipe pipe, const kChar* blockName, kBlock* block); 

/** 
 * Finds a block instance by id. 
 *
 * @public              @memberof kPipe
 * @param   pipe        Pipe object.  
 * @param   id          Id of the block to find.
 * @param   block       Receives block instance. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kPipe_ResolveBlockById(kPipe pipe, k32u id, kBlock* block); 

/** 
 * Finds a port instance by name. 
 *
 * @public              @memberof kPipe
 * @param   pipe        Pipe object.  
 * @param   blockName   Name of the block to find.
 * @param   portName    Name of the port to find.  
 * @param   port        Receives port instance. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kPipe_ResolvePort(kPipe pipe, const kChar* blockName, const kChar* portName, kPort* port); 

/** 
 * Removes a message route from the pipe.
 *
 * @public              @memberof kPipe
 * @param   pipe        Pipe object.  
 * @param   route       The route to be removed.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kPipe_DeleteRoute(kPipe pipe, kRoute route);

/** 
 * Removes all message routes from the pipe.
 *
 * @public              @memberof kPipe
 * @param   pipe        Pipe object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kPipe_ClearRoutes(kPipe pipe);

/** 
 * Returns the number of routes in the pipe. 
 *
 * @public              @memberof kPipe
 * @param   pipe        Pipe object.  
 * @return              Number of routes in the pipe. 
 */
kFsFx(kSize) kPipe_RouteCount(kPipe pipe); 

/** 
 * Gets the route at the specified index. 
 *
 * @public              @memberof kPipe
 * @param   pipe        Pipe object.  
 * @param   index       Index of the route.  
 * @return              Route at the specified index. 
 */
kFsFx(kRoute) kPipe_RouteAt(kPipe pipe, kSize index); 

/** 
* Enables the pipe timer service.
* 
* Note, the pipe timer service does not run when the pipe is operating in replay mode.
*
* @public              @memberof kPipe
* @param   pipe        Pipe object.
* @param   enable      Specifies whether to enable or disable the timer block.
* @return              Operation status.
*/
kFsFx(kStatus) kPipe_EnableTimer(kPipe pipe, kBool enable);

/** 
* Reports whether the pipe timer is currently enabled.
*
* @public              @memberof kPipe
* @param   pipe        Pipe object.
* @return              Timer enabled status.
*/
kFsFx(kBool) kPipe_TimerEnabled(kPipe pipe);

/** 
* Reports constraint and validity information for the TimerEnabled setting.
*
* @public              @memberof kPipe
* @param   pipe        Pipe object.
* @param   info        Receives the info structure.
* @return              Operation status.
*/
kFsFx(kStatus) kPipe_TimerEnabledInfo(kPipe pipe, kInfoBool* info);

/** 
* Sets the period for the pipe timer service.
*
* @public              @memberof kPipe
* @param   pipe        Pipe object.
* @param   periodUs    Timer period, in microseconds.
* @return              Operation status.
*/
kFsFx(kStatus) kPipe_SetTimerPeriod(kPipe pipe, k64u periodUs);

/** 
* Reports the current period for the piper timer service.
*
* @public              @memberof kPipe
* @param   pipe        Pipe object.
* @return              Timer period, in microseconds.
*/
kFsFx(k64u) kPipe_TimerPeriod(kPipe pipe);

/** 
* Reports constraint and validity information for the TimerPeriod setting.
* 
* @public              @memberof kPipe
* @param   pipe        Pipe object.
* @param   info        Receives the info structure.
* @return              Operation status.
*/
kFsFx(kStatus) kPipe_TimerPeriodInfo(kPipe pipe, kInfo64u* info);

#include <kFireSync/Client/kPipe.x.h>

#endif
