/** 
 * @file    kEventManager.x.h
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_EVENT_MANAGER_X_H
#define K_FIRESYNC_EVENT_MANAGER_X_H

typedef struct kEventManagerClass
{
    kObjectClass base; 

    kNode node;                         //Node (parent). 
    kSize index;                        //Index within node's event manager list. 
    kEventManagerType eventType;        //Event manager type.
    
    kBool enabled;                      //Is this event manager enabled?
    kEventControl controlType;          //Control mode setting.
    k32u controlId;                     //Control id setting.
    kSyncDomain domain;                 //Domain setting.
    kEventEncoder encoderMode;          //Encoder mode setting.
    k64u offset;                        //Offset setting.
    k64u period;                        //Period setting.
    k64u count;                         //Count setting.    
    kBool autoPeriodEnabled;            //AutoPeriodEnabled setting.
    kBool gateEnabled;                  //Gate enabled setting; 
    kEventMissStyle missStyle;          //Event manager miss style.
    kBool cancellationEnabled;          //Event sequence cancellation enabled setting.
    kBool supportsMissHandling;         //Is handling of missed targets supported?

} kEventManagerClass; 

kDeclareClassEx(kFs, kEventManager, kObject)
        
kFsFx(kStatus) kEventManager_Construct(kEventManager* manager, kNode node, kSize index, kAlloc allocator); 

kFsFx(kStatus) kEventManager_Init(kEventManager manager, kType type, kNode node, kSize index, kAlloc alloc); 
kFsFx(kStatus) kEventManager_VRelease(kEventManager manager); 

kFsFx(kStatus) kEventManager_Parse(kEventManager manager, kXml xml, kXmlItem item); 
kFsFx(kStatus) kEventManager_Format(kEventManager manager, kXml xml, kXmlItem item); 

kFsFx(kStatus) kEventManager_VerificationLabel(kEventManager manager, kChar* label, kSize capacity); 
kFsFx(kStatus) kEventManager_Verify(kEventManager manager); 

kFsFx(kStatus) kEventManager_ClearEx(kEventManager manager, kNodeClearOption options);

kFsFx(k64u) kEventManager_MinimumEventPeriod(k64u modulePeriod, kSize toleranceMultiplier); 

kFsFx(kBool) kEventManager_MissHandlingSupported(kEventManager manager);

/**
 * TODO: Move function declarations below to public API once FSS-411 / FSH-1708 is completed.
 */

/**
 * Sets the event manager miss style.
 *
 * Warning: This feature is not yet ready for use. Please refer to FSS-411 / FSH-1708 for details.
 *
 * @public              @memberof kEventManager
 * @param   manager     Event manager object.
 * @param   missStyle   Event manager miss style.
 * @return              Operation status.
 */
kFsFx(kStatus) kEventManager_SetMissStyle(kEventManager manager, kEventMissStyle missStyle);

/**
 * Reports the event manager miss style.
 *
 * Warning: This feature is not yet ready for use. Please refer to FSS-411 / FSH-1708 for details.
 *
 * @public              @memberof kEventManager
 * @param   manager     Event manager object.
 * @return              Miss style of the event manager.
 */
kFsFx(kEventMissStyle) kEventManager_MissStyle(kEventManager manager);

/**
 * Reports constraint and validity information for the MissStyle setting.
 *
 * Warning: This feature is not yet ready for use. Please refer to FSS-411 / FSH-1708 for details.
 *
 * @public              @memberof kEventManager
 * @param   manager     Event manager object.
 * @param   info        Receives the info structure.
 * @return              Operation status.
 */
kFsFx(kStatus) kEventManager_MissStyleInfo(kEventManager manager, kInfoBits* info);

/**
 * Enables or disables event sequence cancellation.
 *
 * Warning: This feature is not yet ready for use. Please refer to FSS-411 / FSH-1708 for details.
 *
 * @public              @memberof kEventManager
 * @param   manager     Event manager object.
 * @param   enabled     kTRUE to enable event sequence cancellation; kFALSE otherwise.
 * @return              Operation status.
 */
kFsFx(kStatus) kEventManager_EnableCancellation(kEventManager manager, kBool enabled);

/**
 * Reports whether event sequence cancellation is enabled.
 *
 * Warning: This feature is not yet ready for use. Please refer to FSS-411 / FSH-1708 for details.
 *
 * @public              @memberof kEventManager
 * @param   manager     Event manager object.
 * @return              kTRUE if event sequence cancellation is enabled; kFALSE otherwise.
 */
kFsFx(kBool) kEventManager_CancellationEnabled(kEventManager manager);

/**
 * Reports constraint and validity information for the CancellationEnabled setting.
 *
 * Warning: This feature is not yet ready for use. Please refer to FSS-411 / FSH-1708 for details.
 *
 * @public              @memberof kEventManager
 * @param   manager     Event manager object.
 * @param   info        Receives the info structure.
 * @return              Operation status.
 */
kFsFx(kStatus) kEventManager_CancellationEnabledInfo(kEventManager manager, kInfoBool* info);

#endif
