/** 
 * @file    kEventManager.h
 * @brief   Declares the kEventManager type. 
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_EVENT_MANAGER_H
#define K_FIRESYNC_EVENT_MANAGER_H

#include <kFireSync/kNodeDef.h>

/**
 * @class   kEventManager
 * @extends kObject
 * @ingroup kFireSync-Client
 * @brief   Represents event manager configuration.
 * 
 * Event managers provide signals ("events") to trigger other modules such as cameras, lights, and digital outputs. Multiple
 * devices can be driven by a single event manager, providing one approach for inter-module synchronization. 
 *
 * Event managers provide a Control setting to support different methods of generating events (kEventControl). Methods 
 * range from continuous/periodic events, to software-scheduled events, to input-triggered events. 
 * 
 * Event managers support time and encoder domains (kSyncDomain). A single Domain setting per event manager determines how units 
 * are interpreted for other event manager settings (i.e., encoder ticks vs. FireSync time units). If the encoder domain is selected, 
 * an additional EncoderMode setting determines how the event mangager responds to the encoder (kEventEncoder).
 * 
 * Event managers provide an event request queue with a minimum depth of 256 elements, enabling multiple event-generation requests to 
 * be enqueued for future processing. This enables software-scheduled and input-triggered events to be generated far ahead of time, 
 * with multiple requests enqueued before the first request is serviced. 
 * 
 * Each enqueued event request can generate a sequence of events. An event sequence is characterized by the Count setting (number of 
 * events in the sequence) and the Period setting (time/distance between each event). Additionally, for input-triggered events, the Offset 
 * setting determines the distance (time/encoder) between the trigger (e.g., digital input) and the generated event sequence. 
 * 
 * If the Period setting is too low, modules such as cameras and lights may drop exposures (e.g., kHEALTH_ID_CAMERA_EXPOSURE_DROPS). 
 * To avoid this problem, the event Period should be large enough to accomodate the minimum time required by any modules driven by the 
 * event manager. For time-based events, the minimum event Period can be calculated and applied automatically using the 
 * EnableAutoPeriod option. 
 * 
 * For input-triggered and software-scheduled event sequences, it is possible for an entire event sequence to be dropped if the next 
 * sequence is scheduled to start before the previous sequence has completed (kHEALTH_ID_EVENT_SEQUENCE_DROP_COUNT).  In any control 
 * mode, it is possible for individual events to be dropped due to FireSync synchronization protocol glitches (kHEALTH_ID_EVENT_DROP_COUNT).  
 * 
 * The following example illustrates how to configure a single event manager to trigger both a camera and a light in a 
 * continuous/periodic time sequence: 
 * 
@code {.c}
kStatus ConfigureContinuous(kSystem system, kNode node)
{
    const k64u PERIOD_NS = 100000 * 1000;       //~100000 us period (in FireSync nanoseconds)
    const k64u EXPOSURE_NS = 10000 * 1000;      //~10000 us exposure (in FireSync nanoseconds)
    kEventManager eventManager = kNode_EventManagerAt(node, 0); 
    kCamera camera = kNode_CameraAt(node, 0); 
    kCameraState cameraState = kNULL; 
    kLight light = kNode_LightAt(node, 0); 
    kLightState lightState = kNULL; 

    //configure the event manaager
    kCheck(kEventManager_Clear(eventManager)); 
    kCheck(kEventManager_Enable(eventManager, kTRUE)); 
    kCheck(kEventManager_SetControl(eventManager, kEVENT_CONTROL_CONTINUOUS)); 
    kCheck(kEventManager_SetDomain(eventManager, kSYNC_DOMAIN_TIME)); 
    kCheck(kEventManager_SetPeriod(eventManager, PERIOD_NS)); 

    //configure the camera to respond to event manager 0
    kCheck(kCamera_Clear(camera, 1)); 
    kCheck(kCamera_Enable(camera, kTRUE)); 
    kCheck(kCamera_SetControl(camera, kCAMERA_CONTROL_TRIGGERED_BY_EVENT)); 
    kCheck(kCamera_SetControlId(camera, 0));   //event manager index 0

    //set camera Exposure (by default, Delay will be zero, and Gap will be determined automatically)
    cameraState = kCamera_StateAt(camera, 0); 
    kCheck(kCameraState_SetExposureNs(cameraState, EXPOSURE_NS)); 

    //configure the light to respond to event manager 0
    kCheck(kLight_Clear(light, 1)); 
    kCheck(kLight_Enable(light, kTRUE)); 
    kCheck(kLight_SetControl(light, kLIGHT_CONTROL_TRIGGERED_BY_EVENT)); 
    kCheck(kLight_SetControlId(light, 0));    //event manager index 0

    //set light Exposure (by default, Delay will be zero, and Gap will be determined automatically)
    lightState = kLight_StateAt(light, 0); 
    kCheck(kLightState_SetExposureNs(lightState, EXPOSURE_NS)); 
  
    return kOK; 
}
@endcode
 */
//typedef kObject kEventManager;        --forward-declared in kFsDef.x.h

/** 
 * Resets event manager mode settings.
 *
 * @public              @memberof kEventManager
 * @param   manager     Event manager object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kEventManager_Clear(kEventManager manager);

/** 
 * Gets the event manager type.
 *
 * @public              @memberof kEventManager
 * @param   manager     EventManager object.  
 * @return              Type of the event manager module.
 */
kFsFx(kEventManagerType) kEventManager_Type(kEventManager manager);

/** 
 * Gets the id of the event manager module.
 *
 * The id of an event manager module is the same as its index within the node's 
 * event manager list.
 *
 * @public              @memberof kEventManager
 * @param   manager     EventManager object.  
 * @return              Id (index) of the event manager module.
 */
kFsFx(kSize) kEventManager_Id(kEventManager manager);

/** 
 * Enables or disables the event manager. 
 *
 * Disabled event managers are not checked during verification, and do not respond to 
 * Start/Pause/Resume/Stop commands. 
 *
 * @public              @memberof kEventManager
 * @param   manager     Event manager object.  
 * @param   enable      Specifies whether to enable or disable the event manager.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kEventManager_Enable(kEventManager manager, kBool enable);

/** 
 * Reports whether the event manager is currently enabled. 
 *
 * @public              @memberof kEventManager
 * @param   manager     EventManager object.  
 * @return              kTRUE if the event manager is enabled; kFALSE otherwise.
 */
kFsFx(kBool) kEventManager_IsEnabled(kEventManager manager);

/** 
 * Sets the event manager control mode.
 *
 * For control modes that suppport multiple sources (e.g. kEVENT_CONTROL_TRIGGERED_BY_CAMERA), 
 * use the kEventManager_SetControlId function to specify a specific source.
 *
 * @public              @memberof kEventManager
 * @param   manager     Event manager object.  
 * @param   type        Event manager control mode.
 * @return              Operation status. 
 */
kFsFx(kStatus) kEventManager_SetControl(kEventManager manager, kEventControl type);

/** 
 * Reports the event manager control mode. 
 *
 * @public              @memberof kEventManager
 * @param   manager     Event manager object.  
 * @return              Control mode of the event manager. 
 */
kFsFx(kEventControl) kEventManager_Control(kEventManager manager);

/** 
 * Reports constraint and validity information for the Control setting. 
 *
 * @public              @memberof kEventManager
 * @param   manager     Event manager object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kEventManager_ControlInfo(kEventManager manager, kInfoBits* info);

/** 
 * Sets the control source id for an event manager. 
 *
 * For control modes that suppport multiple sources (e.g. kEVENT_CONTROL_TRIGGERED_BY_CAMERA), 
 * this function cam be used to specify a specific source id.
 *
 * @public              @memberof kEventManager
 * @param   manager     Event manager object.  
 * @param   id          Specifies the control source id.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kEventManager_SetControlId(kEventManager manager, k32u id);

/** 
 * Gets the control source id for an event manager.
 *
 * @public              @memberof kEventManager
 * @param   manager     Event manager object.  
 * @return              Control source id.
 */
kFsFx(k32u) kEventManager_ControlId(kEventManager manager);

/** 
 * Reports constraint and validity information for the ControlId setting. 
 *
 * @public              @memberof kEventManager
 * @param   manager     Event manager object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kEventManager_ControlIdInfo(kEventManager manager, kInfo* info);

/** 
 * Sets the event manager synchronization domain.
 *
 * @public              @memberof kEventManager
 * @param   manager     Event manager object.  
 * @param   domain      Event manager synchronization domain.
 * @return              Operation status. 
 */
kFsFx(kStatus) kEventManager_SetDomain(kEventManager manager, kSyncDomain domain);

/** 
 * Reports the event manager synchronization domain. 
 *
 * @public              @memberof kEventManager
 * @param   manager     Event manager object.  
 * @return              Synchronization domain of the event manager. 
 */
kFsFx(kSyncDomain) kEventManager_Domain(kEventManager manager);

/** 
 * Reports constraint and validity information for the Domain setting. 
 *
 * @public              @memberof kEventManager
 * @param   manager     Event manager object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kEventManager_DomainInfo(kEventManager manager, kInfoBits* info);

/** 
 * Sets the event manager encoder mode.
 *
 * @public              @memberof kEventManager
 * @param   manager     Event manager object.  
 * @param   mode        Event manager encoder mode.
 * @return              Operation status. 
 */
kFsFx(kStatus) kEventManager_SetEncoderMode(kEventManager manager, kEventEncoder mode);

/** 
 * Reports the event manager encoder mode. 
 *
 * @public              @memberof kEventManager
 * @param   manager     Event manager object.  
 * @return              Encoder mode. 
 */
kFsFx(kEventEncoder) kEventManager_EncoderMode(kEventManager manager);

/** 
 * Reports constraint and validity information for the EncoderMode setting. 
 *
 * @public              @memberof kEventManager
 * @param   manager     Event manager object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kEventManager_EncoderModeInfo(kEventManager manager, kInfoBits* info);

/** 
 * Sets the event manager offset amount.
 *
 * Offset units are given by the Domain setting. When the synchronization domain is set to kSYNC_DOMAIN_TIME, 
 * units are FireSync nanoseconds (~ 1 ns).
 *
 * @public              @memberof kEventManager
 * @param   manager     Event manager object.  
 * @param   offset      Offset value.
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kEventManager_SetOffset(kEventManager manager, k64u offset, kAdjust adjustment = kADJUST_NONE);

/** 
 * Reports the event manager offset amount. 
 *
 * Offset units are given by the Domain setting. When the synchronization domain is set to kSYNC_DOMAIN_TIME, 
 * units are FireSync nanoseconds (~ 1 ns).
 *
 * @public              @memberof kEventManager
 * @param   manager     Event manager object.  
 * @return              Offset amount.
 */
kFsFx(k64u) kEventManager_Offset(kEventManager manager);

/** 
 * Reports constraint and validity information for the Offset setting. 
 *
 * @public              @memberof kEventManager
 * @param   manager     Event manager object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kEventManager_OffsetInfo(kEventManager manager, kInfo64u* info);

/** 
 * Sets the event manager sequence length.
 *
 * @public              @memberof kEventManager
 * @param   manager     Event manager object.  
 * @param   count       Event sequence length.
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kEventManager_SetCount(kEventManager manager, k64u count, kAdjust adjustment = kADJUST_NONE);

/** 
 * Reports the event manager sequence length. 
 *
 * @public              @memberof kEventManager
 * @param   manager     Event manager object.  
 * @return              Event sequence length. 
 */
kFsFx(k64u) kEventManager_Count(kEventManager manager);

/** 
 * Reports constraint and validity information for the Count setting. 
 *
 * @public              @memberof kEventManager
 * @param   manager     Event manager object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kEventManager_CountInfo(kEventManager manager, kInfo64u* info);

/** 
 * Sets the event manager period.
 *
 * Period units are given by the Domain setting. When the synchronization domain is set to kSYNC_DOMAIN_TIME, 
 * units are FireSync nanoseconds (~ 1 ns).
 *
 * @public              @memberof kEventManager
 * @param   manager     Event manager object.  
 * @param   period      Period value. 
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status.
 */
kFsFx(kStatus) kEventManager_SetPeriod(kEventManager manager, k64u period, kAdjust adjustment = kADJUST_NONE);

/** 
 * Reports the event manager period. 
 *
 * Period units are given by the Domain setting. When the synchronization domain is set to kSYNC_DOMAIN_TIME, 
 * units are FireSync nanoseconds (~ 1 ns).
 *
 * @public              @memberof kEventManager
 * @param   manager     Event manager object.  
 * @return              Period value.
 */
kFsFx(k64u) kEventManager_Period(kEventManager manager);

/** 
 * Reports constraint and validity information for the Period setting. 
 *
 * @public              @memberof kEventManager
 * @param   manager     Event manager object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kEventManager_PeriodInfo(kEventManager manager, kInfo64u* info);

/** 
 * Reports the minimum time period at which events can be generated without causing missed events. 
 *
 * Minimum period values are provided in FireSync nanoseconds (~ 1 ns).
 *
 * @public              @memberof kEventManager
 * @param   manager     Event manager object.  
 * @return              Minimum time period, in nanoseconds. 
 */
kFsFx(k64u) kEventManager_MinimumPeriod(kEventManager manager); 

/** 
 * Enables automatic period calculation for the event manager.
 *
 * @public              @memberof kEventManager
 * @param   state       State object.  
 * @param   enabled     Specifies whether automatic period is enabled.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kEventManager_EnableAutoPeriod(kEventManager state, kBool enabled);

/** 
 * Reports whether automatic period is enabled for the event manager.
 *
 * @public              @memberof kEventManager
 * @param   state       State object.  
 * @return              Returns whether automatic period is enabled for the event manager.
 */
kFsFx(kBool) kEventManager_AutoPeriodEnabled(kEventManager state);

/** 
 * Reports constraint and validity information for the AutoPeriodEnabled setting. 
 *
 * @public              @memberof kEventManager
 * @param   state       State object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kEventManager_AutoPeriodEnabledInfo(kEventManager state, kInfoBool* info);

/** 
 * Enables or disables event gating. 
 *
 * @public              @memberof kEventManager
 * @param   manager     Event manager object.  
 * @param   enabled     kTRUE to enable event gating; kFALSE otherwise.
 * @return              Operation status. 
 */
kFsFx(kStatus) kEventManager_EnableGate(kEventManager manager, kBool enabled);

/** 
 * Reports whether event gating is enabled.  
 *
 * @public              @memberof kEventManager
 * @param   manager     Event manager object.  
 * @return              kTRUE if event gating is enabled; kFALSE otherwise. 
 */
kFsFx(kBool) kEventManager_GateEnabled(kEventManager manager);

/** 
 * Reports constraint and validity information for the GateEnabled setting. 
 *
 * @public              @memberof kEventManager
 * @param   manager     Event manager object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kEventManager_GateEnabledInfo(kEventManager manager, kInfoBool* info);

/** 
 * Schedules an event sequence to begin at a future time or position.
 *
 * Target units are given by the Domain setting. When the synchronization domain is set to kSYNC_DOMAIN_TIME, 
 * units are FireSync microseconds (~ 1 us).
 * 
 * @public              @memberof kEventManager
 * @param   manager     Event manager object.  
 * @param   target      Target value for beginning of event sequence.
 * @return              Operation status. 
 */
kFsFx(kStatus) kEventManager_Schedule(kEventManager manager, k64s target);

/** 
 * Provides a software trigger to an event manager. 
 *
 * @public              @memberof kEventManager
 * @param   manager     Event manager object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kEventManager_Trigger(kEventManager manager);

/** 
 * Gets current event manager statistics.
 *
 * @public              @memberof kEventManager
 * @param   manager     Event manager object.  
 * @param   stats       Receives statistics.
 * @return              Operation status. 
 */
kFsFx(kStatus) kEventManager_Stats(kEventManager manager, kEventStats* stats);

#include <kFireSync/Client/kEventManager.x.h>

#endif
