/** 
 * @file    kBlock.h
 * @brief   Declares the kBlock class. 
 *
 * @internal
 * Copyright (C) 2008-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_BLOCK_H
#define K_FIRESYNC_BLOCK_H

#include <kFireSync/kNodeDef.h>

/**
 * @class   kBlock
 * @extends kObject
 * @ingroup kFireSync-Client
 * @brief   Represents block configuration.
 */
//typedef kObject kBlock;        --forward-declared in kFsDef.x.h

/** 
 * Sets the unique identifier for this block. 
 *
 * The block identifier is used when external references to the block are needed. For example, 
 * the block identifier is used in the kRoutePoint structure to specify the block segment of a 
 * route end point.
 * 
 * Each block belonging to a particular node must have a unique identifier. 
 *
 * @public              @memberof kBlock
 * @param   block       Block object.  
 * @param   id          The identifier for this block.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kBlock_SetId(kBlock block, k32u id);

/** 
 * Gets the unique identifier for this block. 
 *
 * The block identifier is used when external references to the block are needed. For example, 
 * the block identifier is used in the kRoutePoint structure to specify the block segment of a 
 * route end point.
 *
 * @public              @memberof kBlock
 * @param   block       Block object.  
 * @return              Returns the unique identifier for this block. 
 */
kFsFx(k32u) kBlock_Id(kBlock block);

/** 
 * Reports constraint and validity information for the Id setting. 
 *
 * @public              @memberof kBlock
 * @param   block       Block object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kBlock_IdInfo(kBlock block, kInfo32u* info);

/** 
 * Sets a descriptive name for the block. 
 *
 * @public              @memberof kBlock
 * @param   block       Block object.  
 * @param   name        Buffer containing the null-terminated name string.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kBlock_SetName(kBlock block, const kChar* name);

/** 
 * Gets a descriptive name for this block.
 *
 * @public              @memberof kBlock
 * @param   block       Block object.  
 * @return              Descriptive name for block. 
 */
kFsFx(const kChar*) kBlock_Name(kBlock block);

/** 
 * Gets the type name of this block. 
 *
 * @public              @memberof kBlock
 * @param   block       Block object.  
 * @return              Type name. 
 */
kFsFx(const kChar*) kBlock_TypeName(kBlock block);

/**
* Assigns this block to a processing group.
*
* By default, blocks in concurrent pipe environments are independently scheduled for execution. The kBlock_AssignGroup
* function can be used to create larger processing groups, so that multiple blocks are executed together. This technique can
* be used to optimize pipe execution by reducing context switching overhead, particulary when dealing with lightweight
* blocks.
*
* @public              @memberof kBlock
* @param   block       Block object.
* @param   groupId     Processing group id (or k32U_NULL for none).
* @return              Operation status.
*/
kFsFx(kStatus) kBlock_AssignGroup(kBlock block, k32u groupId);

/**
* Reports whether this block has been assigned to a group.
*
* @public              @memberof kBlock
* @param   block       Block object.
* @return              kTRUE if the block is grouped; kFALSE otherwise.
*/
kFsFx(kBool) kBlock_IsGrouped(kBlock block);

/**
* Gets the processing group id associated with this block.
*
* @public              @memberof kBlock
* @param   block       Block object.
* @return              Processing group id (undefined if ungrouped).
*/
kFsFx(k32u) kBlock_GroupId(kBlock block);

/** 
 * Sets configuration for the block. 
 *
 * The XML settings object will be examined by the block and any errors or inconsistencies 
 * will be automatically resolved. Call kBlock_Config to obtain the adjusted settings.
 * 
 * This function involves communication with the remote node. 
 *
 * @public              @memberof kBlock
 * @param   block       Block object.  
 * @param   settings    Specifies the desired block configuration.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kBlock_SetConfig(kBlock block, kXml settings);

/** 
 * Gets configuration for the block. 
 *
 * @public              @memberof kBlock
 * @param   block       Block object.  
 * @param   settings    Receives the current block configuration.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kBlock_Config(kBlock block, kXml settings);

/** 
 * Gets the number of ports belonging to the block. 
 *
 * @public              @memberof kBlock
 * @param   block       Block object.  
 * @return              Returns the number of ports belonging to the block. 
 */
kFsFx(kSize) kBlock_PortCount(kBlock block); 

/** 
 * Gets the port at the specified index.
 *
 * @public              @memberof kBlock
 * @param   block       Block object.  
 * @param   index       Index of the port.  
 * @return              Returns the port at the specified index.
 */
kFsFx(kPort) kBlock_PortAt(kBlock block, kSize index); 

/** 
 * Gets the port at the specified index.
 *
 * @public              @memberof kBlock
 * @param   block       Block object.  
 * @param   id          Index of the port.  
 * @param   port        Index of the port.  
 * @return              Returns the port at the specified index.
 */
kFsFx(kStatus) kBlock_FindPortById(kBlock block, k32u id, kPort* port); 

#include <kFireSync/Client/kBlock.x.h>

#endif
