/** 
 * @file    kAction.h
 * @brief   Declares the kAction class. 
 *
 * @internal
 * Copyright (C) 2014-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_ACTION_H
#define K_FIRESYNC_ACTION_H

#include <kFireSync/kNodeDef.h>
#include <kFireSync/Action/kAxAction.h>

/**
 * @class   kAction
 * @extends kObject
 * @ingroup kFireSync-Client
 * @brief   Represents action configuration.
 */
//typedef kObject kAction;        --forward-declared in kFsDef.x.h

/** 
 * Sets a descriptive name for the action. 
 *
 * @public              @memberof kAction
 * @param   action      Action object.  
 * @param   name        Buffer containing the null-terminated name string.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kAction_SetName(kAction action, const kChar* name);

/** 
 * Gets a descriptive name for this action.
 *
 * @public              @memberof kAction
 * @param   action      Action object.  
 * @return              Descriptive name for action. 
 */
kFsFx(const kChar*) kAction_Name(kAction action);

/** 
 * Gets the type name of this action. 
 *
 * @public              @memberof kAction
 * @param   action      Action object.  
 * @return              Type name. 
 */
kFsFx(const kChar*) kAction_TypeName(kAction action);

/** 
 * Sets the trigger type for the action.
 *
 * @public                  @memberof kAction
 * @param   action          Action object.  
 * @param   triggerType     Trigger type.
 * @return                  Operation status. 
 */
kFsFx(kStatus) kAction_SetTrigger(kAction action, kActionTrigger triggerType);

/** 
 * Gets the trigger type for the action.
 *
 * @public              @memberof kAction
 * @param   action      Action object.  
 * @return              Trigger type.
 */
kFsFx(kActionTrigger) kAction_Trigger(kAction action);

/** 
 * Sets configuration for the action. 
 *
 * The XML settings object will be examined by the action and any errors or inconsistencies 
 * will be automatically resolved. Call kAction_Config to obtain the adjusted settings.
 * 
 * This function involves communication with the remote node. 
 *
 * @public              @memberof kAction
 * @param   action      Action object.  
 * @param   settings    Specifies the desired action configuration.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kAction_SetConfig(kAction action, kXml settings);

/** 
 * Gets configuration for the action. 
 *
 * @public              @memberof kAction
 * @param   action      Action object.  
 * @param   settings    Receives the current action configuration.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kAction_Config(kAction action, kXml settings);

/** 
 * Invokes this action without applying the current settings to this node first. 
 * 
 * If the action changes remote configuration, kNode_Refresh can optionally be called to 
 * refresh local configuration.
 * 
 * @public              @memberof kAction
 * @param   action      Action object.  
 * @param   input       Action input object (can be kNULL). 
 * @param   output      Receives action output object (can be kNULL). 
 * @param   alloc       Memory allocator for action output object (or kNULL for default allocator). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kAction_Invoke(kAction action, kObject input, kObject* output, kAlloc alloc); 

/** 
 * Applies current setting to the node and invokes this action. 
 * 
 * Local configuration is automatically refreshed afterwards so no call to kNode_Refresh is necessary.
 * 
 * @public              @memberof kAction
 * @param   action      Action object.  
 * @param   input       Action input object (can be kNULL). 
 * @param   output      Receives action output object (can be kNULL). 
 * @param   alloc       Memory allocator for action output object (or kNULL for default allocator). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kAction_InvokeAndSync(kAction action, kObject input, kObject* output, kAlloc alloc);

#include <kFireSync/Client/kAction.x.h>

#endif
