/** 
 * @file    kControl6Client.h
 * @brief   Declares the kControl6Client class. 
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_CONTROL_6_CLIENT_H
#define K_FIRESYNC_CONTROL_6_CLIENT_H

#include <kFireSync/kNodeDef.h>

/**
 * @internal
 * @class   kControl6Client
 * @extends kObject
 * @ingroup kFireSync-Client-Network
 * @brief   Implements an FCP-6 client. 
 */
//typedef kObject kControl6Client;            --forward-declared in kFsDef.x.h

/** 
 * Constructs a kControl6Client object.
 *
 * @public              @memberof kControl6Client
 * @return              Operation status. 
 */
kFsFx(kStatus) kControl6Client_Construct(kControl6Client* client, kAlloc allocator); 

kFsFx(kStatus) kControl6Client_SetErrorHandler(kControl6Client client, kCallbackFx function, kPointer receiver); 
kFsFx(kStatus) kControl6Client_SetCancelHandler(kControl6Client client, kCallbackFx function, kPointer receiver); 
kFsFx(kStatus) kControl6Client_BeginOpen(kControl6Client client, kIpAddress address, k32u port); 
kFsFx(kStatus) kControl6Client_EndOpen(kControl6Client client); 
kFsFx(kStatus) kControl6Client_Close(kControl6Client client); 

kFsFx(kStatus) kControl6Client_SetProtocolVersion(kControl6Client client, kVersion protocol, kVersion platform); 

kFsFx(kStatus) kControl6Client_ClearFirmware(kControl6Client client, kNodeFirmwareType type); 
kFsFx(kStatus) kControl6Client_WriteFirmware(kControl6Client client, kNodeFirmwareType type, kStream stream, kSize size, kCallbackFx progress, kPointer context); 
kFsFx(kStatus) kControl6Client_ReloadFirmware(kControl6Client client, kNodeFirmwareType type, kStream stream, kSize size);
kFsFx(kStatus) kControl6Client_Reset(kControl6Client client, kBool restartApp);

kFsFx(kStatus) kControl6Client_WriteFile(kControl6Client client, const kChar* path, kStream stream, kSize size, kCallbackFx progress, kPointer context); 
kFsFx(kStatus) kControl6Client_CopyFile(kControl6Client client, const kChar* sourcePath, const kChar* destPath, kCallbackFx progress, kPointer context); 
kFsFx(kStatus) kControl6Client_ReadFile(kControl6Client client, const kChar* path, kStream stream, kCallbackFx progress, kPointer context); 
kFsFx(kStatus) kControl6Client_DeleteFile(kControl6Client client, const kChar* path); 
kFsFx(kStatus) kControl6Client_GetFileInfo(kControl6Client client, const kChar* path, kStorageItem* item);
kFsFx(kStatus) kControl6Client_CreateDirectory(kControl6Client client, const kChar* path); 
kFsFx(kStatus) kControl6Client_DeleteDirectory(kControl6Client client, const kChar* path); 
kFsFx(kStatus) kControl6Client_ListDirectoryEntries(kControl6Client client, const kChar* path, kBool includeFiles, kBool includeDirectories, kArrayList entries); 
kFsFx(kStatus) kControl6Client_ListDirectoryEntriesEx(kControl6Client client, const kChar* path, kArrayList entries); 

kFsFx(kStatus) kControl6Client_GetDeviceConfig(kControl6Client client, kXml xml, k64u* revisionId); 
kFsFx(kStatus) kControl6Client_SetDeviceConfig(kControl6Client client, kXml xml, k64u revisionId); 
kFsFx(kStatus) kControl6Client_CommitDeviceConfig(kControl6Client client); 
kFsFx(kStatus) kControl6Client_GetModeConfig(kControl6Client client, kXml xml, k64u* revisionId); 
kFsFx(kStatus) kControl6Client_SetModeConfig(kControl6Client client, kXml xml, k64u revisionId); 
kFsFx(kStatus) kControl6Client_SaveModeConfig(kControl6Client client, const kChar* path); 
kFsFx(kStatus) kControl6Client_LoadModeConfig(kControl6Client client, const kChar* path); 
kFsFx(kStatus) kControl6Client_CommitModeConfig(kControl6Client client); 
kFsFx(kStatus) kControl6Client_ValidateBlockConfig(kControl6Client client, kXml xml); 

kFsFx(kStatus) kControl6Client_SetDateTime(kControl6Client client, kDateTime dateTime, kBool isAuthoritative);
kFsFx(kStatus) kControl6Client_IncrementInputCounter(kControl6Client client);

kFsFx(kStatus) kControl6Client_CameraOverride(kControl6Client client, kSize index, const kCameraOverrideItem* overrides, kSize count);
kFsFx(kStatus) kControl6Client_LightOverride(kControl6Client client, kSize index, const kLightOverrideItem* overrides, kSize count);

kFsFx(kStatus) kControl6Client_ListProcedures(kControl6Client client, kArrayList procedures);    
kFsFx(kStatus) kControl6Client_InvokeProcedure(kControl6Client client, const kChar* name, kObject input, kObject* output, kAlloc alloc);  

kFsFx(kStatus) kControl6Client_SetVariable(kControl6Client client, const kChar* name, kObject variable); 
kFsFx(kStatus) kControl6Client_GetVariable(kControl6Client client, const kChar* name, kObject* variable, kAlloc alloc); 
kFsFx(kStatus) kControl6Client_GetVariableInfo(kControl6Client client, const kChar* name, kNodeVariableInfo* info); 
kFsFx(kStatus) kControl6Client_DeleteVariable(kControl6Client client, const kChar* name); 
kFsFx(kStatus) kControl6Client_ListVariables(kControl6Client client, kArrayList variables); 

kFsFx(kStatus) kControl6Client_ValidateActionConfig(kControl6Client  client, kXml xml); 
kFsFx(kStatus) kControl6Client_InvokeAction(kControl6Client  client, const kChar* typeName, kXml xml, kObject input, kObject* output, kAlloc alloc); 
kFsFx(kStatus) kControl6Client_InvokeNamedAction(kControl6Client  client, const kChar* name, kObject input, kObject* output, kAlloc alloc); 

kFsFx(kStatus) kControl6Client_ReadMemory(kControl6Client client, kNodeMemorySpace space, k64u offset, kSize length, void* data);
kFsFx(kStatus) kControl6Client_WriteMemory(kControl6Client client, kNodeMemorySpace space, k64u offset, kSize length, const void* data);

kFsFx(kStatus) kControl6Client_ReadRegisters(kControl6Client client, kRegisterModule moduleType, kSize moduleIndex, kArrayList registers);
kFsFx(kStatus) kControl6Client_ReadRegisterOverrides(kControl6Client client, kRegisterModule moduleType, kArrayList registers);
kFsFx(kStatus) kControl6Client_WriteRegisterOverrides(kControl6Client client, kRegisterModule moduleType, kArrayList registers);

kFsFx(kStatus) kControl6Client_I2cRead(kControl6Client client, k32u deviceId, k32u address, kByte* data, kSize size);
kFsFx(kStatus) kControl6Client_I2cWrite(kControl6Client client, k32u deviceId, k32u address, const kByte* data, kSize size);

kFsFx(kStatus) kControl6Client_BeginSpi(kControl6Client client, kSpiDeviceType type);
kFsFx(kStatus) kControl6Client_EndSpi(kControl6Client client);
kFsFx(kStatus) kControl6Client_SpiRead(kControl6Client client, kByte* opCode, kSize opSize, kByte* data, kSize dataSize);
kFsFx(kStatus) kControl6Client_SpiWrite(kControl6Client client, kByte* opCode, kSize opSize, kByte* data, kSize dataSize);

kFsFx(kStatus) kControl6Client_ReadLogHistory(kControl6Client client, kArrayList logItems);
kFsFx(kStatus) kControl6Client_ReadCrashLog(kControl6Client client, kString* crashLog, kAlloc allocator);
kFsFx(kStatus) kControl6Client_ClearCrashLog(kControl6Client client);

kFsFx(kStatus) kControl6Client_BeginStart(kControl6Client client); 
kFsFx(kStatus) kControl6Client_EndStart(kControl6Client client); 
kFsFx(kStatus) kControl6Client_BeginStop(kControl6Client client, kBool synchronizeData); 
kFsFx(kStatus) kControl6Client_EndStop(kControl6Client client); 
kFsFx(kStatus) kControl6Client_BeginEngage(kControl6Client client, k64u startTime, k64s startEncoder); 
kFsFx(kStatus) kControl6Client_EndEngage(kControl6Client client); 

kFsFx(kStatus) kControl6Client_BeginStartReplay(kControl6Client client); 
kFsFx(kStatus) kControl6Client_EndStartReplay(kControl6Client client); 
kFsFx(kStatus) kControl6Client_Replay(kControl6Client client, kObject input, kArrayList output); 

kFsFx(kStatus) kControl6Client_ScheduleEvent(kControl6Client client, k32u eventManagerId, k64s target);    
kFsFx(kStatus) kControl6Client_TriggerEvent(kControl6Client client, k32u eventManagerId); 
kFsFx(kStatus) kControl6Client_TriggerCamera(kControl6Client client, kSize index);
kFsFx(kStatus) kControl6Client_SetLightState(kControl6Client client, kSize index, kBool enabled);
kFsFx(kStatus) kControl6Client_TriggerLight(kControl6Client client, kSize index);
kFsFx(kStatus) kControl6Client_SetDigitalOutState(kControl6Client client, kSize index, kBool state);
kFsFx(kStatus) kControl6Client_TriggerDigitalOut(kControl6Client client, kSize index);
kFsFx(kStatus) kControl6Client_EnqueueAnalogOut(kControl6Client client, kSize index, k32u value);
kFsFx(kStatus) kControl6Client_EnqueueAnalogOutCurrent(kControl6Client client, kSize index, k64f current);
kFsFx(kStatus) kControl6Client_WriteAnalogOutCalibration(kControl6Client client, kSize index, kArray2 calibration, k64u revisionId);
kFsFx(kStatus) kControl6Client_WriteSerialOut(kControl6Client client, kSize index, const void* buffer, kSize size);

kFsFx(kStatus) kControl6Client_WriteRamImage(kControl6Client client, kSize index, kSize stateIndex, kSize imageIndex, kImage image);
kFsFx(kStatus) kControl6Client_WritePrnu(kControl6Client client, kSize index, kImage white, kImage black);
kFsFx(kStatus) kControl6Client_WriteFpn(kControl6Client client, kSize index, kImage offsets);
kFsFx(kStatus) kControl6Client_WriteRangeLut(kControl6Client client, kSize index,
                                                            const kRangeLutParams* params, const kRangeLutArray* xArray,
                                                            const kRangeLutArray* zArray, const kRangeLutArray* validArray);
kFsFx(kStatus) kControl6Client_WritePhaseDecoderLut(kControl6Client client, kSize index, kCameraPhaseDecoderLutType type, kArray1 lut);
kFsFx(kStatus) kControl6Client_SetCameraLvdsPower(kControl6Client client, kSize index, k32u power);
kFsFx(kStatus) kControl6Client_GetCameraLvdsPower(kControl6Client client, kSize index, k32u* power);
kFsFx(kStatus) kControl6Client_WriteAccelerationTestData(kControl6Client client, kSize index, kObject data);
kFsFx(kStatus) kControl6Client_ReadAccelerationTestResult(kControl6Client client, kSize index, kObject* data, kAlloc allocator);

kFsFx(kStatus) kControl6Client_LoadPatterns(kControl6Client client, kSize index, kArray2 patterns);
kFsFx(kStatus) kControl6Client_LoadProjectorRowGains(kControl6Client client, kSize index, kArray1 gains);
kFsFx(kStatus) kControl6Client_LoadProjectorColGains(kControl6Client client, kSize index, kArray1 gains);
kFsFx(kStatus) kControl6Client_ResetEncoder(kControl6Client client);
kFsFx(kStatus) kControl6Client_EnableLed(kControl6Client client, kLed instance, kBool enabled);
kFsFx(kStatus) kControl6Client_SetLedMode(kControl6Client client, kLed instance, kLedMode mode);
kFsFx(kStatus) kControl6Client_SetGpioState(kControl6Client client, kSize bankIndex, k64u mask, k64u state);
kFsFx(kStatus) kControl6Client_GetGpioState(kControl6Client client, kSize bankIndex, k64u* state);

kFsFx(kStatus) kControl6Client_GetInfo(kControl6Client client, kNodeInfo* info); 
kFsFx(kStatus) kControl6Client_GetState(kControl6Client client, kNodeStateInfo* info); 
kFsFx(kStatus) kControl6Client_GetHealth(kControl6Client client, kArrayList stats); 
kFsFx(kStatus) kControl6Client_ReadHealthLog(kControl6Client client, kHealthSummary* summary, kAlloc alloc); 
kFsFx(kStatus) kControl6Client_ClearHealthLog(kControl6Client client); 
kFsFx(kStatus) kControl6Client_GetNodeStats(kControl6Client client, kNodeStats* stats);    
kFsFx(kStatus) kControl6Client_GetEventStats(kControl6Client client, k32u eventManagerId, kEventStats* stats);    
kFsFx(kStatus) kControl6Client_GetCameraStats(kControl6Client client, kSize index, kCameraStats* stats);
kFsFx(kStatus) kControl6Client_GetLightStats(kControl6Client client, kSize index, kLightStats* stats);
kFsFx(kStatus) kControl6Client_GetProjectorStats(kControl6Client client, kSize index, kProjectorStats* stats);
kFsFx(kStatus) kControl6Client_GetAnalogOutStats(kControl6Client client, kSize index, kAnalogOutStats* stats);
kFsFx(kStatus) kControl6Client_GetDigitalOutStats(kControl6Client client, kSize index, kDigitalOutStats* stats);
kFsFx(kStatus) kControl6Client_GetSerialOutStats(kControl6Client client, kSize index, kSerialOutStats* stats);

kFsFx(kStatus) kControl6Client_ResetIoTest(kControl6Client client, kSize index);
kFsFx(kStatus) kControl6Client_GetIoTestStats(kControl6Client client, kSize index, kIoTestStats* stats);
kFsFx(kStatus) kControl6Client_ReadIoTestSerialData(kControl6Client client, kSize index, void* buffer, kSize capacity, kSize* bytesRead);

kFsFx(kStatus) kControl6Client_TestJigCommand(kControl6Client client, kTestJigCommand command, k64u data);
kFsFx(kStatus) kControl6Client_GetTestJigStats(kControl6Client client, kTestJigStats* stats);

kFsFx(kStatus) kControl6Client_LockOutLight(kControl6Client client, kSize index, kBool shouldLock);
kFsFx(kStatus) kControl6Client_SetLightDriverControl(kControl6Client client, kSize index, k64u key, kBool enabled);
kFsFx(kStatus) kControl6Client_GetLightDriverControl(kControl6Client client, kSize index, k64u key, kBool* enabled);
kFsFx(kStatus) kControl6Client_SetLightDriverPower(kControl6Client client, kSize index, k64u key, k32u power, kBool commit);
kFsFx(kStatus) kControl6Client_GetLightDriverPower(kControl6Client client, kSize index, k64u key, k32u* power);
kFsFx(kStatus) kControl6Client_SetLightDriverCurrentLimit(kControl6Client client, kSize index, k64u key, k32u currentLimit, kBool commit);
kFsFx(kStatus) kControl6Client_GetLightDriverCurrentLimit(kControl6Client client, kSize index, k64u key, k32u* currentLimit);
kFsFx(kStatus) kControl6Client_GetLightDriverRemainingSlotCount(kControl6Client client, kSize index, k64u key, kSize* remainingCount);
kFsFx(kStatus) kControl6Client_SetLightDriverInfo(kControl6Client client, kSize index, k64u key, kLightModel model, kVersion revision, k32u deviceId);
kFsFx(kStatus) kControl6Client_GetLightDriverInfo(kControl6Client client, kSize index, k64u key, kLightModel* model, kVersion* revision, k32u* deviceId);
kFsFx(kStatus) kControl6Client_SetLightDriverCalibration(kControl6Client client, kSize index, k64u key, kDataTree ldCal, kBool commit);
kFsFx(kStatus) kControl6Client_GetLightDriverCalibration(kControl6Client client, kSize index, k64u key, kDataTree* ldCal, kAlloc alloc);
kFsFx(kStatus) kControl6Client_ReadLightDriverTemperature(kControl6Client client, kSize index, k64u key, k32s* temperature);

kFsFx(kStatus) kControl6Client_StartOrientationCalibration(kControl6Client client);
kFsFx(kStatus) kControl6Client_GetOrientationStats(kControl6Client client, kOrientationStats* stats);

#include <kFireSync/Client/Network/kControl6Client.x.h>

#endif
