/** 
 * @file    kLc250xInfo.x.h
 *
 * @internal
 * Copyright (C) 2017-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_LC250X_INFO_X_H
#define K_FIRESYNC_LC250X_INFO_X_H

typedef k32s kLc250xTConvMode;

#define kLC250X_TCONV_MODE_128                  (1)                                 ///< Nominal TConv 128 cycles.
#define kLC250X_TCONV_MODE_84                   (2)                                 ///< Nominal TConv 84 cycles.
#define kLC250X_TCONV_MODE_64                   (3)                                 ///< Nominal TConv 64 cycles.

#define kLC250X_TCONV_CYCLES_128                (128)                               ///< Actual TConv 128 cycles.
#define kLC250X_TCONV_CYCLES_84                 (84)                                ///< Actual TConv 84 cycles.
#define kLC250X_TCONV_CYCLES_64                 (64)                                ///< Actual TConv 64 cycles.

#define kLC250X_SYSTEM_CLOCK_FM0                (60093800.0)                        ///< System clock for frame mode 0, in Hz
#define kLC250X_SYSTEM_CLOCK_FM1                (kLC250X_SYSTEM_CLOCK_FM0/2)        ///< System clock for frame mode 1, in Hz
#define kLC250X_SYSTEM_CLOCK_FM2                (kLC250X_SYSTEM_CLOCK_FM0/4)        ///< System clock for frame mode 2, in Hz

#define kLC250X_OPTICAL_BLACK_COLUMNS           (128)                               ///< Optical black columns.
#define kLC250X_OPTICAL_BLACK_ROWS              (8)                                 ///< Optical black rows.
#define kLC250X_MIN_WIDTH                       (64)                                ///< Imager minimum width(pixels).
#define kLC250X_MAX_WIDTH                       (1920)                              ///< Imager maximum width(pixels).
#define kLC250X_WIDTH_GRANULARITY               (64)                                ///< Imager width granularity(pixels).
#define kLC250X_MIN_HEIGHT                      (4)                                 ///< Imager minimum height(pixels).
#define kLC250X_MAX_HEIGHT                      (1280)                              ///< Imager maximum height(pixels).
#define kLC250X_RLUT_WIDTH                      (1280)                              ///< Range LUT width (TODO: FSS-360, FSS-384).
#define kLC250X_RLUT_HEIGHT                     (1920)                              ///< Range LUT height (TODO: FSS-360, FSS-384).
#define kLC250X_HEIGHT_GRANULARITY              (4)                                 ///< Imager height granularity(pixels).
#define kLC250X_LEFT_GRANULARITY                (64)                                ///< Window x granularity(pixels). 
#define kLC250X_TOP_GRANULARITY                 (4)                                 ///< Window y granularity(pixels). 

#define kLC250X_WIDTH_TCONV_128                 (1920)                              ///< Imager width for Tconv 128.
#define kLC250X_WIDTH_TCONV_84                  (1216)                              ///< Imager width for Tconv 84.
#define kLC250X_WIDTH_TCONV_64                  (896)                               ///< Imager width for Tconv 64.

#define kLC250X_STATE_B_TCONV_128               (65)                                ///< State B duration Tconv 128, in system clock cycles.
#define kLC250X_STATE_B_TCONV_84                (65)                                ///< State B duration Tconv 84, in system clock cycles.
#define kLC250X_STATE_B_TCONV_64                (65)                                ///< State B duration Tconv 64, in system clock cycles.

#define kLC250X_STATE_F_TCONV_128               (128)                               ///< State F duration Tconv 128, in system clock cycles.
#define kLC250X_STATE_F_TCONV_84                (114)                               ///< State F duration Tconv 84, in system clock cycles.
#define kLC250X_STATE_F_TCONV_64                (134)                               ///< State F duration Tconv 64, in system clock cycles.

#define kLC250X_MAX_EXPOSURE                    (1000000000)                        ///< Maximum exposure(ns).

#define kLC250X_PIXELS_PER_HBLANKING_CYCLE      (16)                                ///< Increasing horizontal blanking by 1 effectively adds N pixel clocks to row readout.
#define kLC250X_PS_DDR_VIDEO_HBLANKING_CYCLES   (900)                               ///< Count of horizontal blanking cycles to add for PS-DDR-based video; selected to limit worst-case ACP bandwith to 500 MB/s (granularity: 4).

#define kLC250X_MAX_IMAGER_X_SUB                (2)                                 ///< Maximum x-subsampling supported by imager.
#define kLC250X_MAX_IMAGER_Y_SUB                (2)                                 ///< Maximum y-subsampling supported by imager.
#define kLC250X_MIN_BLACK_LEVEL                 (-64)                               ///< Minimum black level supported by PL.
#define kLC250X_MAX_BLACK_LEVEL                 (63)                                ///< Maximum black level supported by PL.
#define kLC250X_ANALOG_GAIN_GRANULARITY         (1)                                 ///< Analog gain step granularity.
#define kLC250X_ANALOG_GAIN_MAX_STEP            (3)                                 ///< Maximum analog gain step.
#define kLC250X_DIGITAL_GAIN_GRANULARITY        (0.0078125)                         ///< Digital gain step granularity.
#define kLC250X_DIGITAL_GAIN_MAX_STEP           (1023)                              ///< Maximum digital gain step.

#define kLC250X_MIN_LVDS_POWER                  (0)                                 ///< Minimum LVDS power supported by imager.
#define kLC250X_MAX_LVDS_POWER                  (7)                                 ///< Maximum LVDS power supported by imager.
#define kLC250X_RELEASE_LVDS_POWER              (4)                                 ///< Default LVDS power value for release sensor.
#define kLC250X_FACTORY_LVDS_POWER              (4)                                 ///< Default LVDS power value for kFactory.

#define kLC250X_DEFAULT_CENTROID_STEP_8K        (32)                                ///< Range LUT Default CG step (8K entries)
#define kLC250X_DEFAULT_CENTROID_STEP_16K       (16)                                ///< Range LUT Default CG step (16K entries)
#define kLC250X_DEFAULT_VALID_STEP              (8)                                 ///< Range LUT Default valid step
#define kLC250X_DEFAULT_SLICE_STEP              (16)                                ///< Range LUT Default slice step

#define kLC250X_INFO_MIN_ISR_GAP_TIME           (55000)                             ///< Crude upper bound on additional gap required for ROI changes on M1/M3 (ns).

#define kLC250X_INFO_PRIME_DELAY                (3000)                             ///< Nominal delay to use with camera priming (FS ns)
#define kLC250X_INFO_PRIME_EXPOSURE             (5000)                             ///< Nominal exposure to use with camera priming (FS ns)
#define kLC250X_INFO_PRIME_GAP                  (10000)                            ///< Nominal gap to use with camera priming (FS ns)

typedef struct kLc250xInfoClass
{
    kCameraInfoClass base; 
} kLc250xInfoClass; 

kDeclareClassEx(kFs, kLc250xInfo, kCameraInfo)
        
kFsFx(kStatus) kLc250xInfo_Init(kLc250xInfo info, kType type, kCameraCapability capabilities, kBool isTrailingRequired, kSize frameSizeGranularity, kVersion plVersion, kAlloc alloc);
kFsFx(kStatus) kLc250xInfo_VRelease(kLc250xInfo info);

kFsFx(kBool) kLc250xInfo_VHasGainChannel(kCameraInfo info, kCameraGain channel);
kFsFx(k32u) kLc250xInfo_VMaxGainStep(kCameraInfo info, kCameraGain channel);
kFsFx(k32u) kLc250xInfo_VGainStep(kLc250xInfo info, kCameraGain channel, k64f gain);
kFsFx(k64f) kLc250xInfo_VGainFactor(kLc250xInfo info, kCameraGain channel, k32u index);

kFsFx(k32u) kLc250xInfo_AnalogGainStep(kLc250xInfo info, k64f gain);
kFsFx(k64f) kLc250xInfo_AnalogGainFactor(kLc250xInfo info, k32u index);
kFsFx(k32u) kLc250xInfo_DigitalGainStep(kLc250xInfo info, k64f gain);
kFsFx(k64f) kLc250xInfo_DigitalGainFactor(kLc250xInfo info, k32u index);

kFsFx(kLc250xTConvMode) kLc250xInfo_ConversionMode(kLc250xInfo info, k32u maxColumns);
kFsFx(k32u) kLc250xInfo_ConversionCycles(kLc250xInfo info, kLc250xTConvMode conversionMode);
kFsFx(k32u) kLc250xInfo_HoriztontalBlankingCycles(kLc250xInfo info, kCameraOutputMode outputMode, kBool ramImageEnabled);
kFsFx(k64f) kLc250xInfo_ConversionTime(kLc250xInfo info, const kCameraInfoParams* params);
kFsFx(k64f) kLc250xInfo_StateBTime(kLc250xInfo info, const kCameraInfoParams* params);
kFsFx(k64f) kLc250xInfo_StateFTime(kLc250xInfo info, const kCameraInfoParams* params);
kFsFx(k64f) kLc250xInfo_ReadoutTime(kLc250xInfo info, const kCameraInfoParams* params);
kFsFx(k64f) kLc250xInfo_VReadoutTime(kLc250xInfo info, const kCameraInfoParams* params);
kFsFx(k64f) kLc250xInfo_VMinInterExposureGap(kLc250xInfo info, const kCameraInfoParams* params);
kFsFx(k64f) kLc250xInfo_VMinInterReadoutGap(kLc250xInfo info, const kCameraInfoParams* params);
kFsFx(k64f) kLc250xInfo_VMinExposure(kLc250xInfo info, const kCameraInfoParams* params);
kFsFx(k64f) kLc250xInfo_VMaxExposure(kLc250xInfo info, const kCameraInfoParams* params);
kFsFx(k64f) kLc250xInfo_VPreExposure(kLc250xInfo info, const kCameraInfoParams* params);
kFsFx(k64f) kLc250xInfo_VPostExposure(kLc250xInfo info, const kCameraInfoParams* params);
kFsFx(k64f) kLc250xInfo_VRowReadoutEfficiency(kCameraInfo info, const kCameraInfoParams* params);
kFsFx(kStatus) kLc250xInfo_VPrimePeriod(kCameraInfo info, k64u* delay, k64u* exposure, k64u* gap);

#endif
