/** 
 * @file    kSobelvAccgdwInfo.h
 * @brief   Declares the kSobelvAccgdwInfo type. 
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_SOBELV_ACCGDW_INFO_H
#define K_FIRESYNC_SOBELV_ACCGDW_INFO_H

#include <kFireSync/Client/Camera/Algorithms/kAcgInfo.h>

/**
 * @class   kSobelvAccgdwInfo
 * @extends kAcgInfo
 * @ingroup kFireSync-Client-Camera-Algorithms
 * @brief   Represents configuration for the SOBELV_ACCGDW algorithm, which provides column-wise 
 *          spot detection using a 1D sobel filter. 
 * 
 * Algorithm configuration options can be modified to emit results from different stages of the 
 * processing pipeline: centroids (aka 'spots'), calibrated 2D ranges, or resampled 
 * 1D ranges. Access to earlier processing stages is provided both for diagnostic/testing 
 * purposes and to support applications that require access to data from prior processing stages. 
 * However, only one type of result can be emitted at a time.
 *   
 * If range output is disabled, the algorithm will emit centroids (spots). One kMsg will be generated
 * for each camera frame and each kMsg will contain one kArrayList<kFpgaWideSpot>. If spot sorting is 
 * disabled, the complete list of unsorted spots (up to the configured maximum number of spots) will be 
 * delivered. In this case, the size of the delivered spot list is governed by the MaxFound property. If 
 * spot sorting is enabled, then the spot list will be reduced to one spot per column using to the configured 
 * spot sorting criteria. In this case, the size of the spot list is governed by the width of the source 
 * image.  
 * 
 * If range output is enabled, the algorithm will convert spots to calibrated range values. (Range calibration 
 * tables must be provided via the kCamera WriteRangeLut method). In this case, the algorithm will produce 
 * kProfileMsg objects, where each kProfileMsg contains results from one or more camera frames. If range 
 * resampling is disabled, the algorithm will emit kProfileMsg<kPoint16s> objects, where each element 
 * represents a 2D range value. If range resampling is enabled, the algorithm will emit kProfileMsg<k16s> 
 * objects, where each element represents a 1D range value that has been resampled to a uniform spacing along 
 * the field of view. Range processing options can be configured via methods inherited from the kAcgInfo base 
 * class. The maximum number of frames included in a single kProfileMsg can be limited via the kCameraState 
 * MaxBatchSize property. 
 */
//typedef kAcgInfo kSobelvAccgdwInfo;            --forward-declared in kFsDef.x.h

/** 
 * Constructs a kSobelvAccgdwInfo object.
 *
 * @public                  @memberof kSobelvAccgdwInfo
 * @param   info            Destination for the constructed object handle. 
 * @param   capabilities    Represents camera features.
 * @param   plConfig        PL configuration id. 
 * @param   plVersion       PL configuration version.
 * @param   allocator       Memory allocator (or kNULL for default). 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kSobelvAccgdwInfo_Construct(kSobelvAccgdwInfo* info, kCameraCapability capabilities, k32u plConfig, kVersion plVersion, kAlloc allocator); 

/** @name EdgeWindow */
//@{

/** 
 * Sets the sobel edge window size.
 * 
 * This property represents the size of the 1D sobel filter window. Odd numbers only.
 *
 * @public              @memberof kSobelvAccgdwInfo
 * @param   info        Algorithm info object.  
 * @param   edgeWindow  Sobel edge window size.
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kSobelvAccgdwInfo_SetEdgeWindow(kSobelvAccgdwInfo info, k32u edgeWindow, kAdjust adjustment = kADJUST_NONE); 

/** 
 * Gets the sobel edge window size.
 *
 * This property represents the size of the 1D sobel filter window. Odd numbers only.
 * 
 * @public              @memberof kSobelvAccgdwInfo
 * @param   info        Algorithm info object.  
 * @return              Returns the sobel edge window size.
 */
kFsFx(k32u) kSobelvAccgdwInfo_EdgeWindow(kSobelvAccgdwInfo info); 

/** 
 * Reports constraint and validity information for the EdgeWindow setting. 
 * 
 * @public              @memberof kSobelvAccgdwInfo
 * @param   info        Algorithm info object.  
 * @param   info32u     Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kSobelvAccgdwInfo_EdgeWindowInfo(kSobelvAccgdwInfo info, kInfo32u* info32u);

//@}
/** @name EdgeThreshold */
//@{

/** 
 * Sets the sobel filter threshold used to detect spot edges.
 *
 * This threshold is applied to the output from the spot detection 1D sobel filter. 
 * +EdgeThreshold is used for leading edges, while -EdgeThreshold is used for trailing edges. 
 *
 * @public              @memberof kSobelvAccgdwInfo
 * @param   info        Algorithm info object.  
 * @param   threshold   Primary spot threshold.   
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kSobelvAccgdwInfo_SetEdgeThreshold(kSobelvAccgdwInfo info, k32u threshold, kAdjust adjustment = kADJUST_NONE); 

/** 
 * Gets the sobel filter threshold used to detect spot edges.
 * 
 * This threshold is applied to the output from the spot detection 1D sobel filter. 
 * +EdgeThreshold is used for leading edges, while -EdgeThreshold is used for trailing edges. 
 *
 * @public              @memberof kSobelvAccgdwInfo
 * @param   info        Algorithm info object.  
 * @return              Edge threshold.
 */
kFsFx(k32u) kSobelvAccgdwInfo_EdgeThreshold(kSobelvAccgdwInfo info); 

/** 
 * Reports constraint and validity information for the EdgeThreshold setting. 
 *
 * @public              @memberof kSobelvAccgdwInfo
 * @param   info        Algorithm info object.  
 * @param   info32u     Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kSobelvAccgdwInfo_EdgeThresholdInfo(kSobelvAccgdwInfo info, kInfo32u* info32u);

//@}
/** @name WidthIntensityThreshold */
//@{

/** 
 * Sets the width intensity threshold.
 *
 * The EdgeThreshold property and sobel filter output are used to establish spot entry/exit boundaries. The 
 * pixel value immediately before spot entry establishes the spot EntryValue (baseline intensity for spot). 
 * 
 * Within the spot entry/exit boundaries, pixels with intensity values greater than or equal to 
 * EntryValue + WidthIntensityThreshold are counted towards the reported spot width. If WidthIntensityThreshold 
 * is set to zero (the default), all pixels from entry to exit will count towards the reported spot width.
 *
 * @public              @memberof kSobelvAccgdwInfo
 * @param   info        Algorithm info object.  
 * @param   threshold   Width intensity threshold (or zero to count all spot pixels).
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kSobelvAccgdwInfo_SetWidthIntensityThreshold(kSobelvAccgdwInfo info, k32u threshold, kAdjust adjustment = kADJUST_NONE); 

/** 
 * Gets the width intensity threshold.
 *
 * The EdgeThreshold property and sobel filter output are used to establish spot entry/exit boundaries. The 
 * pixel value immediately before spot entry establishes the spot EntryValue (baseline intensity for spot). 
 * 
 * Within the spot entry/exit boundaries, pixels with intensity values greater than or equal to 
 * EntryValue + WidthIntensityThreshold are counted towards the reported spot width. If WidthIntensityThreshold 
 * is set to zero (the default), all pixels from entry to exit will count towards the reported spot width.
 * 
 * @public              @memberof kSobelvAccgdwInfo
 * @param   info        Algorithm info object.  
 * @return              Width intensity threshold.
 */
kFsFx(k32u) kSobelvAccgdwInfo_WidthIntensityThreshold(kSobelvAccgdwInfo info); 

/** 
 * Reports constraint and validity information for the WidthIntensityThreshold setting. 
 *
 * @public              @memberof kSobelvAccgdwInfo
 * @param   info        Algorithm info object.  
 * @param   info32u     Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kSobelvAccgdwInfo_WidthIntensityThresholdInfo(kSobelvAccgdwInfo info, kInfo32u* info32u);

//@}
/** @name MinWidth */
//@{

/** 
 * Sets the minimum spot width threshold.
 *
 * Spots with less than MinWidth pixels are excluded from output. The spot pixel count used for this filter includes 
 * all pixels between entry/exit that pass the @ref kSobelvAccgdwInfo_WidthIntensityThreshold filter.
 * 
 * @public              @memberof kSobelvAccgdwInfo
 * @param   info        Algorithm info object.  
 * @param   minWidth    Minimum spot width, in pixels.
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kSobelvAccgdwInfo_SetMinWidth(kSobelvAccgdwInfo info, k32u minWidth, kAdjust adjustment = kADJUST_NONE); 

/** 
 * Gets the minimum spot width threshold.
 *
 * Spots with less than MinWidth pixels are excluded from output. The spot pixel count used for this filter includes 
 * all pixels between entry/exit that pass the @ref kSobelvAccgdwInfo_WidthIntensityThreshold filter.
 * 
 * @public              @memberof kSobelvAccgdwInfo
 * @param   info        Algorithm info object.  
 * @return              Minimum spot width threshold, in pixels.
 */
kFsFx(k32u) kSobelvAccgdwInfo_MinWidth(kSobelvAccgdwInfo info); 

/** 
 * Reports constraint and validity information for the MinWidth setting. 
 *
 * @public              @memberof kSobelvAccgdwInfo
 * @param   info        Algorithm info object.  
 * @param   info32u     Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kSobelvAccgdwInfo_MinWidthInfo(kSobelvAccgdwInfo info, kInfo32u* info32u);

//@}
/** @name MaxWidth */
//@{

/** 
 * Sets the maximum spot width threshold.
 *
 * Spots with more than MaxWidth pixels are excluded from output. The spot pixel count used for this filter includes 
 * all pixels between entry/exit that pass the @ref kSobelvAccgdwInfo_WidthIntensityThreshold filter.
 * 
 * @public              @memberof kSobelvAccgdwInfo
 * @param   info        Algorithm info object.  
 * @param   maxWidth    Maximum spot width threshold, in pixels.   
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kSobelvAccgdwInfo_SetMaxWidth(kSobelvAccgdwInfo info, k32u maxWidth, kAdjust adjustment = kADJUST_NONE); 

/** 
 * Gets the maximum spot width threshold.
 *
 * Spots with more than MaxWidth pixels are excluded from output. The spot pixel count used for this filter includes 
 * all pixels between entry/exit that pass the @ref kSobelvAccgdwInfo_WidthIntensityThreshold filter.
 * 
 * @public              @memberof kSobelvAccgdwInfo
 * @param   info        Algorithm info object.  
 * @return              Maximum spot width threshold, in pixels.
 */
kFsFx(k32u) kSobelvAccgdwInfo_MaxWidth(kSobelvAccgdwInfo info); 

/** 
 * Reports constraint and validity information for the MaxWidth setting. 
 *
 * @public              @memberof kSobelvAccgdwInfo
 * @param   info        Algorithm info object.  
 * @param   info32u     Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kSobelvAccgdwInfo_MaxWidthInfo(kSobelvAccgdwInfo info, kInfo32u* info32u);

//@}
/** @name MinSum */
//@{

/** 
 * Sets the minimum spot sum threshold.
 *
 * Spots with sum values less than this threshold are excluded from output.
 * 
 * @public              @memberof kSobelvAccgdwInfo
 * @param   info        Algorithm info object.  
 * @param   minSum      Minimum spot sum threshold.
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kSobelvAccgdwInfo_SetMinSum(kSobelvAccgdwInfo info, k32u minSum, kAdjust adjustment = kADJUST_NONE); 

/** 
 * Gets the minimum spot sum threshold.
 *
 * @public              @memberof kSobelvAccgdwInfo
 * @param   info        Algorithm info object.  
 * @return              Minimum spot sum threshold.
 */
kFsFx(k32u) kSobelvAccgdwInfo_MinSum(kSobelvAccgdwInfo info); 

/** 
 * Reports constraint and validity information for the MinSum setting. 
 *
 * @public              @memberof kSobelvAccgdwInfo
 * @param   info        Algorithm info object.  
 * @param   info32u     Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kSobelvAccgdwInfo_MinSumInfo(kSobelvAccgdwInfo info, kInfo32u* info32u);

//@}
/** @name MaxFound */
//@{

/** 
 * Sets the maximum number of spots that can be detected. 
 *
 * If the specified maximum number of spots are found, any additional spots will be omitted 
 * from output. 
 * 
 * If spot sorting is disabled, the MaxFound parameter determines the size of the data array that 
 * will be delivered from PL to PS memory, regardless of the number of spots actually found 
 * (empty space at the end of the list is filled with special null values). Accordingly, to avoid 
 * wasting controller memory bandwidth, the MaxFound parameter should normally be set to the smallest 
 * viable value when receiving unsorted spots. 
 * 
 * @public              @memberof kSobelvAccgdwInfo
 * @param   info        Algorithm info object.  
 * @param   maxFound    Maximum number of spots that can be detected. 
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kSobelvAccgdwInfo_SetMaxFound(kSobelvAccgdwInfo info, k32u maxFound, kAdjust adjustment = kADJUST_NONE); 

/** 
 * Gets the maximum number of spots that can be detected. 
 *
 * If the specified maximum number of spots are found, any additional spots will be omitted 
 * from output. 
 * 
 * If spot sorting is disabled, the MaxFound parameter determines the size of the data array that 
 * will be delivered from PL to PS memory, regardless of the number of spots actually found 
 * (empty space at the end of the list is filled with special null values). Accordingly, to avoid 
 * wasting controller memory bandwidth, the MaxFound parameter should normally be set to the smallest 
 * viable value when receiving unsorted spots. 
 * 
 * @public              @memberof kSobelvAccgdwInfo
 * @param   info        Algorithm info object.  
 * @return              Maximum number of spots that can be detected.
 */
kFsFx(k32u) kSobelvAccgdwInfo_MaxFound(kSobelvAccgdwInfo info); 

/** 
 * Reports constraint and validity information for the MaxFound setting. 
 *
 * @public              @memberof kSobelvAccgdwInfo
 * @param   info        Algorithm info object.  
 * @param   info32u     Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kSobelvAccgdwInfo_MaxFoundInfo(kSobelvAccgdwInfo info, kInfo32u* info32u);

//@}
/** @name SpotSortEnabled */
//@{

/** 
 * Enables or disables spot sorting.
 *
 * If spot sorting is enabled, spots will be sorted into slices (columns) and filtered 
 * such that there is one spot per slice. 
 * 
 * @public              @memberof kSobelvAccgdwInfo
 * @param   info        Algorithm info object.  
 * @param   enabled     Enable spot sorting?   
 * @return              Operation status. 
 */
kFsFx(kStatus) kSobelvAccgdwInfo_EnableSpotSort(kAccgd2kInfo info, kBool enabled);

/** 
 * Reports whether spot sorting is enabled.
 * 
 * If spot sorting is enabled, spots will be sorted into slices (columns) and filtered 
 * such that there is one spot per slice. 
 *
 * @public              @memberof kSobelvAccgdwInfo
 * @param   info        Algorithm info object.  
 * @return              kTRUE if sorting is enabled.
 */
kFsFx(kBool) kSobelvAccgdwInfo_SpotSortEnabled(kAccgd2kInfo info);

/** 
 * Reports constraint and validity information for the EnableSpotSort setting. 
 *
 * @public              @memberof kSobelvAccgdwInfo
 * @param   info        Algorithm info object.  
 * @param   infoBool    Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kSobelvAccgdwInfo_SpotSortEnabledInfo(kAccgd2kInfo info, kInfoBool* infoBool);

//@}
/** @name SpotSortInverted */
//@{

/** 
 * Enables or disables reverse spot order.
 * 
 * The default sorting direction yields ascending slice indices in algorithm output data.
 * The SpotSortInverted option can be used to reverse the spot output order. 
 * 
 * @public              @memberof kSobelvAccgdwInfo
 * @param   info        Algorithm info object.  
 * @param   invert      Enable reverse spot-sorting?
 * @return              Operation status. 
 */
kFsFx(kStatus) kSobelvAccgdwInfo_InvertSpotSort(kSobelvAccgdwInfo info, kBool invert);

/** 
 * Reports whether spot sorting order is reversed.
 * 
 * The default sorting direction yields ascending slice indices in algorithm output data.
 * The SpotSortInverted option can be used to reverse the spot output order. 
 *
 * @public              @memberof kSobelvAccgdwInfo
 * @param   info        Algorithm info object.  
 * @return              Is reverse spot sorting enabled?
 */
kFsFx(kBool) kSobelvAccgdwInfo_SpotSortInverted(kSobelvAccgdwInfo info);

/** 
 * Reports constraint and validity information for the SpotSortInverted setting. 
 *
 * @public              @memberof kSobelvAccgdwInfo
 * @param   info        Algorithm info object.  
 * @param   infoBool    Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kSobelvAccgdwInfo_InvertSpotSortInfo(kSobelvAccgdwInfo info, kInfoBool* infoBool);

//@}
/** @name SpotSortType */
//@{

/** 
 * Sets the spot sorting methodology.
 * 
 * When spot sorting is enabled, SpotSortType determines the criterion used to map elements of the 
 * unsorted spot array, which may include multiple spots per slice (column), to indices within the 
 * sorted output array (one spot per slice). 
 * 
 * @public              @memberof kSobelvAccgdwInfo
 * @param   info        Algorithm info object.  
 * @param   type        Sorting methodology.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kSobelvAccgdwInfo_SetSpotSortType(kSobelvAccgdwInfo info, kCameraSpotSort type);

/** 
 * Gets the spot sorting methodology.
 * 
 * When spot sorting is enabled, SpotSortType determines the criterion used to map elements of the 
 * unsorted spot array, which may include multiple spots per slice (column), to indices within the 
 * sorted output array (one spot per slice). 
 * 
 * @public              @memberof kSobelvAccgdwInfo
 * @param   info        Algorithm info object.  
 * @return              Sorting methodology.
 */
kFsFx(kCameraSpotSort) kSobelvAccgdwInfo_SpotSortType(kSobelvAccgdwInfo info);

/** 
 * Reports constraint and validity information for the SpotSortType setting. 
 *
 * @public              @memberof kSobelvAccgdwInfo
 * @param   info        Algorithm info object.  
 * @param   info32u     Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kSobelvAccgdwInfo_SpotSortTypeInfo(kSobelvAccgdwInfo info, kInfo32u* info32u);

//@}
/** @name SpotFormat */
//@{

/** 
 * Sets the spot format.
 * 
 * Most kFpgaWideSpot fields are standard, but some are optional. The SpotFormat property can be used to select 
 * the desired spot format layout.
 * 
 * Note, the kFpgaWideSpot structure is not self-describing. Application software is reponsible for maintaining 
 * knowledge of the SpotFormat configuration setting when interpreting received spot data. 
 * 
 * @public              @memberof kSobelvAccgdwInfo
 * @param   info        Algorithm info object.  
 * @param   format      Spot format.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kSobelvAccgdwInfo_SetSpotFormat(kSobelvAccgdwInfo info, kAcgSpotFormat format);

/** 
 * Gets the spot format.
 * 
 * Most kFpgaWideSpot fields are standard, but some are optional. The SpotFormat property can be used to select 
 * the desired spot format layout.
 * 
 * Note, the kFpgaWideSpot structure is not self-describing. Application software is reponsible for maintaining 
 * knowledge of the SpotFormat configuration setting when interpreting received spot data. 
 * 
 * @public              @memberof kSobelvAccgdwInfo
 * @param   info        Algorithm info object.  
 * @return              Spot format.
 */
kFsFx(kAcgSpotFormat) kSobelvAccgdwInfo_SpotFormat(kSobelvAccgdwInfo info);

/** 
 * Reports constraint and validity information for the SpotFormat setting. 
 *
 * @public              @memberof kSobelvAccgdwInfo
 * @param   info        Algorithm info object.  
 * @param   infoBits    Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kSobelvAccgdwInfo_SpotFormatInfo(kSobelvAccgdwInfo info, kInfoBits* infoBits);

//@}
/** @name EntryDebounceEnabled */
//@{

/** 
 * Enables or disables spot entry debouncing.
 *
 * If enabled, a new spot cannot be detected while in an existing spot. 
 * 
 * @public              @memberof kSobelvAccgdwInfo
 * @param   info        Algorithm info object.  
 * @param   enabled     Enable spot entry debouncing?   
 * @return              Operation status. 
 */
kFsFx(kStatus) kSobelvAccgdwInfo_EnableEntryDebounce(kAccgd2kInfo info, kBool enabled);

/** 
 * Reports whether spot entry debouncing is enabled.
 * 
 * If enabled, a new spot cannot be detected while in an existing spot. 
 * 
 * @public              @memberof kSobelvAccgdwInfo
 * @param   info        Algorithm info object.  
 * @return              kTRUE if spot entry debouncing is enabled.
 */
kFsFx(kBool) kSobelvAccgdwInfo_EntryDebounceEnabled(kAccgd2kInfo info);

/** 
 * Reports constraint and validity information for the EntryDebounceEnabled setting. 
 *
 * @public              @memberof kSobelvAccgdwInfo
 * @param   info        Algorithm info object.  
 * @param   infoBool    Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kSobelvAccgdwInfo_EntryDebounceEnabledInfo(kAccgd2kInfo info, kInfoBool* infoBool);

//@}
/** @name ExitIntensityThreshold */
//@{

/** 
 * Sets the exit intensity threshold.
 *
 * If non-zero, adds an additional spot exit constraint such that spot intensity must fall below 
 * EntryValue + ExitIntensityThreshold. The pixel value immediately before spot entry establishes 
 * the spot EntryValue (baseline intensity for spot). 
 * 
 * If zero, this additional constraint is disabled.
 *
 * @public              @memberof kSobelvAccgdwInfo
 * @param   info        Algorithm info object.  
 * @param   threshold   Exit intensity threshold (or zero to disable constraint).   
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kSobelvAccgdwInfo_SetExitIntensityThreshold(kSobelvAccgdwInfo info, k32u threshold, kAdjust adjustment = kADJUST_NONE); 

/** 
 * Gets the exit intensity threshold.
 *
 * If non-zero, adds an additional spot exit constraint such that spot intensity must fall below 
 * EntryValue + ExitIntensityThreshold. The pixel value immediately before spot entry establishes 
 * the spot EntryValue (baseline intensity for spot). 
 * 
 * If zero, this additional constraint is disabled.
 * 
 * @public              @memberof kSobelvAccgdwInfo
 * @param   info        Algorithm info object.  
 * @return              Exit intensity threshold (or zero if disabled).   
 */
kFsFx(k32u) kSobelvAccgdwInfo_ExitIntensityThreshold(kSobelvAccgdwInfo info); 

/** 
 * Reports constraint and validity information for the ExitIntensityThreshold setting. 
 *
 * @public              @memberof kSobelvAccgdwInfo
 * @param   info        Algorithm info object.  
 * @param   info32u     Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kSobelvAccgdwInfo_ExitIntensityThresholdInfo(kSobelvAccgdwInfo info, kInfo32u* info32u);

//@}

#include <kFireSync/Client/Camera/Algorithms/kSobelvAccgdwInfo.x.h>

#endif
