/** 
 * @file    kAlgInfo.x.h
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_ALG_INFO_X_H
#define K_FIRESYNC_ALG_INFO_X_H

#include <kApi/Data/kXml.h>

typedef struct kAlgInfoClass
{
    kObjectClass base; 
    kCameraCapability capabilities; 
    k32u plConfig; 
    kVersion plVersion;
    kCameraState state;                 //Optional reference to camera state; used in client-side validation/verification. 
} kAlgInfoClass; 

typedef struct kAlgInfoVTable
{
    kObjectVTable base; 
    kStatus (kCall* VToRegisters)(kAlgInfo info, k32u* reg, kSize capacity, kSize* used, kSize activePipeCount, 
        kSize imageWidth, kSize imgXSubShift, kSize plXSubShift, kSize imageHeight, kSize imgYSubShift, kSize plYSubShift);
    kStatus (kCall* VInitRead)(kAlgInfo info, kType type, kCameraCapability capabilities, k32u plConfig, kVersion plVersion, kXml xml, kXml item, kAlloc alloc); 
    kStatus (kCall* VRead)(kAlgInfo info, kXml xml, kXml item);    
    kStatus (kCall* VWrite)(kAlgInfo info, kXml xml, kXml item);    
    kStatus (kCall* VVerify)(kAlgInfo info); 
    kCameraAlgorithm (kCall* VId)(kAlgInfo info);
    kType (kCall* VItemType)(kAlgInfo info);
    kSize (kCall* VItemCount)(kAlgInfo info);
    kBool (kCall* VEquals)(kAlgInfo info, kAlgInfo info1);
} kAlgInfoVTable; 

kDeclareVirtualClassEx(kFs, kAlgInfo, kObject)

/** 
 * Constructs a suitable kAlgInfo object based on the given type id. 
 *
 * @public                  @memberof kAlgInfo
 * @param   info            Receives the constructed kAlgInfo object. 
 * @param   typeId          Numeric type id representing the desired algorithm type. 
 * @param   capabilities    Represents camera features.
 * @param   plConfig        PL configuration.
 * @param   plVersion       PL version.
 * @param   allocator       Memory allocator (or kNULL for default). 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kAlgInfo_Construct(kAlgInfo* info, kCameraAlgorithm typeId, kCameraCapability capabilities, k32u plConfig, kVersion plVersion, kAlloc allocator); 

/** 
 * Constructs a suitable algorithm info object based on the given XML content. 
 *
 * @public                  @memberof kAlgInfo
 * @param   info            Receives the constructed kAlgInfo object. 
 * @param   capabilities    Represents camera features.
 * @param   plConfig        PL configuration.
 * @param   plVersion       PL version.
 * @param   xml             XML object. 
 * @param   item            XML content item. 
 * @param   allocator       Memory allocator (or kNULL for default). 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kAlgInfo_Load(kAlgInfo* info, kCameraCapability capabilities, k32u plConfig, kVersion plVersion, kXml xml, kXmlItem item, kAlloc allocator); 

/** 
 * Constructs a new algorithm object or reuses the given algorithm object based on the XML content. 
 *
 * @public                  @memberof kAlgInfo
 * @param   info            Existing algorithm object; receives an updated kAlgInfo object (in/out). 
 * @param   capabilities    Represents camera features.
 * @param   plConfig        PL configuration.
 * @param   plVersion       PL version.
 * @param   xml             XML object. 
 * @param   item            XML content item. 
 * @param   allocator       Memory allocator (or kNULL for default). 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kAlgInfo_LoadReplace(kAlgInfo* info, kCameraCapability capabilities, k32u plConfig, kVersion plVersion, kXml xml, kXmlItem item, kAlloc allocator);

/** 
 * Saves the algorithm info to the specified XML object. 
 *
 * Unlike the kAlgInfo_Export function, this function also writes any type metadata that would be 
 * required to later parse the XML content using the kAlgInfo_Load function. 
 *
 * @public              @memberof kAlgInfo
 * @param   info        Algorithm info object. 
 * @param   xml         XML object. 
 * @param   item        XML content item. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kAlgInfo_Save(kAlgInfo info, kXml xml, kXmlItem item); 

kFsFx(kStatus) kAlgInfo_Init(kAlgInfo info, kType type, kCameraCapability capabilities, k32u plConfig, kVersion plVersion, kAlloc alloc); 
kFsFx(kStatus) kAlgInfo_VRelease(kAlgInfo info); 

kFsFx(kStatus) kAlgInfo_VClone(kAlgInfo info, kAlgInfo other, kAlloc valueAlloc, kObject context); 

kFsFx(kStatus) kAlgInfo_Verify(kAlgInfo info);

kFsFx(kStatus) kAlgInfo_ToRegisters(kAlgInfo info, k32u* reg, kSize capacity, kSize* used, kSize activePipeCount, 
        kSize imageWidth, kSize imgXSubShift, kSize plXSubShift, kSize imageHeight, kSize imgYSubShift, kSize plYSubShift);
kFsFx(kStatus) kAlgInfo_VToRegisters(kAlgInfo info, k32u* reg, kSize capacity, kSize* used, kSize activePipeCount, 
        kSize imageWidth, kSize imgXSubShift, kSize plXSubShift, kSize imageHeight, kSize imgYSubShift, kSize plYSubShift);

kFsFx(kStatus) kAlgInfo_InitRead(kAlgInfo info, kType type, kCameraCapability capabilities, k32u plConfig, kVersion plVersion, kXml xml, kXmlItem item, kAlloc alloc); 
kFsFx(kStatus) kAlgInfo_VInitRead(kAlgInfo info, kType type, kCameraCapability capabilities, k32u plConfig, kVersion plVersion, kXml xml, kXmlItem item, kAlloc alloc); 

kFsFx(kStatus) kAlgInfo_Read(kAlgInfo info, kXml xml, kXmlItem item); 
kFsFx(kStatus) kAlgInfo_VRead(kAlgInfo info, kXml xml, kXmlItem item); 

kFsFx(kStatus) kAlgInfo_Write(kAlgInfo info, kXml xml, kXmlItem item); 
kFsFx(kStatus) kAlgInfo_VWrite(kAlgInfo info, kXml xml, kXmlItem item); 

kFsFx(kStatus) kAlgInfo_SetCameraState(kAlgInfo info, kCameraState state); 

kFsFx(kCameraAlgorithm) kAlgInfo_Id(kAlgInfo info);
kFsFx(kCameraAlgorithm) kAlgInfo_VId(kAlgInfo info);

kFsFx(kType) kAlgInfo_ItemType(kAlgInfo info);
kFsFx(kType) kAlgInfo_VItemType(kAlgInfo info);

kFsFx(kSize) kAlgInfo_ItemCount(kAlgInfo info);
kFsFx(kSize) kAlgInfo_VItemCount(kAlgInfo info);

kFsFx(kBool) kAlgInfo_Equals(kAlgInfo info, kAlgInfo info1);
kFsFx(kBool) kAlgInfo_VEquals(kAlgInfo info, kAlgInfo info1);

#endif
